<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Trigger;
use App\Models\TriggerSchedule;
use App\Models\ScheduleCampaign;
use App\Models\ScheduleCampaignSentLog;

class Cleaner extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'relayzo:cleaner';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clean DB and extra files';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        // Clean unnecessary files
        try {
            $this->cleanFiles();
        } catch(\Exception $e) {
            \Log::error('relayzo:cleaner:files => '.$e->getMessage());
        }

        // Clean scheduled triggers
        try {
            $this->cleanScheduledTriggers();
        } catch(\Exception $e) {
            \Log::error('relayzo:cleaner:triggers => '.$e->getMessage());
        }

        // Clean trigger logs
        try {
            $this->cleanTriggerLogs();
        } catch(\Exception $e) {
            \Log::error('relayzo:cleaner:trigger_logs => '.$e->getMessage());
        }

        // Clean activity logs
        try {
            $this->cleanActivityLogs();
        } catch(\Exception $e) {
            \Log::error('relayzo:cleaner:activity_logs => '.$e->getMessage());
        }

        // Clean completed campaign sent logs
        try {
            $this->cleanCompletedCampaignSentLogs();
        } catch(\Exception $e) {
            \Log::error('relayzo:cleaner:campaign_sent_logs => '.$e->getMessage());
        }
    }

    /**
     * Clean unnecessary files
     */
    protected function cleanFiles()
    {
        // Remove completed campaign files - optimize with batch processing
        $schedules = \App\Models\ScheduleCampaign::whereStatus('Completed')
            ->select('id', 'user_id')
            ->orderBy('id', 'desc')
            ->get();

        if($schedules->isNotEmpty()) {
            foreach($schedules as $schedule) {
                $path_schedule_campaign = str_replace('[user-id]', $schedule->user_id, config('custom.path_schedule_campaign'));
                $dir = $path_schedule_campaign.$schedule->id.DIRECTORY_SEPARATOR;
                
                if(is_dir($dir)) {
                    try {
                        $this->removeDirectory($dir);
                    } catch (\Exception $e) {
                        \Log::error('relayzo:cleaner:remove_dir => '.$e->getMessage());
                    }
                }
            }
        }
        
        // If campaign is deleted - optimize with batch processing
        $users = \App\Models\User::select('id')->get();
        $existing_campaign_ids = \App\Models\ScheduleCampaign::pluck('id')->toArray();

        foreach($users as $user) {
            $path_schedule_campaign = str_replace('[user-id]', $user->id, config('custom.path_schedule_campaign'));
            
            if(!is_dir($path_schedule_campaign)) continue;
            
            try {
                $dirs = array_diff(scandir($path_schedule_campaign), ['.', '..']);
            } catch (\Exception $e) {
                continue;
            }

            foreach($dirs as $dir_id) {
                // Skip if campaign still exists
                if(in_array($dir_id, $existing_campaign_ids)) continue;

                $dir = $path_schedule_campaign.$dir_id.DIRECTORY_SEPARATOR;
                if(is_dir($dir)) {
                    try {
                        $this->removeDirectory($dir);
                    } catch (\Exception $e) {
                        \Log::error('relayzo:cleaner:remove_orphaned_dir => '.$e->getMessage());
                    }
                }
            }
        }
    }

    /**
     * Remove directory and all its contents recursively
     */
    private function removeDirectory($dir)
    {
        if(!is_dir($dir)) return;
        
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach($files as $file) {
            $file_path = $dir . $file;
            if(is_dir($file_path)) {
                $this->removeDirectory($file_path);
            } else {
                unlink($file_path);
            }
        }
        rmdir($dir);
    }

    // Clean scheduled triggers if triggers deleted - optimize with batch processing
    protected function cleanScheduledTriggers()
    {
        // Get all existing trigger IDs
        $existing_trigger_ids = Trigger::pluck('id')->toArray();
        
        // Delete orphaned trigger schedules in batch
        TriggerSchedule::whereNotIn('trigger_id', $existing_trigger_ids)->delete();
    }

    /**
     * Clean the trigger logs up to 15 days - optimize with chunked deletion
     */
    protected function cleanTriggerLogs()
    {
        $cutoff_date = \Carbon\Carbon::now()->subDay(15);
        
        // Use chunked deletion for better performance with large datasets
        \DB::table('trigger_logs')
            ->where('created_at', '<', $cutoff_date)
            ->orderBy('id')
            ->chunkById(1000, function($logs) {
                \DB::table('trigger_logs')->whereIn('id', $logs->pluck('id'))->delete();
            });
    }

    /**
     * Clean the activity logs - optimize with chunked deletion
     */
    protected function cleanActivityLogs()
    {
        try {
            $settings = \App\Models\Setting::find(config('custom.app_id'));
            $days = json_decode($settings->attributes)->activity_log_retention;
        } catch (\Exception $e) {
            $days = config('custom.activity_log_retention');
        }
        
        $cutoff_date = \Carbon\Carbon::now()->subDay($days);
        
        // Use chunked deletion for better performance with large datasets
        \DB::table('activity_log')
            ->where('created_at', '<', $cutoff_date)
            ->orderBy('id')
            ->chunkById(1000, function($logs) {
                \DB::table('activity_log')->whereIn('id', $logs->pluck('id'))->delete();
            });
    }

    /**
     * Clean sent logs for completed campaigns
     */
    protected function cleanCompletedCampaignSentLogs()
    {
        // Get IDs of completed campaigns
        $completed_campaign_ids = ScheduleCampaign::whereStatus('Completed')->pluck('id');

        if($completed_campaign_ids->isNotEmpty()) {
            // Delete sent logs for completed campaigns
            $deleted_count = ScheduleCampaignSentLog::whereIn('schedule_campaign_id', $completed_campaign_ids)->delete();
            
            $this->info('Cleaned ' . $deleted_count . ' sent log entries for ' . $completed_campaign_ids->count() . ' completed campaigns');
        }
    }
}
