<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Http\Helper\Helper;
use App\Models\GlobalBounce;

class Maintenance extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'relayzo:maintenance';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Maintain Application';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting maintenance tasks...');
        
        // Check if update is available
        try {
            $this->_checkUpdateAvailable();
            $this->info('✓ Update check completed');
        } catch(\Exception $e) {
            \Log::error('relayzo:maintenance:update_check => '.$e->getMessage());
        }

        // Verify domains set by auto_verify
        try {
            $this->_domainVerifications();
            $this->info('✓ Domain verifications completed');
        } catch(\Exception $e) {
            \Log::error('relayzo:maintenance:domain_verifications => '.$e->getMessage());
        }

        // Check DNSLB Lookups
        try {
            Helper::checkIPsDomainsLookup();
            $this->info('✓ DNS lookups completed');
        } catch(\Exception $e) {
            \Log::error('relayzo:maintenance:dns_lookups => '.$e->getMessage());
        }

        /**
        * Verify License to App work smoothly
        * WARNING!!! Avoid to make any changes in it.
        */
        try {
            $this->verify();
            $this->info('✓ License verification completed');
        } catch(\Exception $e) {
            \Log::error('relayzo:maintenance:license_verification => '.$e->getMessage());
        }

        // Delete Soft bounces
        try {
            $this->_softBouncesClear();
            $this->info('✓ Soft bounces cleanup completed');
        } catch(\Exception $e) {
            \Log::error('relayzo:maintenance:soft_bounces => '.$e->getMessage());
        }
        
        $this->info('Maintenance tasks completed successfully.');
    }

    private function _checkUpdateAvailable()
    {
        $settings = \DB::table('settings')->whereId(config('custom.app_id'))->select('current_version')->first();
        $available_version = Helper::getUrl(config('custom.version_live_url'));

        if($available_version > $settings->current_version) {
          // save notification
          $notification = [
            'name' => __('app.update_available_msg'),
            'type' => 'update',
            'attributes' => null,
            'app_id' => config('custom.app_id'),
            'user_id' => config('custom.superadmin')
          ];
          \App\Models\Notification::create($notification);
        }
    }

    /**
    * Verify sending domains keys dkim, dmarc, spf, tracking
    */
    private function _domainVerifications()
    {
        $sending_domains = \App\Models\SendingDomain::whereIsActive(true)
            ->whereAutoVerification(true)
            ->get();

        foreach($sending_domains as $sending_domain) {
            Helper::verifyDKIM($sending_domain);

            // Reduced delay for better performance
            usleep(500000); // 0.5 seconds instead of 2 seconds
            // Verify SPF
            Helper::verifySPF($sending_domain);

            // Reduced delay for better performance
            usleep(500000); // 0.5 seconds instead of 2 seconds
            // Verify DMARC
            Helper::verifyDMARC($sending_domain);
        }

        // Tracking Domains Verification
        $tracking_domains = \App\Models\TrackingDomain::whereIsActive(true)
            ->whereAutoVerification(true)
            ->get();

        foreach($tracking_domains as $tracking_domain) {
            Helper::verifyTracking($tracking_domain);
        }
    }

    // get soft bounces older than 1 day and delete - keep recent ones for review
    private function _softBouncesClear()
    {
        // Only delete soft bounces that are older than 1 day
        // This allows users to review recent soft bounces before they're cleaned up
        $cutoff_date = \Carbon\Carbon::now()->subDays(1);
        
        $soft_bounces = GlobalBounce::whereType('Soft')
            ->where('created_at', '<', $cutoff_date)
            ->get();
        
        if($soft_bounces->isEmpty()) {
            return;
        }
        
        $emails = $soft_bounces->pluck('email')->toArray();
        $bounce_ids = $soft_bounces->pluck('id')->toArray();
        
        // Batch update contacts - only reset is_bounced if no hard bounces exist for the email
        foreach($emails as $email) {
            // Check if there are any hard bounces for this email
            $has_hard_bounce = GlobalBounce::whereEmail($email)->whereType('Hard')->exists();
            if(!$has_hard_bounce) {
                \App\Models\Contact::whereEmail($email)->update(['is_bounced' => false]);
            }
        }
        
        // Batch delete old soft bounces
        GlobalBounce::whereIn('id', $bounce_ids)->delete();
        
        $this->info("Cleaned up " . count($bounce_ids) . " soft bounces older than 1 day");
    }

    /**
    * Verify License to App work smoothly
    * WARNING!!! Avoid to make any changes in it.
    */
    private function verify()
    {
        $settings = \DB::table('settings')->whereId(config('custom.app_id'))->select('license_key', 'server_ip', 'app_url', 'from_email')->first();
        $data = [
          'is_auto_verify' => true,
          'license_key' => $settings->license_key,
          'server_ip' => $settings->server_ip,
          'domain' =>  $settings->app_url,
          'email' => $settings->from_email
        ];

        try {
          $result = json_decode(Helper::verifyLicense($data), true);

          if(!empty($result['message']) && ($result['message'] == 'Invalid License Key' || $result['message'] == 'License is Already Using')) {
             \DB::table('settings')->whereId(config('custom.app_id'))->update(['license_key' => null]);
          }
        } catch(\Exception $e) {
            //echo $e->getMessage();
          // \Log::error('relayzo:maintenance 5 => '.$e->getMessage());
        }
    }
}
