<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Webklex\PHPIMAP\ClientManager;
use App\Models\Bounce;
use App\Models\Contact;
use App\Http\Helper\Helper;

class ProcessBounces extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'process:bounces';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Process application bounces';

    public $code = '5.1.1';
    public $type = 'Hard';
    public $short_detail = null;
    public $message_id = null;
    public $to = null;
    public $full_detail = null;
    public $app_id = 1;

    /**
     * Execute the console command.
     */
    public function handle()
    {
        // Proces system bounces
        try {
          $this->systemBounces();
        } catch (\Exception $e) {
          $error = $e->getMessage();
          \Log::error("process:bounces => ".$error);
        }
    }


    private function systemBounces()
    {
        $bounces = Bounce::active()->get();

        foreach($bounces as $bounce) {
          $validate_cert = $bounce->validate_cert == 'Yes' ? true : false;
          $password = !empty($bounce->password) ? \Crypt::decrypt($bounce->password) : '';

          // Create a new ClientManager instance
            $clientManager = new ClientManager();

            // Use configuration directly from database
            $config = [
                'host'          => $bounce->host,
                'port'          => $bounce->port,
                'encryption'    => $bounce->encryption,
                'validate_cert' => $validate_cert,
                'username'      => $bounce->username,
                'password'      => $password,
                'protocol'      => $bounce->method
            ];

            $client = $clientManager->make($config);

            // Test basic connectivity first
            if (!$this->testConnection($bounce->host, $config)) {
                $this->error("Basic connectivity test failed. Skipping this bounce server.");
                continue;
            }

          try {
            //Connect to the IMAP Server
            // $this->info("Connecting to: " . $bounce->host);
            
            // Try to connect with timeout handling
            try {
            $client->connect();
                // $this->info("Successfully connected to: " . $bounce->host);
            } catch (\Exception $connectError) {
                $this->error("Connection failed: " . $connectError->getMessage());
                $this->error("Host: " . $bounce->host);
                $this->error("Port: " . $bounce->port);
                $this->error("Encryption: " . $bounce->encryption);
                $this->error("Username: " . $bounce->username);
                
                throw $connectError;
            }
            //Get all Mailboxes
            try {
              // $this->info("Getting folders from: " . $bounce->host);
              
              // Check client connection status
              // $this->info("Client connection status: " . ($client->isConnected() ? 'Connected' : 'Disconnected'));
              
              $aFolder = $client->getFolders(false); // false for Gmail to avoid subfolder issues
              // $this->info("Found " . count($aFolder) . " folders");
              
              // Debug: List all available folders
              foreach ($aFolder as $folder) {
                // $this->info("Available folder: " . $folder->name);
              }
              
              // Filter folders to check for bounces
              $aFolder = $this->getCustomFolders($aFolder);
              // $this->info("Filtered to " . count($aFolder) . " folders for bounce processing");
            } catch (\Exception $e) {
              $error = $e->getMessage();
              $this->error("getFolders error: " . $error);
              print_r($error);
              \Log::error("process:bounces getFolders => ".$error);
              $aFolder = [];
            }
            //Loop through every Mailbox
            foreach($aFolder as $oFolder){
              $folderName = $oFolder->name;
              // $this->info("Processing folder: " . $folderName);
              
              try {
                // Test basic folder access first
                // $this->info("Testing folder access for: {$folderName}");
                
                // Test folder connection with a simple method
                try {
                  // Try to get folder status first
                  $status = $oFolder->getStatus();
                  // $this->info("Folder '{$folderName}' status: " . json_encode($status));
                  
                  // Use status message count instead of query count
                  $messageCount = $status['messages'] ?? 0;
                  // $this->info("Folder '{$folderName}' has {$messageCount} total messages");
                  
                  if ($messageCount == 0) {
                    // $this->info("No messages in folder '{$folderName}', skipping");
                    continue;
                  }
                } catch (\Exception $countError) {
                  $this->error("Error getting folder status for {$folderName}: " . $countError->getMessage());
                  continue; // Skip this folder if we can't get status
                }
                
                //Get all Messages from the current Mailbox $oFolder
                // $this->info("Processing all messages from folder: {$folderName}");
                
                try {
                  // Try the most basic approach - get all messages without pagination
                  // $this->info("Attempting to get all messages from folder: {$folderName}");
                  
                  // First, let's try to get messages with a very basic query
                  $aMessage = $oFolder->messages()->all()->get();
                  // $this->info("Found " . count($aMessage) . " messages in folder {$folderName}");
                  
                  if (count($aMessage) == 0) {
                    // $this->info("No messages found in folder {$folderName}");
                    continue; // Skip to next folder
                  }
                  
                  // Process all messages at once instead of pagination
                  // $this->info("Processing all " . count($aMessage) . " messages from folder {$folderName}");
                  
                } catch (\Exception $queryError) {
                  $this->error("Error getting messages for {$folderName}: " . $queryError->getMessage());
                  // $this->info("Trying alternative approach with basic query for folder: {$folderName}");
                  
                  // Try with basic query
                  try {
                    $aMessage = $oFolder->query()->all()->get();
                    // $this->info("Basic query found " . count($aMessage) . " messages");
                    
                    if (count($aMessage) == 0) {
                      // $this->info("No messages found with basic query");
                      continue; // Skip to next folder
                    }
                  } catch (\Exception $altError) {
                    $this->error("Basic query also failed: " . $altError->getMessage());
                    // $this->info("Trying raw IMAP approach for folder: {$folderName}");
                    
                    // Try raw IMAP approach as last resort
                    try {
                      // Get the raw IMAP connection and try basic commands
                      $connection = $oFolder->getClient()->getConnection();
                      // $this->info("Using raw IMAP connection for folder: {$folderName}");
                      
                      // Try to get message UIDs first
                      $uids = $connection->search(['ALL']);
                      // $this->info("Raw IMAP found " . count($uids) . " message UIDs");
                      
                      if (empty($uids)) {
                        // $this->info("No message UIDs found with raw IMAP");
                        continue;
                      }
                      
                      // Process messages using raw IMAP UIDs
                      // $this->info("Raw IMAP access successful, processing messages by UID");
                      // Try to get messages by UIDs
                      try {
                        $aMessage = [];
                        foreach ($uids as $uid) {
                          try {
                            $message = $oFolder->query()->getMessageByUid($uid);
                            if ($message) {
                              $aMessage[] = $message;
                            }
                          } catch (\Exception $uidError) {
                            // Skip this UID if it fails
                            $this->warn("Failed to get message UID {$uid}: " . $uidError->getMessage());
                            continue;
                          }
                        }
                        // $this->info("Retrieved " . count($aMessage) . " messages via raw IMAP");
                        if (empty($aMessage)) {
                          continue;
                        }
                      } catch (\Exception $processError) {
                        $this->error("Failed to process raw IMAP messages: " . $processError->getMessage());
                        continue;
                      }
                      
                    } catch (\Exception $rawError) {
                      $this->error("Raw IMAP access also failed: " . $rawError->getMessage());
                      // $this->info("All message retrieval methods failed for folder: {$folderName}");
                      continue; // Skip to next folder
                    }
                  }
                }
                
                try {
                  foreach($aMessage as $oMessage) {
                      // $this->info("Processing message: " . $oMessage->getSubject());
                      
                      try {
                        $processed = false;
                        $aAttachment = $oMessage->getAttachments();
                        // $this->info("Message has " . count($aAttachment) . " attachments");
                        
                        // Process attachments first (most common case)
                        if (count($aAttachment) > 0) {
                          $aAttachment->each(function ($oAttachment) use ($oMessage, $bounce, &$processed) {
                            try {
                              $content = $oAttachment->getContent();
                              // $this->info("Processing attachment: " . $oAttachment->getName());
                              
                              // Get bounce type and short detail
                              $code = Helper::extractString($content, 'Status:', "\n");

                              // Avoid to save the data for X-OutGoing-Spam-Status: No, score=-1.0
                              if(stripos($code, 'score') === false) {
                                if(!empty($code)) {
                                  $bounce_code_detail = Helper::bouceCodes($code);
                                  if(!empty($bounce_code_detail)) {
                                    $this->type = $bounce_code_detail['type'];
                                    $this->short_detail = $bounce_code_detail['description'];
                                    $this->code = $code;
                                    // $this->info("Found bounce code: " . $code);
                                  }
                                }
                                $this->full_detail = !empty($oMessage->getTextBody()) ? $oMessage->getTextBody() : $oMessage->getHTMLBody(true);
                              }

                              // Try to extract Message-ID and RZ-Type-ID from attachment
                              $message_id = null;
                              $RZ_type = null;
                              
                              if(strpos($content, 'Message-ID:') !== false) {
                                $message_id = Helper::extractString($content, 'Message-ID:', "\n");
                                // $this->info("Found Message-ID in attachment: " . $message_id);
                              }
                              
                              try {
                                // To process only campaign bounces; RZ_type can be other like split-test, drip etc
                                $RZ_type = Helper::extractString($content, 'RZ-Type-ID:', "\n");
                              } catch(\Exception $e) {
                                // Try alternative extraction methods
                                $RZ_type = Helper::extractString($content, 'RZ-Type-ID', "\n");
                                if (empty($RZ_type)) {
                                  $RZ_type = null;
                                }
                              }

                              // Process bounce if we have RZ_type or Message-ID
                              if (!empty($RZ_type) || !empty($message_id)) {
                                if (empty($RZ_type)) {
                                  $RZ_type = 'campaign';
                                }
                                
                                list($app_id, $stat_log_id, $stat_id, $to_email, $section) = Helper::toEamil_Section($RZ_type, $bounce->app_id, 'Bounced');

                                // Validate before saving
                                if (!empty($stat_id) && !empty($to_email)) {
                                  Helper::saveBounce($stat_id, $to_email, $stat_log_id, $section, $this->code, $this->type, $this->short_detail, $this->full_detail, $bounce->app_id);
                                  
                                  // update contact set bounced
                                  Contact::whereEmail($to_email)->update([
                                    'is_bounced' => true
                                  ]);
                                  
                                  // $this->info("Processed bounce for: " . $to_email);
                                  $processed = true;
                                } else {
                                  $this->warn("Skipping bounce - missing stat_id or to_email. stat_id: {$stat_id}, to_email: {$to_email}, RZ_type: {$RZ_type}");
                                  \Log::warning("process:bounces skipped - missing data", [
                                    'stat_id' => $stat_id,
                                    'to_email' => $to_email,
                                    'RZ_type' => $RZ_type,
                                    'message_id' => $message_id
                                  ]);
                                }
                              }
                            } catch (\Exception $attachError) {
                              $this->error("Error processing attachment: " . $attachError->getMessage());
                              \Log::error("process:bounces attachment processing => ".$attachError->getMessage());
                            }
                          });
                        }
                        
                        // If no attachments or processing failed, try processing email body
                        if (!$processed) {
                          // $this->info("No attachments or attachment processing failed, trying email body");
                          try {
                            $bodyContent = !empty($oMessage->getTextBody()) ? $oMessage->getTextBody() : $oMessage->getHTMLBody(true);
                            
                            if (!empty($bodyContent)) {
                              // Get bounce type and short detail from body
                              $code = Helper::extractString($bodyContent, 'Status:', "\n");
                              
                              // Also try alternative bounce code formats
                              if (empty($code)) {
                                $code = Helper::extractString($bodyContent, 'Diagnostic-Code:', "\n");
                              }
                              if (empty($code)) {
                                $code = Helper::extractString($bodyContent, 'X-Postfix-Queue-ID:', "\n");
                              }

                              // Avoid to save the data for X-OutGoing-Spam-Status: No, score=-1.0
                              if(stripos($code, 'score') === false) {
                                if(!empty($code)) {
                                  $bounce_code_detail = Helper::bouceCodes($code);
                                  if(!empty($bounce_code_detail)) {
                                    $this->type = $bounce_code_detail['type'];
                                    $this->short_detail = $bounce_code_detail['description'];
                                    $this->code = $code;
                                    // $this->info("Found bounce code in body: " . $code);
                                  }
                                }
                                $this->full_detail = $bodyContent;
                              }

                              // Try to extract RZ-Type-ID from body (check HTML for relayzo ref)
                              $RZ_type = null;
                              
                              // First try RZ-Type-ID header
                              try {
                                $RZ_type = Helper::extractString($bodyContent, 'RZ-Type-ID:', "\n");
                              } catch(\Exception $e) {
                                // Try alternative extraction
                                $RZ_type = Helper::extractString($bodyContent, 'RZ-Type-ID', "\n");
                              }
                              
                              // If not found, try getRelayzoRef from HTML (like ProcessReplies does)
                              if (empty($RZ_type)) {
                                try {
                                  $htmlBody = $oMessage->getHTMLBody(true);
                                  if (!empty($htmlBody)) {
                                    $RZ_type = Helper::getRelayzoRef($htmlBody);
                                  }
                                } catch(\Exception $e) {
                                  // Ignore
                                }
                              }
                              
                              // If still no RZ_type, try to extract from Message-ID or other headers
                              if (empty($RZ_type)) {
                                $message_id = Helper::extractString($bodyContent, 'Message-ID:', "\n");
                                if (empty($message_id)) {
                                  // Try from message headers
                                  $headers = $oMessage->getHeader();
                                  if (!empty($headers)) {
                                    $message_id = $headers->get('message-id');
                                  }
                                }
                                
                                if (!empty($message_id)) {
                                  // $this->info("Found Message-ID in body: " . $message_id);
                                  // Default to campaign if we have Message-ID but no RZ_type
                                  $RZ_type = 'campaign';
                                }
                              }

                              // Process bounce if we have RZ_type
                              if (!empty($RZ_type)) {
                                list($app_id, $stat_log_id, $stat_id, $to_email, $section) = Helper::toEamil_Section($RZ_type, $bounce->app_id, 'Bounced');

                                // Validate before saving
                                if (!empty($stat_id) && !empty($to_email)) {
                                  Helper::saveBounce($stat_id, $to_email, $stat_log_id, $section, $this->code, $this->type, $this->short_detail, $this->full_detail, $bounce->app_id);
                                  
                                  // update contact set bounced
                                  Contact::whereEmail($to_email)->update([
                                    'is_bounced' => true
                                  ]);
                                  
                                  // $this->info("Processed bounce from body for: " . $to_email);
                                  $processed = true;
                                } else {
                                  $this->warn("Skipping bounce from body - missing stat_id or to_email. stat_id: {$stat_id}, to_email: {$to_email}, RZ_type: {$RZ_type}");
                                  \Log::warning("process:bounces skipped from body - missing data", [
                                    'stat_id' => $stat_id,
                                    'to_email' => $to_email,
                                    'RZ_type' => $RZ_type
                                  ]);
                                }
                              } else {
                                $this->warn("Could not extract RZ-Type-ID or Message-ID from bounce email body");
                                \Log::warning("process:bounces - no RZ-Type-ID or Message-ID found in body", [
                                  'subject' => $oMessage->getSubject(),
                                  'from' => $oMessage->getFrom()
                                ]);
                              }
                            }
                          } catch (\Exception $bodyError) {
                            $this->error("Error processing email body: " . $bodyError->getMessage());
                            \Log::error("process:bounces body processing => ".$bodyError->getMessage());
                          }
                        }

                        if($bounce->delete_after_processing == 'Yes' && $processed) {
                          // Delete the message after process
                          $oMessage->delete();
                          // $this->info("Deleted message after processing");
                        } elseif($bounce->delete_after_processing == 'Yes' && !$processed) {
                          $this->warn("Message not deleted - bounce was not successfully processed");
                        }
                      } catch (\Exception $e) {
                        $this->error("Error processing message: " . $e->getMessage());
                        \Log::error("process:bounces message processing => ".$e->getMessage());
                      }
                    }
                  } catch (\Exception $e) {
                    $this->error("Error processing messages: " . $e->getMessage());
                    \Log::error("process:bounces message processing => ".$e->getMessage());
                  }
                } catch (\Exception $e) {
                  $this->error("Error processing folder {$folderName}: " . $e->getMessage());
                  \Log::error("process:bounces folder {$folderName} => ".$e->getMessage());
                }     
            }
          } catch (\Exception $e) {
            $error = $e->getMessage();
            \Log::error("process:bounces => ".$error);
          } finally {
            try {
              if (method_exists($client, 'isConnected') ? $client->isConnected() : true) {
                $client->disconnect();
              }
            } catch (\Throwable $t) {
              // swallow disconnect errors but log
              \Log::warning('process:bounces disconnect warning: '.$t->getMessage());
            }
          }
        }
    }




    /**
     * Get custom IMAP folders (for relayzo.com and others)
     */
    private function getCustomFolders($allFolders)
    {
        $customFolders = [];
        
        // Exact folder names (case-sensitive for exact matches)
        $exactFolders = ['INBOX', 'Bounces', 'Failed'];
        
        // Spam-related folder names (case-insensitive matching)
        $spamKeywords = ['spam', 'junk', 'bulk'];
        
        foreach ($allFolders as $folder) {
            $folderName = $folder->name;
            $folderNameLower = strtolower($folderName);
            $added = false;
            
            // Check for exact matches first (INBOX, Bounces, Failed)
            if (in_array($folderName, $exactFolders)) {
                $customFolders[] = $folder;
                // $this->info("Added exact match folder: " . $folderName);
                $added = true;
            }
            
            // Check for spam-related folders (case-insensitive)
            if (!$added) {
                foreach ($spamKeywords as $keyword) {
                    if (stripos($folderNameLower, $keyword) !== false) {
                        $customFolders[] = $folder;
                        // $this->info("Added spam-related folder: " . $folderName . " (matched keyword: {$keyword})");
                        $added = true;
                        break;
                    }
                }
            }
            
            // Also check for common folder patterns like "[Gmail]/Spam", "Junk E-mail", etc.
            if (!$added) {
                // Remove common prefixes/suffixes and check
                $normalizedName = preg_replace('/^\[.*?\]\//', '', $folderName); // Remove [Gmail]/ prefix
                $normalizedName = strtolower(trim($normalizedName));
                
                // Check normalized name against spam keywords
                foreach ($spamKeywords as $keyword) {
                    if (stripos($normalizedName, $keyword) !== false) {
                        $customFolders[] = $folder;
                        // $this->info("Added normalized spam folder: " . $folderName . " (normalized: {$normalizedName})");
                        $added = true;
                        break;
                    }
                }
            }
        }
        
        // If no specific folders found, use all folders (to ensure we don't miss anything)
        if (empty($customFolders)) {
            // $this->info("No specific folders found, using all available folders to ensure no bounces are missed");
            return $allFolders;
        }
        
        // $this->info("Total folders to process: " . count($customFolders));
        return collect($customFolders);
    }

    /**
     * Test connection to email provider
     */
    private function testConnection($provider, $config)
    {
        // $this->info("Testing connection configuration:");
        // $this->info("Host: " . $config['host']);
        // $this->info("Port: " . $config['port']);
        // $this->info("Encryption: " . $config['encryption']);
        // $this->info("Username: " . $config['username']);
        // $this->info("Validate Cert: " . ($config['validate_cert'] ? 'Yes' : 'No'));
        
        // Test basic connectivity
        // $this->info("Testing basic connectivity...");
        $connection = @fsockopen($config['host'], $config['port'], $errno, $errstr, 10);
        
        if (!$connection) {
            $this->error("Cannot connect to {$config['host']}:{$config['port']}");
            $this->error("Error: $errstr ($errno)");
            return false;
        } else {
            // $this->info("Basic connectivity test passed");
            fclose($connection);
            return true;
        }
    }
}
