<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Webklex\PHPIMAP\ClientManager;
use App\Models\Fbl;
use App\Http\Helper\Helper;

class ProcessSpams extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'process:spams';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Process application spams';
    public $app_id = 1;

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $flbs = Fbl::active()->get();
        
        if($flbs->isEmpty()) {
            // $this->info('No FBL servers to process.');
            return;
        }
        
        // $this->info("Processing {$flbs->count()} FBL servers...");
        
        foreach($flbs as $fbl) {
            $validate_cert = $fbl->validate_cert == 'Yes' ? true : false;
            $password = !empty($fbl->password) ? \Crypt::decrypt($fbl->password) : '';

            $clientManager = new ClientManager();

            $client = $clientManager->make([
                'host'          => $fbl->host,
                'port'          => $fbl->port,
                'encryption'    => $fbl->encryption,
                'validate_cert' => $validate_cert,
                'username'      => $fbl->username,
                'password'      => $password,
                'protocol'      => $fbl->method
            ]);

            try {
                //Connect to the IMAP Server
                $client->connect();

                //Get all Mailboxes
                try {
                    // $this->info("Getting folders from: " . $fbl->host);
                    
                    // Check client connection status
                    // $this->info("Client connection status: " . ($client->isConnected() ? 'Connected' : 'Disconnected'));
                    
                    $aFolder = $client->getFolders(false); // false for Gmail to avoid subfolder issues
                    // $this->info("Found " . count($aFolder) . " folders");
                    
                    // Debug: List all available folders
                    foreach ($aFolder as $folder) {
                        // $this->info("Available folder: " . $folder->name);
                    }
                    
                    // Filter folders to check for spams
                    $aFolder = $this->getCustomFolders($aFolder);
                    // $this->info("Filtered to " . count($aFolder) . " folders for spam processing");
                } catch (\Exception $e) {
                    $error = $e->getMessage();
                    $this->error("getFolders error: " . $error);
                    \Log::error("process:spams getFolders => ".$error);
                    $aFolder = [];
                }

                //Loop through every Mailbox
                foreach($aFolder as $oFolder){
                    $folderName = $oFolder->name;
                    // $this->info("Processing folder: " . $folderName);
                    
                    try {
                        // Test basic folder access first
                        // $this->info("Testing folder access for: {$folderName}");
                        
                        // Test folder connection with a simple method
                        try {
                            // Try to get folder status first
                            $status = $oFolder->getStatus();
                            // $this->info("Folder '{$folderName}' status: " . json_encode($status));
                            
                            // Use status message count instead of query count
                            $messageCount = $status['messages'] ?? 0;
                            // $this->info("Folder '{$folderName}' has {$messageCount} total messages");
                            
                            if ($messageCount == 0) {
                                // $this->info("No messages in folder '{$folderName}', skipping");
                                continue;
                            }
                        } catch (\Exception $countError) {
                            $this->error("Error getting folder status for {$folderName}: " . $countError->getMessage());
                            continue; // Skip this folder if we can't get status
                        }
                        
                        //Get all Messages from the current Mailbox $oFolder
                        // $this->info("Processing all messages from folder: {$folderName}");
                        
                        try {
                            // Try the most basic approach - get all messages without pagination
                            // $this->info("Attempting to get all messages from folder: {$folderName}");
                            
                            // First, let's try to get messages with a very basic query
                            $aMessage = $oFolder->messages()->all()->get();
                            // $this->info("Found " . count($aMessage) . " messages in folder {$folderName}");
                            
                            if (count($aMessage) == 0) {
                                // $this->info("No messages found in folder {$folderName}");
                                continue; // Skip to next folder
                            }
                            
                            // Process all messages at once instead of pagination
                            // $this->info("Processing all " . count($aMessage) . " messages from folder {$folderName}");
                            
                        } catch (\Exception $queryError) {
                            $this->error("Error getting messages for {$folderName}: " . $queryError->getMessage());
                            // $this->info("Trying alternative approach with basic query for folder: {$folderName}");
                            
                            // Try with basic query
                            try {
                                $aMessage = $oFolder->query()->all()->get();
                                // $this->info("Basic query found " . count($aMessage) . " messages");
                                
                                if (count($aMessage) == 0) {
                                    // $this->info("No messages found with basic query");
                                    continue; // Skip to next folder
                                }
                            } catch (\Exception $altError) {
                                $this->error("Basic query also failed: " . $altError->getMessage());
                                // $this->info("Trying raw IMAP approach for folder: {$folderName}");
                                
                                // Try raw IMAP approach as last resort
                                try {
                                    // Get the raw IMAP connection and try basic commands
                                    $connection = $oFolder->getClient()->getConnection();
                                    // $this->info("Using raw IMAP connection for folder: {$folderName}");
                                    
                                    // Try to get message UIDs first
                                    $uids = $connection->search(['ALL']);
                                    // $this->info("Raw IMAP found " . count($uids) . " message UIDs");
                                    
                                    if (empty($uids)) {
                                        // $this->info("No message UIDs found with raw IMAP");
                                        continue;
                                    }
                                    
                                    // Process messages using raw IMAP UIDs
                                    // $this->info("Raw IMAP access successful, processing messages by UID");
                                    // Try to get messages by UIDs
                                    try {
                                        $aMessage = [];
                                        foreach ($uids as $uid) {
                                            try {
                                                $message = $oFolder->query()->getMessageByUid($uid);
                                                if ($message) {
                                                    $aMessage[] = $message;
                                                }
                                            } catch (\Exception $uidError) {
                                                // Skip this UID if it fails
                                                $this->warn("Failed to get message UID {$uid}: " . $uidError->getMessage());
                                                continue;
                                            }
                                        }
                                        // $this->info("Retrieved " . count($aMessage) . " messages via raw IMAP");
                                        if (empty($aMessage)) {
                                            continue;
                                        }
                                    } catch (\Exception $processError) {
                                        $this->error("Failed to process raw IMAP messages: " . $processError->getMessage());
                                        continue;
                                    }
                                    
                                } catch (\Exception $rawError) {
                                    $this->error("Raw IMAP access also failed: " . $rawError->getMessage());
                                    // $this->info("All message retrieval methods failed for folder: {$folderName}");
                                    continue; // Skip to next folder
                                }
                            }
                        }
                        
                        try {
                            foreach($aMessage as $oMessage) {     
                                try {
                                    // $this->info("Processing message: " . $oMessage->getSubject());
                                    
                                    $processed = false;
                                    
                                    // Try to get HTML body first (most common case)
                                    try {
                                        $full_detail = $oMessage->getHTMLBody(true);
                                    } catch (\Exception $htmlError) {
                                        // If HTML body fails, try text body
                                        try {
                                            $full_detail = $oMessage->getTextBody();
                                            if (empty($full_detail)) {
                                                $full_detail = $oMessage->getHTMLBody(true);
                                            }
                                        } catch (\Exception $textError) {
                                            $this->warn("Could not retrieve message body: " . $textError->getMessage());
                                            \Log::warning("process:spams - could not get message body", [
                                                'subject' => $oMessage->getSubject(),
                                                'error' => $textError->getMessage()
                                            ]);
                                            continue;
                                        }
                                    }

                                    if (empty($full_detail)) {
                                        $this->warn("Message body is empty, skipping");
                                        \Log::warning("process:spams - empty message body", [
                                            'subject' => $oMessage->getSubject()
                                        ]);
                                        continue;
                                    }

                                    try {
                                        // To process only campaign spams; RZ_type can be other like split-test, drip etc
                                        $RZ_type = Helper::getRelayzoRef($full_detail);
                                    } catch(\Exception $e) {
                                        $RZ_type = 'campaign';
                                        \Log::warning("process:spams getRelayzoRef => ".$e->getMessage());
                                    }

                                    list($app_id, $stat_log_id, $stat_id, $to_email, $section) = Helper::toEamil_Section($RZ_type, $fbl->app_id, 'Spammed');
                                    
                                    // Validate before saving
                                    if (!empty($stat_id) && !empty($to_email)) {
                                        Helper::saveSpam($stat_id, $to_email, $stat_log_id, $section, $full_detail, $app_id);
                                        // $this->info("Saved spam for email: {$to_email}");
                                        $processed = true;
                                    } else {
                                        $this->warn("Skipping spam - missing stat_id or to_email. stat_id: {$stat_id}, to_email: {$to_email}, RZ_type: {$RZ_type}");
                                        \Log::warning("process:spams skipped - missing data", [
                                            'stat_id' => $stat_id,
                                            'to_email' => $to_email,
                                            'RZ_type' => $RZ_type
                                        ]);
                                    }
                                    
                                    if($fbl->delete_after_processing == 'Yes' && $processed) {
                                        // Delete the message after process
                                        $oMessage->delete();
                                        // $this->info("Deleted message after processing");
                                    } elseif($fbl->delete_after_processing == 'Yes' && !$processed) {
                                        $this->warn("Message not deleted - spam was not successfully processed");
                                    }
                                
                                } catch (\Exception $e) {
                                    $this->error("Error processing message: " . $e->getMessage());
                                    \Log::error("process:spams message processing => ".$e->getMessage());
                                    // Continue processing other messages
                                }
                            }
                        } catch (\Exception $e) {
                            $this->error("Error processing messages: " . $e->getMessage());
                            \Log::error("process:spams message processing => ".$e->getMessage());
                        }
                    } catch (\Exception $e) {
                        $this->error("Error processing folder {$folderName}: " . $e->getMessage());
                        \Log::error("process:spams folder {$folderName} => ".$e->getMessage());
                    }
                }
            } catch (\Exception $e) {
                \Log::error("process:spams => ".$e->getMessage());
            } finally {
                try {
                    if (method_exists($client, 'isConnected') ? $client->isConnected() : true) {
                        $client->disconnect();
                    }
                } catch (\Throwable $t) {
                    \Log::warning('process:spams disconnect warning: '.$t->getMessage());
                }
            }
        }
    }
    
    /**
     * Filter folders to only process relevant ones for spam processing
     */
    private function getCustomFolders($allFolders)
    {
        $customFolders = [];
        
        // Exact folder names (case-sensitive for exact matches)
        $exactFolders = ['INBOX', 'Failed'];
        
        // Spam-related folder names (case-insensitive matching)
        $spamKeywords = ['spam', 'junk', 'bulk', 'trash', 'deleted'];
        
        foreach ($allFolders as $folder) {
            $folderName = $folder->name;
            $folderNameLower = strtolower($folderName);
            $added = false;
            
            // Check for exact matches first (INBOX, Failed)
            if (in_array($folderName, $exactFolders)) {
                $customFolders[] = $folder;
                // $this->info("Added exact match folder: " . $folderName);
                $added = true;
            }
            
            // Check for spam-related folders (case-insensitive)
            if (!$added) {
                foreach ($spamKeywords as $keyword) {
                    if (stripos($folderNameLower, $keyword) !== false) {
                        $customFolders[] = $folder;
                        // $this->info("Added spam-related folder: " . $folderName . " (matched keyword: {$keyword})");
                        $added = true;
                        break;
                    }
                }
            }
            
            // Also check for common folder patterns like "[Gmail]/Spam", "Junk E-mail", etc.
            if (!$added) {
                // Remove common prefixes/suffixes and check
                $normalizedName = preg_replace('/^\[.*?\]\//', '', $folderName); // Remove [Gmail]/ prefix
                $normalizedName = strtolower(trim($normalizedName));
                
                // Check normalized name against spam keywords
                foreach ($spamKeywords as $keyword) {
                    if (stripos($normalizedName, $keyword) !== false) {
                        $customFolders[] = $folder;
                        // $this->info("Added normalized spam folder: " . $folderName . " (normalized: {$normalizedName})");
                        $added = true;
                        break;
                    }
                }
            }
        }
        
        // If no specific folders found, use all folders (to ensure we don't miss anything)
        if (empty($customFolders)) {
            // $this->info("No specific folders found, using all available folders to ensure no spams are missed");
            return $allFolders;
        }
        
        // $this->info("Total folders to process: " . count($customFolders));
        return collect($customFolders);
    }
}
