<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\SplitTest;
use App\Models\SplitTestStat;
use App\Models\SplitTestStatLog;
use App\Models\SendingServer;
use App\Models\Contact;
use App\Http\Helper\Helper;
use App\Services\ElasticEmailService;
use App\Services\SendGridService;
use App\Services\RssFeedService;
use DB;

class RunSplitTestsCampaigns extends Command
{
  protected $signature = 'run:split-test-campaigns {id?} {thread_no?}';
  protected $description = 'Run campaigns for the scheduled split tests';

  public function __construct()
  {
      parent::__construct();
  }

  public function handle()
  {
    $id = $this->argument('id');
    if($id) {
      $thread_no = $this->argument('thread_no');

      // Get a scheduled split test need to send
      $split_test = SplitTest::findOrFail($id);

      // Check sending status and set status to completed
      $this->_checkSplitTestStatus($split_test->id);

      // For some reason if thread_no > total_threads
      if($thread_no > $split_test->total_threads) {
        SplitTest::whereId($split_test->id)->update(['thread_no' => 1, 'status' => 'Resume']);
        exit;
      }

      // If campaign is set for hourly limit then updte the time to 1 hour laterl send further
      $split_test_sending_speed = json_decode($split_test->sending_speed);
      if($split_test_sending_speed->speed == 'limited') {
        $carbon = new \Carbon\Carbon();

        $initial_limit = $split_test_sending_speed->initial_limit??$split_test_sending_speed->limit;
        $next_sending_datetime_minutes =  round(($initial_limit/$split_test_sending_speed->limit) * 60);
        $next_datetime = $carbon->parse($carbon->now(), config('app.timezone'))->addMinutes($next_sending_datetime_minutes);

        SplitTest::whereId($split_test->id)->update(['status' => 'RunningLimit', 'sending_datetime' => $next_datetime]);
      } else {
        $this->_setSplitTestStatus($split_test->id, 'Running');
        $next_datetime = null;
      }

      $this->_sendSplitTest($split_test, $thread_no, $next_datetime);


    } else {
      $this->_processSplitTest();
    }
  }

  /**
   * Send split test to subscribers
  */
  protected function _sendSplitTest($split_test, $thread_no, $next_datetime)
  {
    // Retrun array of sending servers
    $split_test->sending_server_ids;
    $sending_servers = SendingServer::getActiveSeningServers(json_decode($split_test->sending_server_ids, true), 'array');

    // If no sending server then no need to do anything
    if(empty($sending_servers)) {
      $this->_setSplitTestStatus($split_test->id, 'System Paused');
    }

    // Will be use to reset the sending_server array when  all picked
    $sending_servers_data = $sending_servers;

    // Get split test stat info will be use later
    $split_test_stat = SplitTestStat::whereSplitTestId($split_test->id)->first();

    $settings = DB::table('settings')->whereId(config('custom.app_id'))->first();
    $tracking = $settings->tracking == 'enabled' ? true : false;
    $app_url = $settings->app_url;

    // mail headers will be set for admin 
    $mail_headers = json_decode(DB::table('users')->select('mail_headers')->whereAppId($split_test->app_id)->whereParentId(0)->value('mail_headers'), true);


    if(json_decode($split_test->sending_speed)->speed == 'limited') {
      $condition = $thread_no;
      $limit = json_decode($split_test->sending_speed)->limit;      
      $wait = floor(3600 / $limit); // divide on 1 hour

      // pause afte send
      if(!empty(json_decode($split_test->sending_speed)->pause_limit)) {
        $pause_limit = (int) json_decode($split_test->sending_speed)->pause_limit;
        // if it will greater than 0 then add pause time to the next sending batch, 
        if($pause_limit > 0) {
          $carbon = new \Carbon\Carbon();
          $pause_next_datetime = $carbon->parse($next_datetime, config('app.timezone'))->addMinutes($pause_limit);
          SplitTest::whereId($split_test->id)->update(['send_datetime' => $pause_next_datetime]);
        }
      }
    } else {
      $condition = $split_test->total_threads;
      $limit = null;
      $wait = false;
    }

    if(empty($split_test->winning_campaign_id)) {
      // get broadcasts
      $broadcasts = \App\Models\Broadcast::whereIn('id', json_decode($split_test->broadcast_ids, true))->get()->toArray();
    } else {
      $broadcasts = \App\Models\Broadcast::whereId($split_test->winning_campaign_id)->get()->toArray();
    }
    
    // Will be use to reset the sending_server array when  all picked
    $broadcasts_data = $broadcasts;

    // Need to execute the loop according to the threads start with thread no
    for($file_no=$thread_no; $file_no<=$condition; $file_no+=$split_test->threads) {
      $split_test->increment('thread_no');

      $path_split_test = str_replace('[user-id]', $split_test->user_id, config('custom.path_split_test'));
      $file = $path_split_test.$split_test->id.DIRECTORY_SEPARATOR. $file_no . '.csv';

      if(file_exists($file)) {
        $file_offsets = DB::table('file_offsets')->where('file', $file)->first();
        // Get file offset
        $offset_file = empty($file_offsets) ? 0 : $file_offsets->offset;
        if($offset_file) {
          // Delete entry after picking up offset may be use in next time
          DB::table('file_offsets')->where('file', $file)->delete();
        }
        $reader = \League\Csv\Reader::createFromPath($file, 'r');
        // Make associative array with names and skipp header
        $reader->setHeaderOffset(0);
        // It may possible split test paused in past then it wil continue form the same recored
        $stmt = (new \League\Csv\Statement())->offset($offset_file);
        //$contacts_csv = $reader->getRecords();
        $contacts_csv = $stmt->process($reader);

        $total_records = Helper::getCsvCount($file);

        foreach ($contacts_csv as $offset => $contact_csv) {
          // Need to stop the split test instantally as split test status is paused
          // Decrement should work for latest value
          $split_test = SplitTest::findOrFail($split_test->id);
          if ($split_test->status == 'Paused' || $split_test->status == 'System Paused') {
            // Store record no into db to execuet the file from same location instead from start when split test resumed
            DB::table('file_offsets')->insert([
              'file' => $file,
              'offset' => --$offset // Minus 1 before to save other wise next recored will be picked when resumed
            ]);

            // The same file should be select when resumed
            $split_test->decrement('thread_no', 1);
            if($limit) {
              $carbon = new \Carbon\Carbon();
              $sending_datetime = $carbon->parse($carbon->now(), config('app.timezone'));
              SplitTest::whereId($split_test->id)->update(['sending_datetime' => $sending_datetime]);
            }
            // Stop Split Test sending
            exit;
          }

          // Sending servers info that assigned previously to reset the sending servers when all picked
          if(empty($sending_servers)) {
            $sending_servers = $sending_servers_data;
          }
          // get sending server
          $sending_server = array_shift($sending_servers);

          // Refil broadcasts data
          if(empty($broadcasts)) {
            $broadcasts = $broadcasts_data;
          }
          // get broadcast
          $broadcast = array_shift($broadcasts);

          $from_name = (empty($contact_csv['FROM_NAME']) || trim($contact_csv['FROM_NAME']) === '') ? $sending_server['from_name'] : $contact_csv['FROM_NAME'];
          $from_email = (empty($contact_csv['FROM_EMAIL']) || trim($contact_csv['FROM_EMAIL']) === '') ? $sending_server['from_email'] : $contact_csv['FROM_EMAIL'];
          $reply_email = (empty($contact_csv['REPLY_EMAIL']) || trim($contact_csv['REPLY_EMAIL']) === '') ? $sending_server['reply_email'] : $contact_csv['REPLY_EMAIL'];

          $sending_domain = Helper::getSendingDomainFromEmail($from_email);
          // if no domain found
          try {
            $domain = $sending_domain->protocol.$sending_domain->domain;
          } catch(\Exception $e) {
            continue;
          }
          $message_id = Helper::getCustomMessageID($domain);

          // Try to connect with SendingServer
          $connection = Helper::configureSendingNode($sending_server['type'], $sending_server['sending_attributes'], $message_id);
          if($connection['success']) {
            $contact = Contact::whereId($contact_csv['ID'])->with('customFields')->first();


            if(!empty($sending_server['tracking_domain'])) {
              $tracking_domain = $sending_server['tracking_domain'];
            } else {
              $tracking_domain = Helper::getAppURL();
            }

            $from_name = Helper::replaceSpintags($from_name);
            $reply_email = Helper::replaceSpintags($reply_email);
            $reply_email = filter_var($reply_email, FILTER_VALIDATE_EMAIL) ? $reply_email : null;



            $broadcast_content = (new RssFeedService())->parseAndRender($broadcast['content_html']);

            // Replace spintags
            $broadcast_content = Helper::replaceSpintags(Helper::decodeString($broadcast_content));
            $email_subject = Helper::replaceSpintags(Helper::decodeString($broadcast['email_subject']));

            // Replace custom field
            $broadcast_content = Helper::replaceCustomFields($broadcast_content, $contact->customFields);
            $email_subject = Helper::replaceCustomFields($email_subject, $contact->customFields);

            // Replace Spintax [Random: hi|hello|hey]
            $broadcast_content = Helper::spinTaxParser($broadcast_content);
            $email_subject = Helper::spinTaxParser($email_subject);

            // Default Parser
            $broadcast_content = Helper::defaultValueParser($broadcast_content);
            $email_subject = Helper::defaultValueParser($email_subject);


            // Create SplitTestStatLog and the id would be use to track the email
            $split_test_stat_log_data = [
              'split_test_stat_id' => $split_test_stat->id,
              'message_id' => $message_id,
              'email' => $contact_csv['EMAIL'],
              'list' => $contact_csv['LIST'],
              'broadcast' => $broadcast['name'],
              'sending_server' => $sending_server['name'],
            ];
            try {
              $split_test_stat_log = SplitTestStatLog::create($split_test_stat_log_data);
            } catch(\Exception $e) {
                  continue;
            }

            // Data that will be use to replce the system variables
            $data_values = [
              'broadcast-id'   => $broadcast['id'],
              'broadcast-name' => $broadcast['name'],
              'sender-name'    => $from_name,
              'sender-email'   => $sending_server['from_email'],
              'domain'         => $tracking_domain,
              'message-id'     => $message_id,
              'campaign-id'    => base64_encode($split_test_stat_log->id),
              'type'           => 'st'
            ];

            // Replace system variables
            $subject = Helper::replaceSystemVariables($contact, $email_subject, $data_values);
            $content = Helper::replaceSystemVariables($contact, $broadcast_content, $data_values);

            // for to get replies info
            $content .= "<span id='rz-ref-splittest-{$split_test_stat_log->id}-{$split_test->app_id}-rz-ref'></span>";

            // If tracking is enabled, for TEXT format the tracing will not work
            if($tracking) {
              // click tracking should be before track_opens becuase don't want to convert that url
              $content = Helper::convertLinksForClickTracking($split_test_stat_log->id, $tracking_domain, $content, $app_url, '/st/click/');

              // Make open tracking url and pixel
              $track_open = $tracking_domain.'/st/open/'.base64_encode($split_test_stat_log->id);
              $content .= "<div style='float:left; clear:both; font-family:Arial; margin:40px auto; width:100%; line-height:175%; font-size:11px; color:#434343'><img border='0' src='".$track_open."' width='1' height='1' alt=''></div>";
            }

            // If sending type that supported by framework will be send with a same way
            if(in_array($sending_server['type'], Helper::sendingServersFramworkSuported())) {

                $message = new \Symfony\Component\Mime\Email();
                $message->from(new \Symfony\Component\Mime\Address($from_email, "$from_name"));
                $message->to($contact_csv['EMAIL']);
                $message->subject($subject);
                !empty($reply_email) ? $message->replyTo($reply_email) : '';
                if(!empty($sending_server['bounce']['email'])) {
                  $message->returnPath($sending_server['bounce']['email']);
                }

                // adding the envelope becuase bounce email having issue
                $envelope = new \Symfony\Component\Mailer\Envelope(
                    new \Symfony\Component\Mime\Address($from_email, "$from_name"), // From email
                    [new \Symfony\Component\Mime\Address($contact_csv['EMAIL'])] // Envelope recipient(s)
                );

                $headers= $message->getHeaders();
                $headers->addIdHeader('Message-ID', $message_id);
                // Header will use to process the bounces and fbls etc.
                $headers->addTextHeader('RZ-Type-ID', "splittest-{$split_test_stat_log->id}-{$split_test_stat->app_id}");
                // Required header for good inboxing - use proper unsubscribe URL
                $unsub_url = $tracking_domain . '/contact/do-unsub/' . base64_encode($contact->id) . "/{$split_test_stat_log->id}/{$data_values['type']}";
                $headers->addTextHeader('List-Unsubscribe', "<{$unsub_url}>");
                $headers->addTextHeader('List-Unsubscribe-Post', 'List-Unsubscribe=One-Click');

                // Header will be use to process reports for Amazon
                if($sending_server['type'] == 'amazon_ses_api' && !empty(json_decode($sending_server['sending_attributes'])->process_reports) && json_decode($sending_server['sending_attributes'])->process_reports == 'yes') {
                  if(!empty(json_decode($sending_server['sending_attributes'])->amazon_configuration_set)) {
                    $headers->addTextHeader('X-SES-CONFIGURATION-SET', json_decode($sending_server['sending_attributes'])->amazon_configuration_set);
                  }
                }

                if(!empty($mail_headers)) {
                  foreach($mail_headers as $header_key => $header_value) {
                    $header_value = Helper::replaceSpintags($header_value);
                    $header_value = Helper::replaceCustomFields($header_value, $contact->customFields);
                    $header_value = Helper::replaceSystemVariables($contact, $header_value, $data_values);
                    $header_value = Helper::spinTaxParser($header_value);

                    $header_key = Helper::replaceHyphen($header_key);

                    $message->getHeaders()->addTextHeader($header_key, $header_value);
                  }
                }

                $message->html($content);
                $message->text(htmlspecialchars_decode(strip_tags($content)));

                // Will be check either need to load sending servers again or not
                $load_sending_servers = false;
                try {
                  $connection['transport']->send($message);
                  $status = 'Sent';
                } catch(\Exception $e) {
                  \Log::error('run:splittest-campaigns => '.$e->getMessage());
                  $status = 'Failed';
                }
            } elseif($sending_server['type'] == 'sendgrid_api') {
              $load_sending_servers = false;
              try {
                SendGridService::send(json_decode($sending_server['sending_attributes'])->api_key,
                $from_email, $from_name, $contact_csv['EMAIL'], $reply_email, $subject, $content, 'html', $message_id,
                );
                $status = 'Sent';
              } catch(\Exception $e) {
                \Log::error('run:splittest-sendgrid => '.$e->getMessage());
                $status = 'Failed';
              }
            } elseif($sending_server['type'] == 'elastic_email_api') {
              try {
                $load_sending_servers = false;
                ElasticEmailService::send(json_decode($sending_server['sending_attributes'])->api_key,
                  $from_email, $from_name, $contact_csv['EMAIL'], $reply_email, $subject, $content, $message_id
                );
                $status = 'Sent';
              } catch(\Exception $e) {
                \Log::error('run:splittest-elastic_email => '.$e->getMessage());
                $status = 'Failed';
              }
            } elseif($sending_server['type'] == 'postal_api') {
              try {
                $result = \App\Services\PostalAPIService::send(
                    json_decode($sending_server['sending_attributes'])->postal_server_url,
                    json_decode($sending_server['sending_attributes'])->api_key,
                    $from_email, $from_name, $contact_csv['EMAIL'], $reply_email, $subject, $content, 'html', $message_id
                );
                if($result === 'send') {
                    $status = 'Sent';
                } else {
                    $status = 'Failed';
                }
              } catch(\Exception $e) {
                \Log::error('run:splittest-postal => '.$e->getMessage());
                $status = 'Failed';
              }
            }

            $sending_server_data = Helper::updateSendingServerCounters($sending_server['id']);

            if($sending_server_data['sending_server_paused']) {
              $load_sending_servers = true;
            }
            // update sent counter with 1 for both tables
            $split_test->increment('sent');
            $split_test_stat->increment('sent');

            // Check if sent >= decision percentage with fresh data of sent
            if(empty($split_test->winning_campaign_id)) {
              $total_sent = SplitTest::whereId($split_test->id)->value('sent');
              if($total_sent >= $split_test->percentage_sent) {
                // Update sending datatime for leftover sending
                $action_data = json_decode($split_test->action_data);
                $carbon = new \Carbon\Carbon();
                if($action_data->send_remaining_after_duration == 'minutes') {
                  $next_sending_datetime = $carbon->parse($carbon->now(), config('app.timezone'))->addMinutes((int) $action_data->send_remaining_after);
                } elseif($action_data->send_remaining_after_duration == 'hours') {
                  $next_sending_datetime = $carbon->parse($carbon->now(), config('app.timezone'))->addHours((int) $action_data->send_remaining_after);
                } elseif($action_data->send_remaining_after_duration == 'days') {
                  $next_sending_datetime = $carbon->parse($carbon->now(), config('app.timezone'))->addDays((int) $action_data->send_remaining_after);
                }

                // should be 3 minute less, so set correct winning campaing id
                $decision_datetime = date('Y-m-d H:i:s', strtotime("{$next_sending_datetime} - 3 minute"));
                // Update to next sending datetime
                if($split_test->action != 'show_results') {
                  SplitTest::whereId($split_test->id)->update([
                    'decision_datetime' => $decision_datetime,
                  ]);
                } else {
                  SplitTest::whereId($split_test->id)->update([
                    'sending_datetime' => $next_sending_datetime,
                    'decision_datetime' => $decision_datetime,
                    'status' => 'Paused'
                  ]);
                  // The same file should be select when resumed
                  $split_test->decrement('thread_no', 1);
                }
                exit;
              }
            }

            // Update status
            SplitTestStatLog::whereId($split_test_stat_log->id)->update(['status' => $status]);

            // Check sending status and set status to completed
            $this->_checkSplitTestStatus($split_test->id);

          } else {
            // If sending server connection failed then need to update it as system inactive
            // Removing single and double quote due to output issue with js alert at frontend 
            SendingServer::whereId($sending_server['id'])->update(['status' => 'System Inactive', 'notification' => str_replace( ["'",'"'], '', explode('.',$connection['msg'])[0] )]); 

            $load_sending_servers = true;
          }

          if($load_sending_servers) {
            // Retrun new array of sending servers after make a sending server as system inactive
            $sending_servers_data = SendingServer::getActiveSeningServers(explode(',', $split_test->sending_server_ids), 'array');

            // If no sending server then no need to do anything
            if(empty($sending_servers_data)) {
              $this->_setSplitTestStatus($split_test->id, 'System Paused');
            }
          }

          if($wait) sleep($wait);
        } // End foreach $contacts

        // Check sending status and set status to completed
        $this->_checkSplitTestStatus($split_test->id);
        try {
          // Delete File after process
          unlink($file);
        } catch (\Exception $e) {
          \Log::error('run:splittest-campaigns => '.$e->getMessage());
        }
      }
    } // End for loop $threads
  } // End Function

  /**
   * Needs to send parallel request so doing this
  */
  protected function _processSplitTest()
  {
    Helper::getUrl(Helper::getAppURL().'/process_split_test');
  }

  /**
   * Check sending status of split test if all sent the set status to completed
  */
  protected function _checkSplitTestStatus($id)
  {
    // picking up fresh entries
    $split_test = SplitTest::findOrFail($id);
    if($split_test->sent >= $split_test->scheduled) {
      SplitTestStat::whereSplitTestId($split_test->id)->update(['end_datetime' => \Carbon\Carbon::now()]);
      SplitTest::whereId($split_test->id)->update(['status' => 'Completed']);
      exit;
    }
  }

  /**
   * Update scheduled split test status
  */
  protected function _setSplitTestStatus($id, $status)
  {
    SplitTest::whereId($id)->update(['status' => $status]);
  }
}
