<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\ScheduleCampaign;
use Carbon\Carbon;

class ScheduledAutoNext extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'schedule:auto-next';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Duplicate completed campaigns with auto_scheduled_next enabled';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting ScheduledAutoNext process...');
        
        // Find completed campaigns with auto_scheduled_next = true
        $completedCampaigns = ScheduleCampaign::where('status', 'Completed')
            ->where('auto_scheduled_next', true)
            ->get();
            
        if ($completedCampaigns->isEmpty()) {
            $this->info('No completed campaigns with auto_scheduled_next found.');
            return;
        }
        
        $duplicatedCount = 0;
        
        foreach ($completedCampaigns as $campaign) {
            try {
                // Create new campaign with updated values
                // Now new campaign will be set as auto scheduled next to true otherwise it then old campaing will call each time.
                $newCampaign = $campaign->replicate();
                $newCampaign->auto_scheduled_next = true;
                $newCampaign->status = 'Preparing';
                $newCampaign->send = 'now'; // Set to 'now' for immediate sending
                $newCampaign->send_datetime = now();
                $newCampaign->created_at = now();
                $newCampaign->updated_at = now();
                
                // Reset campaign-specific fields
                $newCampaign->total_threads = 0;
                $newCampaign->thread_no = 0;
                $newCampaign->total = 0;
                $newCampaign->scheduled = 0;
                $newCampaign->sent = 0;
                $newCampaign->scheduled_detail = null;
                
                // Update name to indicate it's a new scheduled campaign
                $newCampaign->name = $campaign->name;
                
                $newCampaign->save();

                \App\Jobs\CampaignPrepare::dispatch($newCampaign->id)
                    ->delay(now()->addSeconds(10));
                
                // Update the original campaign to disable auto_scheduled_next
                $campaign->auto_scheduled_next = false;
                $campaign->save();
                
                $duplicatedCount++;
                $this->info("Duplicated campaign: {$campaign->name} -> {$newCampaign->name}");
                
            } catch (\Exception $e) {
                $this->error("Failed to duplicate campaign {$campaign->name}: " . $e->getMessage());
            }
        }
        
        $this->info("ScheduledAutoNext completed. Duplicated {$duplicatedCount} campaigns.");
        
        return Command::SUCCESS;
    }
}