<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Http\Helper\Helper;

class SetCounter extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'set:counter';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Set the counters';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting counter updates...');
        
        try {
            $this->setListsCounts();
            $this->info('✓ List counts updated');
        } catch(\Exception $e) {
            \Log::error('set:counter-setListsCounts: '.$e->getMessage());
        }

        try {
            $this->setSendingServerSendingLimitCounters();
            $this->info('✓ Sending server counters updated');
        } catch(\Exception $e) {
            \Log::error('set:counter-setSendingServerSendingLimitCounters => '.$e->getMessage());
        }

        try {
            $this->setSendingServerStatus();
            $this->info('✓ Sending server status updated');
        } catch(\Exception $e) {
            \Log::error('set:counter-setSendingServerStatus => '.$e->getMessage());
        }

        // Should be after setSendingServerStatus() so sending servers already checked
        try {
            $this->setScheduledCampaignStatus();
            $this->info('✓ Scheduled campaign status updated');
        } catch(\Exception $e) {
            \Log::error('set:counter-setScheduledCampaignStatus => '.$e->getMessage());
        }

        // Reset thread_no if it exceeds total_threads for Not Completed campaigns
        try {
            $this->resetThreadNoForExceeded();
            $this->info('✓ Thread no reset check completed');
        } catch(\Exception $e) {
            \Log::error('set:counter-resetThreadNoForExceeded => '.$e->getMessage());
        }

        // Re-Run campaign if stopped for any reason
        try {
            $this->rerunCampaign();
            $this->info('✓ Campaign rerun check completed');
        } catch(\Exception $e) {
            \Log::error('set:counter-rerunCampaign => '.$e->getMessage());
        }

        // Mark unsubscribed contacts as suppressed
        try {
            $this->markUnsubscribedAsSuppressed();
            $this->info('✓ Unsubscribed contacts marked as suppressed');
        } catch(\Exception $e) {
            \Log::error('set:counter-markUnsubscribedAsSuppressed => '.$e->getMessage());
        }
        
        $this->info('Counter updates completed successfully.');
    }

    /**
     * Re-Run the running status campaing stopped for some reason
    */
    protected function rerunCampaign() 
    {
      $schedules = \App\Models\ScheduleCampaign::whereIn('status', ['Running'])
          ->orderBy('id', 'desc')
          ->get();
        if(count($schedules) > 0) {
          foreach($schedules as $schedule) {
            $schedle_campaign_stat_id = \App\Models\ScheduleCampaignStat::whereScheduleCampaignId($schedule->id)->value('id');
            $campaign_stat_log = \App\Models\ScheduleCampaignStatLog::whereScheduleCampaignStatId($schedle_campaign_stat_id)
              ->orderBy('id', 'desc')
              ->limit(1)
              ->get();
            // get last email send difference in minutes
            $diff = \Carbon\Carbon::parse($campaign_stat_log[0]->created_at)->diffInMinutes(\Carbon\Carbon::now());
            // if more than 15 min
            if($diff > 15) {
              // update to scheduled from running, then it will again start
              \App\Models\ScheduleCampaign::whereId($schedule->id)->update(['status' => 'Scheduled']);
            }
          }
        }
    }

    /**
    * Re-set the sending limit and set the next timestamp - optimize with batch processing
    */
    protected function setListsCounts()
    {
        $lists = \App\Models\Lists::select('id')->get();
        
        if($lists->isEmpty()) {
            return;
        }
        
        // Batch update all lists at once
        foreach($lists as $list) {
            $total_contacts = \App\Models\Contact::whereListId($list->id)->count();
            
            \App\Models\Lists::whereId($list->id)->update([
                'total_contacts' => $total_contacts
            ]);
        }
    }

    /**
    * Re-set the sending limit and set the next timestamp
    */
    protected function setSendingServerSendingLimitCounters()
    {
        \App\Models\SendingServer::where('hourly_sent_next_timestamp', '<', \Carbon\Carbon::now())
          ->update([
            'hourly_sent' => 0
            //'hourly_sent_next_timestamp' => \Carbon\Carbon::now()->addHour(1)
          ]);
        
        // Reset daily counters and also reset next_send_timestamp for warmup servers
        // so they can start sending immediately on the new day
        $daily_reset_servers = \App\Models\SendingServer::where('daily_sent_next_timestamp', '<', \Carbon\Carbon::now())
          ->where('daily_sent', '>', 0) // Only if they actually sent something
          ->get();
        
        foreach($daily_reset_servers as $server) {
          $update_data = ['daily_sent' => 0];
          
          // For warmup servers, reset next_send_timestamp so they can send immediately
          $speed_attributes = json_decode($server->speed_attributes);
          if($speed_attributes && $speed_attributes->speed == 'warmup') {
            $update_data['next_send_timestamp'] = \Carbon\Carbon::now();
          }
          
          \App\Models\SendingServer::whereId($server->id)->update($update_data);
        }
        
        // Also reset any servers that haven't sent yet today
        \App\Models\SendingServer::where('daily_sent_next_timestamp', '<', \Carbon\Carbon::now())
          ->where('daily_sent', 0)
          ->update(['daily_sent' => 0]);
    }

    /**
   * Update the sending server status to active for System Inactive / System Paused
  */
  protected function setSendingServerStatus()
  {
    $sending_servers = \App\Models\SendingServer::WhereIn('status', ['System Inactive', 'System Paused'])->get();
    foreach($sending_servers as $sending_server) {
      $status = null;
      if($sending_server->status == 'System Paused') {
        $speed_attributes = json_decode($sending_server->speed_attributes);
        if($speed_attributes->speed == 'limited' || $speed_attributes->speed == 'warmup') {
          if($speed_attributes->duration == 'hourly') {
            if($sending_server->hourly_sent < $speed_attributes->limit) {
              $status = 'Active';
            }
          } elseif($speed_attributes->duration == 'daily') {
            if($sending_server->daily_sent < $speed_attributes->limit) {
              $status = 'Active';
            }
          }
        } else {
          $status = 'Active';
        }
      } elseif($sending_server->status == 'System Inactive') {
        $connection = Helper::configureSendingNode($sending_server->type, $sending_server->sending_attributes);
        if($connection['success']) {
          $status = 'Active';
        }
      }

      if(!empty($status)) {
        $update_data = [
          'status' => $status,
          'notification' => null
        ];
        
        // For warmup servers, also reset next_send_timestamp so they can be picked up immediately
        $speed_attributes = json_decode($sending_server->speed_attributes);
        if($speed_attributes && $speed_attributes->speed == 'warmup' && $sending_server->daily_sent < $speed_attributes->limit) {
          $update_data['next_send_timestamp'] = \Carbon\Carbon::now();
        }
        
        \App\Models\SendingServer::whereId($sending_server->id)->update($update_data);
      }
    }
  }

  /**
   * Update the scheduled campaigns status to active System Paused
  */
  protected function setScheduledCampaignStatus()
  {
    $schedules = \App\Models\ScheduleCampaign::whereStatus('System Paused')
      ->select('id', 'sending_server_ids', 'sending_speed')->get();
    foreach($schedules as $schedule) {
      $sending_serves = \App\Models\SendingServer::WhereIn('id', explode(',', $schedule->sending_server_ids))
      ->where('status', 'Active')
      ->get();

      if(count($sending_serves) > 0) {
        // Scheduled and RunningLimit will be entertain again
        if(json_decode($schedule->sending_speed)->speed == 'limited') {
          \App\Models\ScheduleCampaign::whereId($schedule->id)
            ->update([
              'status' => 'RunningLimit',
              'send_datetime' => \Carbon\Carbon::now()
            ]);
        } else {
          \App\Models\ScheduleCampaign::whereId($schedule->id)
            ->update([
              'status' => 'Resume',
              'thread_no' => 1
            ]);
        };
        
      }
    }
  }

  /**
   * Reset thread_no to 0 if it exceeds total_threads for Not Completed campaigns
  */
  protected function resetThreadNoForExceeded()
  {
    $schedules = \App\Models\ScheduleCampaign::where('status', '!=', 'Completed')
      ->whereNotNull('total_threads')
      ->whereColumn('thread_no', '>', 'total_threads')
      ->get();

    if($schedules->count() > 0) {
      foreach($schedules as $schedule) {
        \App\Models\ScheduleCampaign::whereId($schedule->id)
          ->update([
            'thread_no' => 0
          ]);
      }
    }
  }

  /**
   * Mark unsubscribed contacts as suppressed if unsubscribed_mark_as_suppressed is enabled
   */
  protected function markUnsubscribedAsSuppressed()
  {
    // Get all users where unsubscribed_mark_as_suppressed is true
    $users = \App\Models\User::where('unsubscribed_mark_as_suppressed', true)
      ->select('id', 'app_id')
      ->get();

    if($users->isEmpty()) {
      return;
    }

    $suppression_type_id = config('custom.group_suppression'); // Should be 5

    foreach($users as $user) {
      // Get or create "Unsubscribed" group for this app_id
      $group = \App\Models\Group::where('name', 'Unsubscribed')
        ->where('type_id', $suppression_type_id)
        ->where('app_id', $user->app_id)
        ->first();

      if(!$group) {
        // Create the group if it doesn't exist
        $group = \App\Models\Group::create([
          'name' => 'Unsubscribed',
          'type_id' => $suppression_type_id,
          'app_id' => $user->app_id,
          'user_id' => $user->id
        ]);
      }

      // Get all unsubscribed contacts for this app_id
      $unsubscribed_contacts = \App\Models\Contact::where('is_unsubscribed', true)
        ->where('app_id', $user->app_id)
        ->select('email')
        ->distinct()
        ->get();

      if($unsubscribed_contacts->isEmpty()) {
        continue;
      }

      $added_count = 0;
      $skipped_count = 0;

      foreach($unsubscribed_contacts as $contact) {
        // Check if suppression already exists (unique constraint on group_id + email)
        $exists = \App\Models\Suppression::where('group_id', $group->id)
          ->where('email', $contact->email)
          ->exists();

        if(!$exists) {
          try {
            \App\Models\Suppression::create([
              'email' => $contact->email,
              'group_id' => $group->id,
              'app_id' => $user->app_id,
              'user_id' => $user->id
            ]);
            $added_count++;
          } catch(\Exception $e) {
            // Handle any other exceptions (e.g., duplicate key constraint)
            \Log::warning('set:counter-markUnsubscribedAsSuppressed: Failed to add suppression for email ' . $contact->email . ': ' . $e->getMessage());
            $skipped_count++;
          }
        } else {
          $skipped_count++;
        }
      }

      if($added_count > 0) {
        \Log::info("set:counter-markUnsubscribedAsSuppressed: Added {$added_count} suppressions for app_id {$user->app_id}, skipped {$skipped_count} duplicates");
      }
    }
  }

}
