<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Trigger;
use App\Models\TriggerSchedule;
use App\Models\Contact;
use App\Models\Lists;
use DB;

class Triggers extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'relayzo:triggers';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Run saved triggers';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        // First need to delete or inactive the drips for Inactive triggers others email will be send to those
        $this->_inActiveTriggers();

        // Get active triggers and with execution_datetime
        $triggers = Trigger::whereIsActive(true)
          ->where('execution_datetime', '<=', \Carbon\Carbon::now())
          ->get();

        if($triggers->isEmpty()) {
            $this->info('No active triggers to process.');
            return;
        }
        
        $this->info("Processing {$triggers->count()} triggers...");

        if($triggers->isNotEmpty()) {
          foreach($triggers as $trigger) {
            $trigger_attributes = json_decode($trigger->attributes);

            //print_r($trigger_attributes);
            if($trigger->based_on == 'list') {
              /*
              If trigger action is start drip then only need to insert data into schedule drip
              rest of the functionality will remain same and send emails via ProressTrigger
              */
              if($trigger->action == 'start_drip') {
                if(!\App\Models\DripSchedule::whereTriggerId($trigger->id)->exists()) {
                  // Save schedule drip data
                  $drip_schedule = $this->_saveDripSchedule($trigger);
                  // Need to save data for drip stat also
                  $this->_saveDripStat($drip_schedule);
                } else {
                  \App\Models\DripSchedule::whereTriggerId($trigger->id)->update(['status' => 'running']);
                }
                continue;
              }

              $contacts = Contact::whereIn('list_id', $trigger_attributes->list_ids);

              // get Contact datetime for condition either send for previous for new
              $contact_datetime_marker = $this->_getContactDatetimeMarker($trigger);
              // What contacts need to be get
              $contacts = $contacts->where('created_at', '>=', $contact_datetime_marker);

              $contacts = $contacts->whereIsActive(true)
                ->whereIsUnsubscribed(false)
                ->orderBy('email')
                ->orderBy('id')
                ->chunk(1000, function ($contacts) use ($trigger) {
                  $trigger_attributes = json_decode($trigger->attributes);
                  foreach ($contacts as $contact) {
                    try {
                        // If bounced
                        if(DB::table('global_bounces')->whereEmail($contact->email)->exists()) continue;

                        $contact_send_datetime = $this->_contactSendDatetime($trigger_attributes, $contact->created_at);

                        // Save trigger log data
                        $this->_saveTriggerLog($trigger_attributes, $trigger, $contact->id, $contact_send_datetime);
                    } catch(\Exception $e) {
                        \Log::error('relayzo:triggers contact processing => '.$e->getMessage());
                    }
                  }
                });
            } elseif($trigger->based_on == 'segment' || $trigger->based_on == 'campaign') {
              try {
                if($trigger->based_on == 'campaign') {
                  $segment_type = 'Campaign';
                  $query = \App\Models\Segmentation::querySegmentCampaign($trigger_attributes->schedule_campaign_stat_ids, $trigger->user_id, $trigger_attributes->action_campaign);
                  $query->orderBy('schedule_campaign_stat_logs.id', 'DESC')->distinct();
                } else {
                    // get segment
                    $segment = \App\Models\Segmentation::whereIn('id', $trigger_attributes->segment_ids)->first();
                    $segment_type = $segment->type;
                    if($segment_type == 'List') {
                      $query = \App\Models\Segmentation::querySegmentList($segment->attributes);
                      $query->orderBy('id', 'DESC')->distinct();
                    } elseif($segment_type == 'Campaign') {
                      //$query = \App\Models\Segmentation::querySegmentCampaign($segment->attributes, $segment->user_id);
                      //$query->orderBy('schedule_campaign_stat_logs.id', 'DESC')->distinct();
                    } 
                }
                
              } catch(\Exception $e) {
                \Log::error('relayzo:triggers => '.$e->getMessage());
              }

              $contact_datetime_marker = $this->_getContactDatetimeMarker($trigger);

              if($segment_type == 'List') {
                // What contacts need to be get
                $query = $query->whereIsActive(true)->whereIsUnsubscribed(false)->where('created_at', '>=', $contact_datetime_marker);
              }

              // Get segmented data with 1000 limit
              $query->chunk(1000, function($contacts) use ($trigger, $segment_type, $contact_datetime_marker) {
                $trigger_attributes = json_decode($trigger->attributes);
                foreach($contacts as $contact) {
                  try {
                    // If bounced
                    if(DB::table('global_bounces')->whereEmail($contact->email)->exists()) continue;

                    $contact_send_datetime = $this->_contactSendDatetime($trigger_attributes, $contact->created_at);

                    // We did not added the contact query about send datetime for campaign so need to check here
                    if($segment_type == 'Campaign') {
                      // $contact is basically schedule_campaign_stat_logs in campaign case
                      // get Contact Id with list id
                      $list_id = Lists::whereName(trim($contact->list))->value('id');
                      $contact_id = Contact::whereEmail(trim($contact->email))->whereListId($list_id)->value('id');
                      //if(Contact::whereActive('Yes')->whereUnsubscribed('No')->where('created_at', '>=', $contact_datetime_marker)->exists()){
                        $this->_saveTriggerLog($trigger_attributes, $trigger, $contact_id,  $contact_send_datetime);
                     // }
                    } else {
                      // else if segment_type == list, already query has been filtered
                      $this->_saveTriggerLog($trigger_attributes, $trigger, $contact->id, $contact_send_datetime);
                    }
                  } catch(\Exception $e) {
                    \Log::error('relayzo:triggers segment processing => '.$e->getMessage());
                  }
                }
              });
            } elseif($trigger->based_on == 'date') {
              // run only when datetime is 
              if(strtotime($trigger_attributes->send_datetime) < strtotime(\Carbon\Carbon::now())) {
                /*
                If trigger action is start drip then only need to insert data into schedule drip
                rest of the functionality will remain same and send emails via ProressTrigger
                */
                if($trigger->action == 'start_drip') {
                  if(!\App\Models\DripSchedule::whereTriggerId($trigger->id)->exists()) {
                    // Save schedule drip data
                    $drip_schedule = $this->_saveDripSchedule($trigger);
                    // Need to save data for drip stat also
                    $this->_saveDripStat($drip_schedule);
                  } else {
                    \App\Models\DripSchedule::whereTriggerId($trigger->id)->update(['status' => 'running']);
                  }
                  continue;
                }

                $contacts = Contact::whereIn('list_id', $trigger_attributes->list_ids);
                // get Contact datetime for condition either send for pervious for new
                $contact_datetime_marker = $this->_getContactDatetimeMarker($trigger);
                // What contacts need to be get
                $contacts = $contacts->where('created_at', '>=', $contact_datetime_marker);
                $contacts = $contacts->whereIsActive(true)
                  ->whereIsUnsubscribed(false)
                  ->orderBy('email')
                  ->orderBy('id')
                  ->chunk(1000, function ($contacts) use ($trigger) {
                    $trigger_attributes = json_decode($trigger->attributes);
                    foreach ($contacts as $contact) {
                      try {
                         // If bounced
                        if(DB::table('global_bounces')->whereEmail($contact->email)->exists()) continue;

                        $contact_send_datetime = $this->_contactSendDatetime($trigger_attributes, $contact->created_at);

                        // Save trigger log data
                        $this->_saveTriggerLog($trigger_attributes, $trigger, $contact->id, $contact_send_datetime);
                      } catch(\Exception $e) {
                        \Log::error('relayzo:triggers date processing => '.$e->getMessage());
                      }
                    }
                  });
              }
            }
          } // end foreach triggers
        } // end if no trigger
        
        $this->info('Trigger processing completed successfully.');
    }

    private function _getContactDatetimeMarker($trigger) {
        $trigger_log = DB::table('trigger_logs')->whereTriggerId($trigger->id)->orderBy('id', 'desc')->first();

        // If no trigger data into logs
        $now = \Carbon\Carbon::now();
        if(empty($trigger_log)) {
          $trigger_log_datetime = $now;
        } else {
          $trigger_log_datetime = $trigger_log->created_at;
        }

        // Save last trigger date time
        DB::table('trigger_logs')->insert(['trigger_id' => $trigger->id, 'created_at' => $now]);

        $trigger_attributes = json_decode($trigger->attributes);
        if($trigger_attributes->send_to_existing == 'no') {
          $contact_datetime_marker = $trigger_log_datetime;
        } elseif($trigger_attributes->send_to_existing == 'yes') {
          // default would be for all if no entry in log
          if(empty($trigger_log)) {
            $contact_datetime_marker = '2000-01-01 00:00:00';
          } else {
            $contact_datetime_marker = $trigger_log_datetime;
          }
        }
        return $contact_datetime_marker;
    }


    // Get contact send datetime for trigger
    private function _contactSendDatetime($trigger_attributes, $contact_created_at)
    {
        $carbon = new \Carbon\Carbon();
        // If trigger is instant otherwise according to time is set
        if($trigger_attributes->trigger_start == 'instant') {
          $contact_send_datetime = $carbon->now();
        } else {
          $duration = (int) $trigger_attributes->trigger_start_limit;
          if($trigger_attributes->trigger_start_duration == 'minutes') {
            $contact_send_datetime = $carbon->parse($contact_created_at)->addMinutes($duration);
          } elseif($trigger_attributes->trigger_start_duration == 'hours') {
            $contact_send_datetime = $carbon->parse($contact_created_at)->addHours($duration);
          } elseif($trigger_attributes->trigger_start_duration == 'days') {
            $contact_send_datetime = $carbon->parse($contact_created_at)->addDays($duration);
          }
        }
        return $contact_send_datetime;
    }

    /**
    * Save drip schedule
    */
    private function _saveDripSchedule($trigger)
    {
        $trigger_attributes = json_decode($trigger->attributes);
        $drip_schedule = [
          'name' => $trigger->name,
          'drip_group_id' => $trigger_attributes->drip_group_id,
          'list_ids' => implode(',', $trigger_attributes->list_ids),
          'sending_server_ids' => implode(',', json_decode($trigger_attributes->sending_server_ids)),
          'send_to_existing' => ($trigger_attributes->send_to_existing == 'no' ? 'No' : 'Yes'),
          'status' => 'running',
          'trigger_id' => $trigger->id,
          'user_id' => $trigger->user_id,
          'app_id' => $trigger->app_id
        ];
        // Save DripSchedule data
        $drip_schedule =  \App\Models\DripSchedule::create($drip_schedule);

        return $drip_schedule;
    }

    /**
    * Save data for drip schedule stat
    */
    private function _saveDripStat($drip_schedule) {
        $data['drip_schedule_id'] = $drip_schedule->id;
        $data['schedule_by'] = \App\Models\User::getUserValue($drip_schedule->user_id, 'name');
        $data['drip_schedule_name'] = $drip_schedule->name;
        $data['drip_group_name'] = \App\Models\Group::whereId($drip_schedule->drip_group_id)->value('name');
        $data['app_id'] = $drip_schedule->app_id;
        $data['user_id'] = $drip_schedule->user_id;

        \App\Models\DripScheduleStat::create($data);
    }

    // Save trigger logs
    private function _saveTriggerLog($trigger_attributes, $trigger, $contact_id, $contact_send_datetime)
    {
        // Pick sending server id
        $sending_servers = json_decode($trigger_attributes->sending_server_ids);
        $sending_server_id_key = array_rand($sending_servers, 1);

        // if action is send campaign
        if($trigger->action == 'send_campaign') {
          $trigger_schedules = [
            'trigger_id' => $trigger->id,
            'contact_id' => $contact_id,
            'send_datetime' => $contact_send_datetime,
            'email_subject' => $trigger_attributes->email_subject ?? null,
            'content_html' => $trigger_attributes->content_html ?? null,
            'sending_server_id' => $sending_servers[$sending_server_id_key] ?? null,
            'action' => $trigger->action
          ];
          // Save data to send email
          TriggerSchedule::create($trigger_schedules);
        }
    }

    private function _inActiveTriggers()
    {
        // Get inactive triggres
        $triggers = Trigger::whereisActive(false)->get();
        if(!empty($triggers)) {
          foreach($triggers as $trigger) {
            if($trigger->action == 'start_sequence') {
              \App\Models\DripSchedule::whereTriggerId($trigger->id)->update(['status' => 'paused']);
            } else {
              // Delete data from Trigger Logs
              TriggerSchedule::whereTriggerId($trigger->id)->delete();
            }
          }
        }
    }
}
