<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Models\CustomField;
use App\Http\Helper\Helper;

class CustomFieldController extends Controller
{
    /**
    * Retrun custom fields
    */
    public function getCustomFields(Request $request): JsonResponse
    {
        $result = CustomField::whereAppId($request->user('api')->app_id);

        if(!empty($request->start_from))
          $result->offset($request->start_from);

        $limit = !empty($request->limit) ? $request->limit : 100; 
        $data = $result->limit($limit)->get();

        return Helper::apiReturn(__('app.success'), $data);
    }

    /**
    * Retrun a custom field detail
    */
    public function get(Request $request, $id): JsonResponse
    {
        $custom_field = CustomField::whereId($id)->whereAppId($request->user('api')->app_id)->first();

        if(!empty($custom_field)) {
          $data = [
            'id' => $custom_field->id,
            'name' => $custom_field->name,
            'type' => $custom_field->type,
            'values' => Helper::splitLineBreakWithComma($custom_field->values),
            'created_at' => Helper::datetimeDisplay($custom_field->created_at)
          ];
          $status = __('app.success');
        } else {
          $status = __('app.error');
          $data = ['message' => __('app.no_record_found')];
        }

        return Helper::apiReturn($status, $data);
    }

    /**
    * Add a custom field 
    */
    public function add(Request $request)
    {
        $input = $this->_customFieldData($request);

        if(in_array($input['type'], ['radio', 'checkbox', 'dropdown'])) {
          $input['values'] = implode(',', $input['values']);
        }

        $status = __('app.error');
        if(empty($input['name'])) {
          $data = [
            'message' => 'name ' . __('app.is_required'),
          ];
        } else {
          $custom_field = CustomField::whereAppId($input['app_id'])->whereName($input['name'])->select('id')->first();
          if(!empty($custom_field->id)) {
            $data = [
              'id' => $custom_field->id,
              'message' => __('app.custom_field') .' '. __('app.already_exist'),
            ];
          } else {
            $custom_field = CustomField::create($input);
            activity('create')->withProperties(['app_id' => $input['app_id']])->log(__('app.custom_field') . " ({$input['name']}) ". __('app.log_create'));

            $status = __('app.success');
            $data = [
              'id' => $custom_field->id,
              'name' => $input['name'],
              'type' => $input['type']
            ];
          }
        }
        return Helper::apiReturn($status, $data);
    }

    /**
    * Update a custom field 
    */
    public function update(Request $request, $id)
    {
        $input = $this->_customFieldData($request);
        $status = __('app.error');
        $data = [];
        
        $custom_field = CustomField::findOrFail($id);
        $custom_field->fill($input)->save();

        activity('update')->withProperties(['app_id' => $input['app_id']])->log(__('app.custom_field') . " ({$input['name']}) ". __('app.log_update'));

        $status = __('app.success');
        $data = [
            'id' => $id,
            'message' => __('app.custom_field') .' '. __('app.log_update'),
        ];


        return Helper::apiReturn($status, $data);
    }

    /**
    * Return inputed data
    */
    public function _customFieldData($request)
    {
        $input = $request->all();
        $input['name'] = !empty($input['name']) ? trim($input['name']) : null;
        $input['tag'] = \Illuminate\Support\Str::slug($input['name'], '-');
        $input['values'] = !empty($input['values']) ? $input['values'] : null;
        $input['type'] = !empty($input['type']) ? $input['type'] : 'text';
        $input['user_id'] = $request->user('api')->id;
        $input['app_id'] = $request->user('api')->app_id;
        return $input;
    }

    /**
    * Delete a custom field 
    */
    public function destroy(Request $request, $id)
    {
        $name = CustomField::whereId($id)->whereAppId($request->user('api')->app_id)->value('name');
        if(!empty($name)) {
          $destroy = CustomField::destroy($id);
          activity('delete')->withProperties(['app_id' => $request->user('api')->app_id])->log(__('app.custom_field') . " ({$name}) ". __('app.log_delete'));

          $status = __('app.success');
          $data = [
            'id' => $id,
            'name' => $name
          ];
        } else {
          $status = __('app.error');
          $data = ['message' => __('app.no_record_found')];
        }
        return Helper::apiReturn($status, $data);
    }
}
