<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Models\Group;
use App\Http\Helper\Helper;

class GroupController extends Controller
{
    
    /**
    * Retrun group types
    */
    public function getGroupTypes(Request $request): JsonResponse
    {
        $group_types = [
            __('app.lists') => 1,
            __('app.campaigns') => 2,
            __('app.delivery_servers') => 3,
            __('app.sending_domains') => 4,
            __('app.suppressions') => 5,
        ];
        return Helper::apiReturn(__('app.success'), $group_types);
    }
    
    /**
    * Retrun groups
    */
    public function getGroups(Request $request): JsonResponse
    {
        $result = Group::whereAppId($request->user('api')->app_id)
          ->select('id', 'name', 'type_id', 'created_at');

        if(!empty($request->type_id))
          $result->whereTypeId($request->type_id);

        if(!empty($request->start_from))
          $result->offset($request->start_from);

        $limit = !empty($request->limit) ? $request->limit : 100; 
        $data = $result->limit($limit)->get();

        return Helper::apiReturn(__('app.success'), $data);
    }

    /**
    * Retrun a group
    */
    public function get(Request $request, $id): JsonResponse
    {
        $group = Group::whereAppId($request->user('api')->app_id)->whereId($id)->first();

        if(!empty($group)) {
          $data = [
            'id' => $group->id,
            'name' => $group->name,
            'type_id' => $group->type_id,
            'created_at' => Helper::datetimeDisplay($group->created_at)
          ];
          $status = __('app.success');
        } else {
          $status = __('app.error');
          $data = ['message' => __('app.no_record_found')];
        }

        return Helper::apiReturn($status, $data);
    }

    /**
    * Add a group
    */
    public function add(Request $request): JsonResponse
    {
        $input = $this->_groupData($request);
        $input['type_id'] = !empty($request->type_id) ? $request->type_id : null;

        $status = __('app.error');
        if(empty($input['name'])) {
          $data = [
            'message' => 'name ' . __('app.is_required'),
          ];
        } elseif(empty($input['type_id'])) {
          $data = [
            'message' => 'type_id ' . __('app.is_required'),
          ];
        } elseif(!in_array($input['type_id'],[1,2,3,4,5])) {
          // type_id = available groups type_ids
          $data = [
            'message' => 'type_id ' . __('app.not_exist'),
          ];
        } else {
          $group = Group::whereAppId($input['app_id'])->whereTypeId($input['type_id'])->whereName($input['name'])->select('id')->first();
          if(!empty($group->id)) {
            $data = [
              'id' => $group->id,
              'message' => __('app.group') .' '. __('app.already_exist'),
            ];
          } else {
            $group = Group::create($input);
            activity('create')->withProperties(['app_id' => $input['app_id']])->log(__('app.group') . " ({$input['name']}) ". __('app.log_create'));

            $status = __('app.success');
            $data = [
              'id' => $group->id,
              'name' => $input['name']
            ];
          }
        }

        return Helper::apiReturn($status, $data);
    }

    /**
    * Update a group 
    */
    public function update(Request $request, $id): JsonResponse
    {
        $input = $this->_groupData($request);
        $status = __('app.error');
        $data = [];
        if(empty($input['name'])) {
          $data = [
            'message' => 'name ' . __('app.is_required'),
          ];
        } elseif(!Group::whereId($id)->whereAppId($input['app_id'])->exists()) {
          $data = [
            'message' => 'id ' . __('app.not_exist'),
          ];
        } elseif($request->name) {
          Group::whereId($id)->whereAppId($input['app_id'])->update(['name' => $input['name']]);
          activity('update')->withProperties(['app_id' => $input['app_id']])->log(__('app.group') . " ({$input['name']}) ". __('app.log_update'));

          $status = __('app.success');
          $data = [
            'id' => $id,
            'message' => __('app.group') .' '. __('app.log_update'),
          ];
        }

        return Helper::apiReturn($status, $data);
    }

    /**
    * Return inputed data
    */
    public function _groupData($request)
    {
        $input = $request->all();
        $input['name'] = !empty($input['name']) ? trim($input['name']) : null;
        $input['user_id'] = $request->user('api')->id;
        $input['app_id'] = $request->user('api')->app_id;
        return $input;
    }

    /**
    * Delete a group 
    */
    public function destroy(Request $request, $id): JsonResponse
    {
        $name = Group::whereAppId($request->user('api')->app_id)->whereId($id)->value('name');
        if(!empty($name)) {
          $destroy = Group::destroy($id);
          activity('delete')->withProperties(['app_id' => $request->user('api')->app_id])->log(__('app.group') . " ({$name}) ". __('app.log_delete'));

          $status = __('app.success');
          $data = [
            'id' => $id,
            'name' => $name
          ];
        } else {
          $status = __('app.error');
          $data = ['message' => __('app.no_record_found')];
        }
        return Helper::apiReturn($status, $data);
    }
}
