<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Requests\CreateBounceRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use App\Models\Bounce;
use App\Http\Helper\Helper;
use Illuminate\View\View;
use Webklex\PHPIMAP\ClientManager;
use Auth;
use Crypt;

class BounceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        Helper::checkPermissions('bounce_servers');
        $page = "bounce_servers";
        $page_title = __('app.bounce_servers');
        $breadcrumbs = [
            __('app.bounce_servers') => route('bounces.index'),
            __('app.manage') => '#'
        ];
        $groups = \App\Models\Group::groups(config('custom.group_bounce_server'));
        return view('bounces.index',compact('page', 'page_title', 'breadcrumbs', 'groups'));
    }

    /**
    * Retrun JSON datatable data
    */
    public function getBounces(Request $request)
    {
        $result = Bounce::leftJoin('groups', 'groups.id', '=', 'bounces.group_id')
          ->select('bounces.id', 'bounces.email', 'bounces.host', 'bounces.method', 'bounces.is_active', 'bounces.is_validate', 'bounces.created_at', 'groups.id as group_id', 'groups.name as group_name')
          ->app();

        $columns = ['bounces.id', 'bounces.id', 'bounces.email', 'groups.name', 'bounces.host', 'bounces.method', 'bounces.is_active', 'bounces.is_validate', 'bounces.created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $bounces = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($bounces as $bounce) {
          $checkbox = "<input class=\"form-check-input\" type=\"checkbox\" value=\"{$bounce->id}\">";
          $actions = '<div class="btn-group">
            <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').' <span class="caret"></span></button><ul class="dropdown-menu" role="menu">';
          $actions.= '<li><a class="dropdown-item" href="'.route('bounces.edit', [$bounce->id]).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="'.route('bounce.copy', [$bounce->id]).'"><i class="bi bi-copy"></i> '.__('app.copy').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="move(\''.$bounce->id.'\', \''.htmlentities($bounce->email).'\')"><i class="bi bi-arrows"></i> '.__('app.move').'</a></li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$bounce->id.'\', \''.route('bounces.destroy', [$bounce->id]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul>
            </div>';

            $status = $bounce->is_active 
            ? "<span class='badge bg-success-gradient'>".__('app.yes')."</small>"
            : "<span class='badge bg-danger-gradient'>".__('app.no')."</small>";

          $validate = $bounce->is_validate 
            ? "<span class='badge bg-success-gradient'>".__('app.yes')."</small>"
            : "<span class='badge bg-danger-gradient'>".__('app.no')."</small>";

          $group_name = $bounce->group_id && $bounce->group_name ? "<span id='{$bounce->group_id}'>$bounce->group_name</span>" : "<span id='0'>-</span>";

          $data['data'][] = [
            "DT_RowId" => "row_{$bounce->id}",
            $checkbox,
            $bounce->id,
            $bounce->email,
            $group_name,
            $bounce->host,
            $bounce->method,
            $status,
            $validate,
            Helper::datetimeDisplay($bounce->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        Helper::checkPermissions('bounce_servers');
        $page = "bounce_servers";
        $page_title = __('app.add_new_bounce_server');
        $breadcrumbs = [
            __('app.bounce_servers') => route('bounces.index'),
            __('app.add_new_bounce_server') => '#'
        ];
        $groups = \App\Models\Group::groups(config('custom.group_bounce_server'));
        return view('bounces.create',compact('page', 'page_title', 'breadcrumbs', 'groups'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CreateBounceRequest $request): RedirectResponse
    {
        $data = $this->_bounceData($request);
        $bounce = Bounce::create($data);

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.bounce') . " ({$request->email}) ". __('app.log_create'));

        if(isset($data['save_continue']))
            return redirect()->route('bounces.create')
            ->with('success', ($request->email . ' ' . __('app.created_successfully')));
        elseif(isset($data['save_validate']))
            return redirect()->route('bounces.edit', $bounce->id)
            ->with('success', ($request->email . ' ' . __('app.created_successfully')));
        else
            return redirect()->route('bounces.index')
            ->with('success', ($request->email . ' ' . __('app.created_successfully')));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Bounce $bounce): View
    {
        Helper::checkPermissions('bounce_servers');
        $page = "bounce_servers";
        $page_title = __('app.edit_bounce_server');
        $breadcrumbs = [
            __('app.bounce_servers') => route('bounces.index'),
            __('app.edit_bounce_server') => '#'
        ];
        $groups = \App\Models\Group::groups(config('custom.group_bounce_server'));
        return view('bounces.edit',compact('page', 'page_title', 'breadcrumbs', 'bounce', 'groups'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(CreateBounceRequest $request, Bounce $bounce): RedirectResponse
    {
        $data = $this->_bounceData($request);
        $bounce->fill($data)->save();

        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.bounce') . " ({$bounce->email}) ". __('app.log_update'));

        return redirect()->back()
            ->with('success', ($request->email . ' ' . __('app.updated_successfully')));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $names = json_encode(array_values(Bounce::whereIn('id', $ids)->pluck('email')->toArray()));
          $destroy = Bounce::whereIn('id', $ids)->delete();
        } else {
          $names = Bounce::whereId($id)->value('email');
          $destroy = Bounce::destroy($id);
        }

        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.bounce') . " ({$names}) ". __('app.log_delete')); 

        $data = [
                'success' => true,
                'message' => __('app.deleted_successfully')
            ];
        return response()->json($data, 200); 
    }

    /**
    * Retrun data for store or update
    */
    private function _bounceData($request)
    {
        $input = $request->except('_token');
        $input['password'] = !empty($request->password) ? Crypt::encrypt($request->password) : null;
        $input['is_validate'] = false;
        $input['app_id'] = Auth::user()->app_id;
        $input['user_id'] = Auth::user()->id;
        return $input;
    }

    /**
    * Return after validate for IMAP or POP3
    */
    public function validateImap(Request $request)
    {
        $validate_cert = $request->validate_cert == 'Yes' ? true : false;

        // Create a new ClientManager instance
        $clientManager = new ClientManager();

        $data = [
            'host'          => $request->host,
            'port'          => $request->port,
            'encryption'    => $request->encryption,
            'validate_cert' => $validate_cert,
            'username'      => trim($request->username),
            'password'      => trim($request->password),
            'protocol'      => $request->method
        ];

        // Set up the client configuration
        $client = $clientManager->make($data);

        try {
            // Connect to the IMAP server
            $client->connect();

            // update is_validate value
            $bounce = Bounce::findOrFail($request->id);
            $bounce->update([
                'is_validate' => true,
            ]);

            return '<span class="text-success">'.__('app.successfully_connected').'</span>';
        } catch (\Exception $e) {
            // Handle connection errors
            $msg = $e->getMessage();
            return "<span class='text-danger'>{$msg}</span>";
        }
    }

    /**
     * Retrun index after copy bounce server
    */
    public function copy(int $id): RedirectResponse
    {
        Helper::checkPermissions('bounce_servers');
        $bounce = Bounce::whereId($id)->app()->first();
        $bounce->email = $bounce->email .' - copy ';
        $bounce->is_active = false;
        $copied_bounce = $bounce->replicate();
        $copied_bounce->save();

        activity('copy')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.bounce') . " ({$bounce->email}) ". __('app.log_copy'));

        return redirect()->route('bounces.index')
                ->with('success', ($bounce->email . ' ' . __('app.copied_successfully')));
    }
}
