<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Requests\CreateBroadcastRequest;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use App\Models\Broadcast;
use App\Http\Helper\Helper;
use App\Services\RssFeedService;
use Auth;

class BroadcastController extends Controller
{
    /**
     * Display a broadcasting of the resource.
     */
    public function index(Request $request): View
    {
        Helper::checkPermissions('campaigns_manage');
        $page = 'campaigns_manage';
        $page_title = __('app.campaigns');
        $breadcrumbs = [
            __('app.campaigns') => route('broadcasts.index'),
            __('app.manage') => '#'
        ];
        $groups = \App\Models\Group::groups(config('custom.group_campaigns'));
        return view('broadcasts.index',compact('page', 'page_title', 'breadcrumbs', 'groups'));
    }

    /**
    * Retrun JSON datatable data
    */
    public function getBroadcasts(Request $request): Void
    {
        $result = Broadcast::join('groups', 'groups.id', '=', 'broadcasts.group_id')
          ->select('broadcasts.id', 'broadcasts.name as name', 'broadcasts.created_at', 'groups.id as group_id', 'groups.name as group_name')
          ->where('broadcasts.app_id', Auth::user()->app_id);
        
        $columns = ['broadcasts.id', 'broadcasts.id', 'broadcasts.name', 'groups.name', 'broadcasts.created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $broadcasts = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($broadcasts as $broadcast) {
          $checkbox = "<input class=\"form-check-input\" type=\"checkbox\" value=\"{$broadcast->id}\">";
          $actions = '<div class="btn-group">';
          $actions .= '<button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').'</button>';
          $actions .= '<ul class="dropdown-menu" style="">';
          $actions .= '<li><a class="dropdown-item" target="_blank" href="'.route('broadcasts.show', [$broadcast->id]).'"><i class=" bi bi-eye"></i> '.__('app.view').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="'.route('broadcasts.edit', [$broadcast->id]).'"><i class=" bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="'.route('broadcast.copy', ['id' => $broadcast->id]).'"><i class="bi bi-copy"></i> '.__('app.copy').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="'.route('schedule_campaigns.create', 'b='.$broadcast->id).'"><i class="bi bi-lightning"></i> '.__('app.schedule').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="move(\''.$broadcast->id.'\', \''.htmlentities($broadcast->name).'\')"><i class="bi bi-arrows"></i> '.__('app.move').'</a></li>';
          $actions .= '<li><a class="dropdown-item" class="dropdown-item" href="javascript:;" onclick="viewModal(\'modal\', \''.route('send.email', [$broadcast->id, 0, 0]).'\')"><i class="bi bi-send"></i> '.__('app.send_test_email').'</a></li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$broadcast->id.'\', \''.route('broadcasts.destroy', [$broadcast->id]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul></div>';

          $group_name = "<span id='{$broadcast->group_id}'>$broadcast->group_name<span>";
          $data['data'][] = [
            "DT_RowId" => "row_{$broadcast->id}",
            $checkbox,
            $broadcast->id,
            '<a class="link-info" target="_blank" href="'.route('broadcasts.show', [$broadcast->id]).'">'.$broadcast->name.'</a>',
            $group_name,
            Helper::datetimeDisplay($broadcast->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request): View
    {
        Helper::checkPermissions('add_new_campaign');
        $page = 'campaigns_create';
        $page_title = __('app.add_new_campaign');
        $breadcrumbs = [
            __('app.campaigns') => route('broadcasts.index'),
            __('app.add_new_campaign') => '#'
        ];

        $html = '';
        if(!empty($request->tid)) {
            $html = \App\Models\Template::whereId(base64_decode($request->tid))->value('content');
        }

        $groups = \App\Models\Group::groups(config('custom.group_campaigns'));
        $sending_domains = \App\Models\SendingDomain::getSendingDomains();
        return view('broadcasts.create',compact('page', 'page_title', 'breadcrumbs', 'groups', 'sending_domains', 'html'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(CreateBroadcastRequest $request): RedirectResponse
    {
        $data = $this->_broadcastData($request);
        $broadcast = Broadcast::create($data);

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.broadcast') . " ({$request->name}) ". __('app.log_create'));

        if(isset($data['save_continue']))
            return redirect()->route('broadcasts.edit', $broadcast->id)
            ->with('success', ($request->name . ' ' . __('app.updated_successfully')));
        else
            return redirect()->route('broadcasts.index')
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
    }

    /**
   * Retrun edit view
  */
  public function edit(Broadcast $broadcast): View
  {
    Helper::checkPermissions('add_new_campaign');
    $page = 'campaigns_create';
    $page_title = __('app.edit_campaign');
    $breadcrumbs = [
        __('app.campaigns') => route('broadcasts.index'),
        __('app.edit_campaign') => '#'
    ];
    $groups = \App\Models\Group::groups(config('custom.group_campaigns'));
    $sending_domains = \App\Models\SendingDomain::getSendingDomains();
    return view('broadcasts.edit')->with(compact('page', 'page_title', 'breadcrumbs', 'broadcast', 'groups', 'sending_domains'));
  }

  /**
   * Update data
  */
  public function update(CreateBroadcastRequest $request, Broadcast $broadcast): RedirectResponse
  {
    $data = $this->_broadcastData($request);
    $broadcast->fill($data)->save();
    activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.broadcast') . " ({$request->name}) ". __('app.log_update'));

    if(isset($data['save_continue']))
        return redirect()->route('broadcasts.edit', $broadcast->id)
            ->with('success', ($data['name'] . ' ' . __('app.updated_successfully')));
    else
        return redirect()->route('broadcasts.index')
        ->with('success', ($data['name'] . ' ' . __('app.updated_successfully')));
  }

    /**
     * Display the specified resource.
     *
     * @param  \App\list  $list
     * @return \Illuminate\Http\Response
     */
    public function show(Broadcast $broadcast): View
    {   
        $content_html = $broadcast->content_html;
        return view('broadcasts.view')->with('content_html', $content_html);
    }

  /**
     * Remove the specified resource from storage.
     *
     * @param  \App\list  $list
     * @return \Illuminate\Http\Response
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
            $ids = array_values($request->ids);
            $names = json_encode(array_values(Broadcast::whereIn('id', $ids)->pluck('name')->toArray()));
            $destroy = Broadcast::whereIn('id', $ids)->delete();
        } else {
            $names = Broadcast::whereId($id)->value('name');
            $destroy = Broadcast::destroy($id);
        }
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.broadcast') . " ({$names}) ". __('app.log_delete'));

        $data = [
          'success' => true,
          'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200);
    }

    /**
    * Retrun data for store or update
    */
    private function _broadcastData($request): Array
    {
        $input = $request->except('_token');
        // creating issue for shortcodes
        //$input['content_html'] = \Purifier::clean($input['content_html']);
        $input['name'] = strip_tags($input['name']);
        $input['email_subject'] = strip_tags($input['email_subject']);
        $input['user_id'] = Auth::user()->id;
        $input['app_id'] = Auth::user()->app_id;
        return $input;
    }

  /**
   * Retrun index after copy broadcast
  */
  public function copy(int $id): RedirectResponse
  {
    $broadcast = Broadcast::whereId($id)->app()->first();
    $broadcast->name = $broadcast->name .' - copy ';
    $broadcast = $broadcast->replicate();
    $broadcast->save();
    return redirect()->route('broadcasts.index')
        ->with('success', ($broadcast->name . ' ' . __('app.msg_successfully_copied')));
  }

  /**
   * Custom Mail Headers for campaign
  */
  public function mailHeaders(Request $request)
  {
    Helper::checkPermissions('mail_headers'); // check user permission
    $page = 'campaigns_mail_headers';
    $page_title = __('app.mail_headers');
    $breadcrumbs = [
        __('app.mail_headers') => '#'
    ];
    if($request->post()) {
      $headers = [];
      foreach($request->mail_headers as $header) {
        if(!empty($header['key'])) {
          $headers[$header['key']] = $header['value'];
        }
      }

      // Save headers
      \App\Models\User::where('id', Auth::user()->id)->update(['mail_headers' => json_encode($headers)]);
      return redirect()->route('broadcast.mail.headers')
        ->with('success', (__('app.mail_headers') . ' ' . __('app.saved_successfully')));

    } else {

      $mail_headers = json_decode(Auth::user()->mail_headers, true);
      return view('broadcasts.mail_headers')->with(compact('page', 'page_title', 'breadcrumbs', 'mail_headers'));
    }
    
  } 
  /**
   * Retrun HTML data of a template
  */
  public function getHTMLContent(int $id)
  {
    return Broadcast::whereId($id)->app()->value('content_html');
  }

  public function checkSpamScore(Request $request)
  {
     $spam_score = Helper::checkSpamScore($request->content);
     return view('broadcasts.spam_score')->with(compact('spam_score'));
  }
}
