<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Illuminate\Validation\Rule;
use Illuminate\Support\Arr;
use App\Models\User;
use App\Models\Package;
use Spatie\Permission\Models\Role;
use App\Http\Helper\Helper;
use Hash;
use DB;
use Auth;
 

class ClientController extends Controller
{
    /**
    * Retrun index view
    */
    public function index(): View
    {
        Helper::checkPermissions('clients'); // check user permission
        $page = 'clients_';
        $page_title = __('app.clients');
        $breadcrumbs = [
            __('app.client_management') => '#',
            __('app.clients') => route('clients.index'),
        ];
        return view('clients.index')->with(compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Retrun JSON datatable data
    */
    public function getClients(Request $request): Void
    {
        $result = User::select('users.id', 'users.name', 'users.email', 'users.is_active', 'users.created_at', 'packages.name as package_name', 'packages.id as package_id')
          ->where('users.is_client', 1)
          ->leftJoin('packages', 'users.package_id', '=', 'packages.id')
          ->where('users.id', '<>', 1) // avoid to get superadmin as came with installation
          ->app();

        $columns = ['id', 'id', 'name', 'email', 'id', 'id', 'is_active', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $users = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($users as $user) {
          $checkbox = "<input type=\"checkbox\" value=\"{$user->id}\" class=\"form-check-input\">";


          $actions = '<div class="btn-group">';
          $actions .= '<button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').'</button>';
          $actions .= '<ul class="dropdown-menu" style="">';
          $actions .= '<li><a class="dropdown-item" href="'.route('clients.edit', $user->id).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
          $actions .= '<li class="dropdown-item">
                <form action="'.route('user.impersonate', $user->id).'" method="POST" style="display: inline;">
                    '.csrf_field().'
                    <button type="submit" class="dropdown-item" style="background: none; border: none; padding: 0; color: inherit; cursor: pointer;">
                        <i class="bi bi-people"></i> '.__('app.impersonate').'
                    </button>
                </form>
            </li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$user->id.'\', \''.route('users.destroy', $user->id).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul>';

          $status = $user->is_active 
            ? '<span class="badge bg-success-gradient">'.__('app.active').'</span>'
            : '<span class="badge bg-danger-gradient">'.__('app.in_active').'</span>';

          if(!empty($user->roles->pluck('display_name')[0])) {
            $role_id = $user->roles->pluck('id')[0];
            $user_role = "<a href='".route('roles.edit', $role_id)."' class='link-info link-offset-2 link-underline-opacity-25 link-underline-opacity-100-hover text-decoration-none'>".$user->roles->pluck('display_name')[0]."</a>";
          } else {
            $user_role = '---';
          }

          $package_name = "<a href='".route('packages.edit', [$user->package_id])."' class='link-info'>$user->package_name</a>";

          $data['data'][] = [
            "DT_RowId" => "row_{$user->id}",
            $checkbox,
            $user->id,
            $user->name,
            $user->email,
            $user_role,
            $package_name,
            $status,
            Helper::datetimeDisplay($user->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(): View
    {
        Helper::checkPermissions('clients');
        $page = 'clients_';
        $page_title = __('app.add_new_client');
        $breadcrumbs = [
            __('app.client_management') => '#',
            __('app.clients') => route('clients.index'),
            __('app.add_new_client') => '#',
        ];
        $roles = Role::select('name','display_name')->where('type', 'client')->get();
        $packages = Package::whereAppId(Auth::user()->app_id)->get();
        return view('clients.create',compact('roles', 'packages', 'page', 'page_title', 'breadcrumbs'));
    }

    public function store(Request $request): RedirectResponse
    {
        $this->validate($request, [
            'name' => 'required',
            'email' => [
                'required',
                'email',
                Rule::unique('users', 'email')->where('is_client', 1) // Unique only for `is_client = 1`
            ],
            'password' => 'required|same:confirm-password',
            'roles' => [
                'required',
                'not_in:'
            ],
            'is_active' => 'required|boolean',
        ]);
            
        $input = $request->all();
        $input['password'] = Hash::make($input['password']);
        $input['app_id'] = User::max('app_id')+1;
        $input['is_client'] = 1;
        $input['parent_id'] = Auth::user()->id;
        $input['contact_customize_listing'] = json_encode(['custom_field_ids' => ["0"]]);
    
        $user = User::create($input);
        $user->assignRole($request->input('roles'));

        // save email/pages data for new client
        $this->_emailPagesData($input['app_id']);

        // save log
        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.client') . " ({$request->name}) ". __('app.log_create'));
    
        return redirect()->route('clients.index')
            ->with('success',__('app.user') .' '. __('app.log_create'));
    }


    /**
    * Retrun edit view
    */
    public function edit(User $user, $id): View
    {
        Helper::checkPermissions('clients');
        $page = 'clients_';
        $page_title = __('app.edit_client');
        $breadcrumbs = [
            __('app.client_management') => '#',
            __('app.clients') => route('clients.index'),
            __('app.edit_client') => '#',
        ];
        $user = User::whereId($id)->app()->first(); // get user detail
        $roles = Role::select('name','display_name')->where('type', 'client')->get();
        $packages = Package::whereAppId(Auth::user()->app_id)->get();
        $userRole = $user->roles->pluck('name','name')->all();
 
        return view('clients.edit',compact('user','roles','userRole', 'page', 'page_title', 'breadcrumbs', 'packages'));
    }

    public function update(Request $request, $id): RedirectResponse
    { 
        $user = User::whereId($id)->app()->first(); // get user detail
        $this->validate($request, [
            'name' => 'required',
            'email' => [
                'required',
                'email',
                Rule::unique('users', 'email')
                    ->where('is_client', 1)
                    ->ignore($user->id) // Ignore the current user's ID
            ],
            'password' => 'nullable|same:confirm-password', // Password is optional for updates
            'roles' => [
                'required',
                'not_in:'
            ],
            'is_active' => 'required|boolean',
        ]);
    
        $input = $request->all();
        if(!empty($input['password'])){ 
            $input['password'] = Hash::make($input['password']);
        } else {
            $input = Arr::except($input,array('password'));    
        }

        $user->update($input);
        DB::table('model_has_roles')->where('model_id',$user->id)->delete();
    
        $user->assignRole($request->input('roles'));

        // save log
        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.client') . " ({$request->name}) ". __('app.log_update'));

        return redirect()->back()
            ->with('success', __('app.user') . ' ' . __('app.updated_successfully'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $names = json_encode(array_values(User::whereIn('id', $ids)->pluck('name')->toArray()));
          $destroy = User::whereIn('id', $ids)->delete();
        } else {
          $names = User::whereId($id)->value('name');
          $destroy = User::destroy($id);
        }

        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.client') . " ({$names}) ". __('app.log_delete'));

        $data = [
                'success' => true,
                'message' => __('app.deleted_successfully')
            ];
        return response()->json($data, 200); 
    }

    private function _emailPagesData ($user_id)
    {
        $data = [
            [
            'name' => 'Notification Email (Contact Unsubscribed)',
            'slug' => 'notify-email-contact-unsub',
            'type' => 'email',
            'app_id' => $user_id,
            'user_id' => Auth::user()->app_id,
            'order' => 4,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'email_subject' => 'Contact has been unsubscribed',
            'content' => '<!DOCTYPE html>
<html>
<head><meta charset="UTF-8">
  <title>Subscriber Unsubscribed</title>
  <style type="text/css">body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            background-color: #f4f4f9;
            color: #333;
            margin: 0;
            padding: 0;
        }
        .container {
            max-width: 600px;
            margin: 20px auto;
            background: #fff;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        .header {
            background-color: #1F1F34;
            color: #fff;
            text-align: center;
            padding: 15px 20px;
        }
        .content {
            padding: 20px;
        }
        .footer {
            text-align: center;
            padding: 10px;
            background-color: #1F1F34;
            font-size: 12px;
            color: #777;
        }
        .footer a {
            color: #f56342;
            text-decoration: none;
        }
  </style>
</head>
<body>
<div class="container">
<div class="header">
<h1>Subscriber Unsubscribed</h1>
</div>

<div class="content">
<p>Hello,</p>

<p>This is to inform you that a subscriber has successfully unsubscribed from the mailing list([$list-name$]).</p>

<p><strong>Email Address:</strong>&nbsp;[$receiver-email$]</p>
</div>

<div class="footer">
<p><span style="color:#ffffff;">&copy; 2025 Relayzo. All rights reserved</span></p>
</div>
</div>
</body>
</html>
'
            ],
            [
            'name' => 'Notification Email (Contact Added)',
            'slug' => 'notify-email-contact-added',
            'type' => 'email',
            'app_id' => $user_id,
            'user_id' => Auth::user()->app_id,
            'order' => 3,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'email_subject' => 'Contact has been added to a list',
            'content' => '<!DOCTYPE html>
<html>
<head><meta charset="UTF-8">
  <title>Subscriber Unsubscribed</title>
  <style type="text/css">body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            background-color: #f4f4f9;
            color: #333;
            margin: 0;
            padding: 0;
        }
        .container {
            max-width: 600px;
            margin: 20px auto;
            background: #fff;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        .header {
            background-color: #1F1F34;
            color: #fff;
            text-align: center;
            padding: 15px 20px;
        }
        .content {
            padding: 20px;
        }
        .footer {
            text-align: center;
            padding: 10px;
            background-color: #1F1F34;
            font-size: 12px;
            color: #777;
        }
        .footer a {
            color: #f56342;
            text-decoration: none;
        }
  </style>
</head>
<body>
<div class="container">
<div class="header">
<h1>Subscriber Added</h1>
</div>

<div class="content">
<p>Hello,</p>

<p>This is to inform you that a subscriber has successfully added to the mailing list([$list-name$]).</p>

<p><strong>Email Address:</strong>&nbsp;[$receiver-email$]</p>
</div>

<div class="footer">
<p><span style="color:#ffffff;">&copy; 2025 Relayzo. All rights reserved</span></p>
</div>
</div>
</body>
</html>
'
            ],
            [
            'name' => 'Unsubscribe Email',
            'slug' => 'unsub-email',
            'type' => 'email',
            'app_id' => $user_id,
            'user_id' => Auth::user()->app_id,
            'order' => 2,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'email_subject' => 'Unsubscribed',
            'content' => '<!DOCTYPE html>
<html>
<head><meta charset="UTF-8">
  <title>Subscriber Unsubscribed</title>
  <style type="text/css">body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            background-color: #f4f4f9;
            color: #333;
            margin: 0;
            padding: 0;
        }
        .container {
            max-width: 600px;
            margin: 20px auto;
            background: #fff;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        .header {
            background-color: #1F1F34;
            color: #fff;
            text-align: center;
            padding: 15px 20px;
        }
        .content {
            padding: 20px;
        }
        .footer {
            text-align: center;
            padding: 10px;
            background-color: #1F1F34;
            font-size: 12px;
            color: #777;
        }
        .footer a {
            color: #f56342;
            text-decoration: none;
        }
  </style>
</head>
<body>
<div class="container">
<div class="header">
<h1>Unsubscribe Confirmation</h1>
</div>

<div class="content">
<p>Hello [$receiver-email$],</p>

<p>This email is to confirm that you have successfully unsubscribed from our mailing list. You will no longer receive updates or promotional emails from us.</p>

<p>We&rsquo;re sorry to see you go, but we respect your decision. If you have any feedback, feel free to share it with us at [$sender-email$].</p>

<p>Best regards,<br />
Team Relayzo</p>
</div>

<div class="footer">
<p><span style="color:#ffffff;">&copy; 2025 Relayzo. All rights reserved</span></p>
</div>
</div>
</body>
</html>
'
            ],
            [
            'name' => 'Welcome Email',
            'slug' => 'welcome-email',
            'type' => 'email',
            'app_id' => $user_id,
            'user_id' => Auth::user()->app_id,
            'order' => 1,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'email_subject' => 'Welcome!',
            'content' => '<!DOCTYPE html>
<html>
<head><meta charset="UTF-8">
  <title>Subscriber Unsubscribed</title>
  <style type="text/css">body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            background-color: #f4f4f9;
            color: #333;
            margin: 0;
            padding: 0;
        }
        .container {
            max-width: 600px;
            margin: 20px auto;
            background: #fff;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        .header {
            background-color: #1F1F34;
            color: #fff;
            text-align: center;
            padding: 15px 20px;
        }
        .content {
            padding: 20px;
        }
        .footer {
            text-align: center;
            padding: 10px;
            background-color: #1F1F34;
            font-size: 12px;
            color: #777;
        }
        .footer a {
            color: #f56342;
            text-decoration: none;
        }
  </style>
</head>
<body>
<div class="container">
<div class="header">
<h1>Welcome to Relayzo!</h1>
</div>

<div class="content">
<p>Hi [$receiver-email$],</p>

<p>We&#39;re thrilled to welcome you to our community! Get ready to receive the latest updates, exclusive offers, and valuable content delivered straight to your inbox.</p>

<p>Thank you for subscribing!</p>

<p>Best regards,<br />
Team Relayzo</p>
</div>

<div class="footer">
<p><span style="color:#ffffff;">&copy; 2025 Relayzo. All rights reserved<br />
<a href="[$unsub-link$]">unsubscribe</a></span></p>
</div>
</div>
</body>
</html>
'
            ],
            [
            'name' => 'Unsubscribe Page (Confirmation)',
            'slug' => 'unsub-confirm',
            'type' => 'page',
            'app_id' => $user_id,
            'user_id' => Auth::user()->app_id,
            'order' => 1,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'email_subject' => null,
            'content_html' => '<!DOCTYPE html>
<html>
<head><meta charset="UTF-8">
  <title>Subscriber Unsubscribed</title>
  <style type="text/css">body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            background-color: #f4f4f9;
            color: #333;
            margin: 0;
            padding: 0;
        }
        .container {
            max-width: 600px;
            margin: 20px auto;
            background: #fff;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        .header {
            background-color: #1F1F34;
            color: #fff;
            text-align: center;
            padding: 15px 20px;
        }
        .content {
            padding: 20px;
        }
        .footer {
            text-align: center;
            padding: 10px;
            background-color: #1F1F34;
            font-size: 12px;
            color: #777;
        }
        .footer a {
            color: #f56342;
            text-decoration: none;
        }
  </style>
</head>
<body>
<div class="container">
<div class="header">
<h1><strong>Confirm Your Unsubscription</strong></h1>
</div>

<div class="content">
<p>We&rsquo;ve received a request to unsubscribe you from our mailing list. To complete the process and stop receiving emails from us, please click the link below to confirm your unsubscription:</p>

<p><a href="[$unsub-confirm$]" rel="noopener"><strong>Confirm Unsubscription</strong></a></p>

<p>If you did not request to unsubscribe, or if this was a mistake, please ignore this email, and you will remain on our list.</p>

<p>We&rsquo;re sorry to see you go, but we respect your decision. If you have any feedback, feel free to share it with us at [$sender-email$].</p>
</div>

<div class="footer">
<p><span style="color:#ffffff;">&copy; 2025 Relayzo. All rights reserved</span></p>
</div>
</div>
</body>
</html>
'
            ],
            [
            'name' => 'Unsubscribe Page (Final)',
            'slug' => 'unsub',
            'type' => 'page',
            'app_id' => $user_id,
            'user_id' => Auth::user()->app_id,
            'order' => 2,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'email_subject' => null,
            'content_html' => '<!DOCTYPE html>
<html>
<head><meta charset="UTF-8">
  <title>Subscriber Unsubscribed</title>
  <style type="text/css">body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            background-color: #f4f4f9;
            color: #333;
            margin: 0;
            padding: 0;
        }
        .container {
            max-width: 600px;
            margin: 20px auto;
            background: #fff;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        .header {
            background-color: #1F1F34;
            color: #fff;
            text-align: center;
            padding: 15px 20px;
        }
        .content {
            padding: 20px;
        }
        .footer {
            text-align: center;
            padding: 10px;
            background-color: #1F1F34;
            font-size: 12px;
            color: #777;
        }
        .footer a {
            color: #f56342;
            text-decoration: none;
        }
  </style>
</head>
<body>
<div class="container">
<div class="header">
<h1>Unsubscribe Confirmation</h1>
</div>

<div class="content">
<p>Hello [$receiver-email$],</p>

<p>Thank you for being part of our community. Your request to unsubscribe has been processed, and you will no longer receive emails from us.</p>

<p>We&rsquo;re sorry to see you go, but we respect your decision. If there&rsquo;s anything we could have done better, we&rsquo;d love to hear your feedback at [$sender-email$].</p>
</div>

<div class="footer">
<p><span style="color:#ffffff;">&copy; 2025 Relayzo. All rights reserved</span></p>
</div>
</div>
</body>
</html>
'
            ]
        ];
        DB::table('emails_pages')->insert($data);
    }
}
