<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Illuminate\Validation\Rule;
use App\Notifications\ContactAdd;
use App\Http\Helper\Helper;
use App\Models\Contact;
use Auth;


class ContactController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        Helper::checkPermissions('contacts_manage');
        $page = 'contacts_manage';
        
        $breadcrumbs = [
            __('app.contacts') => route('contacts.index'),
            __('app.manage') => '#'
        ];
        $list_id = !empty($request->input('list_id')) ? $request->input('list_id') : null;
        $list_name = !empty($list_id) ? \App\Models\Lists::whereId($list_id)->value('name') : null;

        $custom_field_names = [];
        $contact_customize_listing = json_decode(Auth::user()->contact_customize_listing, true);
        if(!empty($contact_customize_listing['custom_field_ids'])) {
            $custom_field_names = \App\Models\CustomField::whereIn('id', $contact_customize_listing['custom_field_ids'])->pluck('name')->toArray();
        }

        $page_title = __('app.contacts') . ($list_name ? " ($list_name)" : '');
        return view('contacts.index',compact('page', 'page_title', 'breadcrumbs', 'list_id', 'list_name', 'custom_field_names'));
    }

 
    public function getContacts(Request $request): void
    {
        $contact_customize_listing = json_decode(Auth::user()->contact_customize_listing, true);
        $custom_field_ids = $contact_customize_listing['custom_field_ids'] ?? [];

        // Base query with leftJoin and conditional custom_field_id filtering
        $result = Contact::leftJoin('contact_custom_fields', function ($join) use ($custom_field_ids) {
            $join->on('contacts.id', '=', 'contact_custom_fields.contact_id');

            if (!empty($custom_field_ids)) {
                $join->whereIn('contact_custom_fields.custom_field_id', $custom_field_ids);
            }
        })
        ->select('contacts.*', 'contact_custom_fields.data as custom_field_data')
        ->app(); // <- If you use a custom scope

        $dispaly_custom_field = !empty($custom_field_ids) &&
            \App\Models\CustomField::whereIn('id', $custom_field_ids)->exists();

        if ($request->list_id) {
            $result->where('contacts.list_id', $request->list_id);
        }

        $columns = ['contacts.id', 'contacts.id', 'email', 'contacts.id', 'is_unsubscribed', 'is_active', 'is_confirmed', 'created_at', 'contact_custom_fields.data', 'contacts.id', 'contacts.id'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $contacts = $result->get();

        $data = Helper::datatableTotals($data['total']);

        foreach ($contacts as $contact) {
            $checkbox = "<input class=\"form-check-input\" type=\"checkbox\" value=\"{$contact->id}\">";

            $actions = '<div class="btn-group">
                <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').' <span class="caret"></span></button><ul class="dropdown-menu" role="menu">';
            $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="viewModal(\'modal\', \''.route('contacts.show', [$contact->id]).'\')"><i class="bi bi-eye"></i> '.__('app.view').'</a></li>';
            $actions .= '<li><a class="dropdown-item" href="'.route('contacts.edit', [$contact->id]).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
            $actions .= '<li><a class="dropdown-item"  href="javascript:;" onclick="viewModal(\'modal\', \''.route('verify.email', ['id' => 0, 'email' => $contact->email]).'\')"><i class="bi bi-person-check"></i> '.__('app.verify_email').'</a></li>';
            $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$contact->id.'\', \''.route('contacts.destroy', [$contact->id]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
            $actions .= '</ul></div>';

            $list_name = !empty($contact->list->name) ? '<a class="link-info" href="javascript:;" onclick="viewModal(\'modal\', \''.route('lists.show', [$contact->list->id]).'\')">'.$contact->list->name.'</a>' : '---';

            $status = $contact->is_active 
                ? "<span class='badge bg-success-gradient'>".__('app.yes')."</span>"
                : "<span class='badge bg-danger-gradient'>".__('app.no')."</span>";

            $confirmed = $contact->is_confirmed
                ? "<span class='badge bg-success-gradient'>".__('app.yes')."</span>"
                : "<span class='badge bg-danger-gradient'>".__('app.no')."</span>";

            $verified = $contact->is_verified
                ? "<span class='badge bg-success-gradient'>".__('app.yes')."</span>"
                : "<span class='badge bg-danger-gradient'>".__('app.no')."</span>";

            $unsubscribed = $contact->is_unsubscribed
                ? "<span class='badge bg-danger-gradient'>".__('app.yes')."</span>"
                : "<span class='badge bg-success-gradient'>".__('app.no')."</span>";

            $email = $contact->email;
            if ($contact->is_bounced) {
                $email = "<span class='text-danger' style='cursor:pointer' title='".__('app.bounced')."'>$email</span>";
            }

            if ($dispaly_custom_field) {
                $data['data'][] = [
                    "DT_RowId" => "row_{$contact->id}",
                    $checkbox,
                    '<a class="link-info" href="javascript:;" onclick="viewModal(\'modal\', \''.route('contacts.show', [$contact->id]).'\')">'.$contact->id.'</a>',
                    $email,
                    $list_name,
                    $contact->custom_field_data ?? '---',
                    $status,
                    $unsubscribed,
                    $confirmed,
                    $verified,         
                    Helper::datetimeDisplay($contact->created_at),
                    $actions
                ];
            } else {
                $data['data'][] = [
                    "DT_RowId" => "row_{$contact->id}",
                    $checkbox,
                    '<a class="link-info" href="javascript:;" onclick="viewModal(\'modal\', \''.route('contacts.show', [$contact->id]).'\')">'.$contact->id.'</a>',
                    $email,
                    $list_name,
                    $status,
                    $unsubscribed,
                    $confirmed,
                    $verified,         
                    Helper::datetimeDisplay($contact->created_at),
                    $actions
                ];
            }
        }

        echo json_encode($data);
    }


    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        // client add contact limit assign to a package
        if(!Helper::allowedLimit(Auth::user()->app_id, 'no_of_contacts', 'contacts')) {
            return view('errors.not_allowed')->with('module', __('app.contacts'));
        }
        Helper::checkPermissions('add_new_contact');
        $page = 'contacts_create';
        $page_title = __('app.add_new_contact');
        $breadcrumbs = [
            __('app.contacts') => route('contacts.index'),
            __('app.add_new_contact') => '#'
        ];
        $list_id = !empty($request->input('list_id')) ? $request->input('list_id') : null; // will use when add contact from list dropdown
        return view('contacts.create',compact('page', 'page_title', 'breadcrumbs', 'list_id'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
       $request->validate([
            'list_ids' => 'required|array|min:1', // Ensure list_ids is not empty and contains at least one value
            'list_ids.*' => 'exists:lists,id', // Ensures each list_id exists in the lists table
            'email' => [
                'required',
                'email',
                Rule::unique('contacts')->where(function ($query) use ($request) {
                    return $query->where('list_id', $request->input('list_ids.0')); // Adjusted to get the first value if it's a single list
                }),
            ],
        ], [
            'list_ids.required' => 'The list is required.',
            'list_ids.min' => 'You must select at least one list.',
            'list_ids.*.exists' => 'Please select the list.',
            'email.unique' => 'The email has already been added to this list.',
        ]);


       if($request->skip_already_exist) {
            $request->validate([
                'email' => 'unique:contacts',            
            ], [
                'email.unique' => 'The email has already been added to another list.',
            ]);
       }


        $data = $this->_contactData($request);
        $contact = Contact::create($data);

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.contact') . " ({$contact->email}) ". __('app.log_create')); // log

        $this->contactCustomData($contact->id, $data);

        // send notfication
        if($contact->list->welcome_email == 'Yes' || !empty($contact->list->notify_email))
            $contact->notify(new ContactAdd('welcome-email', 'notify-email-contact-added'));

        if(isset($data['save_continue']))
            return redirect()->route('contacts.create', 'list_id='.$data['list_id'])
            ->with('success', ($request->email . ' ' . __('app.created_successfully')));
        else
            return redirect()->route('contacts.index')
            ->with('success', ($request->email . ' ' . __('app.created_successfully')));
            
    }

    /**
     * Display the specified resource.
     */
    public function show(Contact $contact): View
    {
        return view('contacts.view')->with(compact('contact'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Contact $contact): View
    {
        Helper::checkPermissions('add_new_contact');
        $page = 'contacts_create';
        $page_title = __('app.edit_contact');
        $breadcrumbs = [
            __('app.contacts') => route('contacts.index'),
            __('app.edit_contact') => '#'
        ];
        return view('contacts.edit',compact('page', 'page_title', 'breadcrumbs', 'contact'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Contact $contact): RedirectResponse
    {
        $request->validate([
            'email' => [
                'required',
                'email',
                Rule::unique('contacts')
                    ->where(function ($query) use ($request) {
                        return $query->where('list_id', $request->input('list_id'));
                    })
                    ->ignore($contact->id), // Ignore the current contact's ID
            ],
        ], [
            'email.unique' => 'The email has already been added to this list.',
        ]);

        $data = $this->_contactData($request);

        // Check if contact is being resubscribed (is_unsubscribed changing from true to false)
        $was_unsubscribed = $contact->is_unsubscribed;
        
        $contact->fill($data)->save();
        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.contact') . " ({$contact->email}) ". __('app.log_update'));

        // If contact was unsubscribed and is now resubscribed, remove from suppression list
        if($was_unsubscribed && !$contact->is_unsubscribed) {
            $this->removeFromSuppressionList($contact->email, $contact->app_id);
        }

        \DB::table('contact_custom_fields')->whereContactId($contact->id)->delete();
        $this->contactCustomData($contact->id, $data);

        return redirect()->route('contacts.edit', $contact->id)
            ->with('success', ($request->email . ' ' . __('app.updated_successfully')));
           
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $emails = json_encode(array_values(Contact::whereIn('id', $ids)->pluck('email')->toArray()));
          $destroy = Contact::whereIn('id', $ids)->delete();
        } else {
          $emails = Contact::whereId($id)->value('email');
          $destroy = Contact::destroy($id);
        }
        
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.contact') . " ({$emails}) ". __('app.log_delete'));

        $data = [
            'success' => true,
            'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200); 
    }

    /**
    * Retrun data for store or update
    */
    private function _contactData($request)
    {
        
        $input = $request->except('_token');
        $input['user_id'] = Auth::user()->id;
        $input['app_id'] = Auth::user()->app_id;

        if(isset($input['list_ids'][0]))
            $input['list_id'] = $input['list_ids'][0]; // for single select the list_id will be at 0 location
        return $input;
    }

    /**
    * Retrun contact custom data
    */
    public function contactCustomData($contact_id, $input)
    {
        if(!empty($input['custom_fields'])) {
            foreach($input['custom_fields'] as $custom_field_id => $data) {
            $data = is_array($data) ? implode('||', $data) : $data ;
                if(!empty($data)) {
                  \DB::table('contact_custom_fields')->insert([
                    'contact_id' => $contact_id,
                    'custom_field_id' => $custom_field_id,
                    'data' => $data
                  ]);
                }
            }
        }
    }

    /**
    * Remove email from suppression list when contact is resubscribed
    */
    private function removeFromSuppressionList($email, $app_id)
    {
        $suppression_type_id = config('custom.group_suppression');
        
        // Find the "Unsubscribed" suppression group for this app_id
        $group = \App\Models\Group::where('name', 'Unsubscribed')
            ->where('type_id', $suppression_type_id)
            ->where('app_id', $app_id)
            ->first();

        if($group) {
            // Remove email from suppression list
            \App\Models\Suppression::where('group_id', $group->id)
                ->where('email', $email)
                ->delete();
        }
    }

    /**
    * Retrun custom fields associate with the list
    */
    public function getListsCustomFields(Request $request)
    {
        $list_id = $request->list_id;
        $contact_data = [];

        if(!empty($request->contact_id)) {
          $contact_custom_data = \DB::table('contact_custom_fields')->whereContactId($request->contact_id)->get();
          foreach($contact_custom_data as $data) {
            $contact_data[$data->custom_field_id] = $data->data;
          }
        }

        $list = \App\Models\Lists::findOrFail($list_id);
        $list_custom_fields = $list->customFields()->get();
        if(!empty($request->view)) {
          $custom_field_ids = !empty($request->custom_field_ids) ? $request->custom_field_ids : null;
          return view('contacts.'.$request->view)->with(compact('list_custom_fields', 'contact_data', 'custom_field_ids'));
        }
        return view('contacts.list_custom_fields')->with(compact('list_custom_fields', 'contact_data'));
    }

    /**
   * Export contacts for a list
  */
  public function contactsExport(Request $request, $list_id, $data=null)
  {

    if(!empty($data) && $data == 'show_modal') {
      $list = \App\Models\Lists::findOrFail($list_id);
      return view('contacts.export')->with(compact('list'));
    }
    
       
    $fields = [
        'is_active' => $request->is_active,
        'is_subscribed' => $request->is_subscribed,
        'is_verified' => $request->is_verified,
        'is_confirmed' => $request->is_confirmed,

    ];

    \App\Jobs\ListExport::dispatch($list_id, Auth::user()->id, $fields)
      ->delay(now()->addSeconds(10));

    // save activity log
    $list = \App\Models\Lists::findOrFail($list_id);
    activity('export')->withProperties(['app_id' => $list->app_id])->log(__('app.list') . " ({$list->name}) ". __('app.log_export'));

    return redirect()->route('lists.index')
        ->with('success', ($list->name . ': ' . __('app.list_export_successfully')));

  }


  /**
   * Import contact into a list
  */
  public function contactsImport(Request $request, $list_id=null)
  {
    if($request->isMethod('post')) {
      if(!empty($request->fieldMapping)) {
        $path_import_list = str_replace('[user-id]', Auth::user()->id, config('custom.path_import_list'));
        $file = $path_import_list.$request->list_id.'-header.csv';
        $file_header = Helper::getCsvFileHeader($file);
        $list = \App\Models\Lists::findOrFail($request->list_id);
        $list_custom_fields = $list->customFields()->pluck('name', 'custom_field_id');

        $drop_down = '<select name="custom_fields[]" class="form-control">';
        $drop_down .= "<option value='none'></option>";
        $drop_down .= "<option value='email'>Email</option>";
        foreach($list_custom_fields as $custom_filed_id => $custom_field_name) {
          $drop_down .= "<option value='{$custom_filed_id}'>{$custom_field_name}</option>";
        }
        $drop_down .= '</select>';

        $string = '';
        foreach($file_header as $header) {
          $string .= "<div class='row mb-3'>
              <label class='col-md-1 col-form-label fw-medium'>{$header}</label>
              <div class='col-md-11'>{$drop_down}</div>
            </div>";
        }
        return $string;
      } elseif($request->do_import) {
        $path_import_list = str_replace('[user-id]', Auth::user()->id, config('custom.path_import_list'));
        $file_name = md5(uniqid()).'.csv';
        $file = $request->file('file');
        $attributes = $request->except('_token', 'file', 'do_import');

        $contact_import = \App\Models\Import::create([
          'type' => 'contact',
          'file' => $path_import_list.$file_name,
          'attributes' => json_encode($attributes),
          'total' => Helper::getCsvCount($file),
          'user_id' => Auth::user()->id,
          'app_id' => Auth::user()->app_id
        ]);

        try {
          $file->move($path_import_list, $file_name); // save original file to import
          
          \App\Jobs\ListImport::dispatch($contact_import->id)
            ->delay(now()->addSeconds(10));

          // save activity log
          $list = \App\Models\Lists::findOrFail($request->list_id);
          activity('import')->withProperties(['app_id' => $list->app_id])->log(__('app.list') . " ({$list->name}) ". __('app.log_import'));

          unlink($path_import_list.$request->list_id.'-header.csv'); // delete header file
        } catch(\Exception $e) {
            //echo $e->getMessage();
            // nothing
        }
        return $contact_import->id;
      } else {
        $request->validate([
            'list_id' => 'required|integer',
            'file' => [
                'required',
                'file',
                function ($attribute, $value, $fail) {
                    $extension = strtolower($value->getClientOriginalExtension());
                    if (!in_array($extension, ['csv', 'txt', 'xls', 'xlsx'])) {
                        $fail('The ' . $attribute . ' must be a file of type: csv, txt, xls, xlsx.');
                    }
                },
                'max:' . Helper::getMaxFileSizeMB(),
            ]
        ]);
        $file = $request->file('file');
        $header = Helper::getCsvFileHeader($file);
        $path_import_list = str_replace('[user-id]', Auth::user()->id, config('custom.path_import_list'));
        Helper::dirCreateOrExist($path_import_list); // create dir if not exist
        $file = $path_import_list.$request->list_id.'-header.csv';
        $writer = \League\Csv\Writer::createFromPath($file, 'w'); // create a .csv file to write data
        $writer->insertOne($header); // write file header
      }
    } else {
      $page = 'contacts_import';
      $page_title = __('app.contacts_import');
      $breadcrumbs = [
            __('app.contacts') => route('contacts.index'),
            __('app.contacts_import') => '#'
        ];
      return view('contacts.import')->with(compact('page', 'page_title', 'breadcrumbs', 'list_id'));
    }
  }

  /**
   * Return status for the import progress
  */
  public function contactsImportStatus($id)
  {
    if(isset($id)) {
        return \App\Models\Import::select('total', 'processed', 'duplicates', 'invalids', 'suppressed', 'bounced')->whereId($id)->first()->toJson();
    }
  }

  /**
   * Return to 1st step to contact to unsubscribed page
  */
  public function unsub($cid, $slid=null, $type='c')
  {
    return redirect()->route('page.show', ['unsub-confirm', $cid, $slid, $type]);
  }

  /**
   * Return to confirm page after click unsubscribed
  */
  public function doUnsub($id, $slid=null, $type='c')
  {
    $id = base64_decode($id);

    $contact = Contact::findOrFail($id);
    $contact->fill(['is_unsubscribed' => true])->save(); // update contact data

    if($type == 'c') {

        if(!empty($slid)) {
            \App\Models\ScheduleCampaignStatLogClick::where('schedule_campaign_stat_log_id', base64_decode($slid))->update([
              'unsub_confirmed' => true
            ]);
        }
    } elseif($type == 't') {
        if(!empty($slid)) {
            \App\Models\TriggerScheduleStatLogClick::where('trigger_schedule_stat_log_id', base64_decode($slid))->update([
              'unsub_confirmed' => true
            ]);
        }
    } elseif($type == 'd') {
        if(!empty($slid)) {
            \App\Models\DripScheduleStatLogClick::where('drip_schedule_stat_log_id', base64_decode($slid))->update([
              'unsub_confirmed' => true
            ]);
        }
    } elseif($type == 'st') {
        if(!empty($slid)) {
            \App\Models\SplitTestStatLogClick::where('split_test_stat_log_id', base64_decode($slid))->update([
              'unsub_confirmed' => true
            ]);
        }
    }

    // If unsubscribed once no need to send other email if click twice
    if($contact->is_unsubscribed) {
      // send email to subscriber and notifiy to list admin
      $contact->notify(new ContactAdd('unsub-email', 'notify-email-contact-unsub'));
    }

    activity('unsub')->withProperties(['app_id' => $contact->app_id])->log(__('app.contact') . " ({$contact->email}) ". __('app.log_unsub'));
    
    return redirect()->route('page.show', ['slug' => 'unsub', 'id' => base64_encode($id)]);
  }


  public function contactsCustomizeListing (Request $request)
  {
    if($request->isMethod('get')) {
        $custom_fields = \App\Models\CustomField::where('app_id', Auth::user()->app_id)->orderBy('id', 'DESC')->get();
        $contact_customize_listing = Auth::user()->contact_customize_listing;
        return view('contacts.customize_listing')->with(compact('custom_fields', 'contact_customize_listing'));
    } else {
        $user = \App\Models\User::find(Auth::id());

        // Decode the JSON field to an array (safe even if it's null or already array)
        $settings = json_decode($user->contact_customize_listing, true) ?? [];

        // Now you can safely assign
        $settings['custom_field_ids'] = $request->custom_field_ids ?? null;

        // Save the updated JSON
        if(!empty($settings['custom_field_ids']))
            $user->contact_customize_listing = json_encode($settings);
        else
            $user->contact_customize_listing = null;

        $user->save();

        return redirect()->route('contacts.index')
            ->with('success', (__('app.customize_listing_set_successfully')));
    }
  }

}
