<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use App\Rules\UniqueCustomField;
use App\Models\CustomField;
use App\Http\Helper\Helper;
use Auth;

class CustomFieldController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        Helper::checkPermissions('custom_fields');
        $page = "lists_custom_fields";
        $page_title = __('app.custom_fields');
        $breadcrumbs = [
            __('app.custom_fields') => route('custom_fields.index'),
            __('app.manage') => '#'
        ];  
        return view('custom_fields.index',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
       * Retrun JSON datatable data
    */
    public function getCustomFields(Request $request): Void
    {
        $result = CustomField::select('id', 'name', 'tag', 'type', 'is_required', 'created_at')
            ->app();

        $columns = ['id', 'id', 'name', 'tag', 'type', 'is_required', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $custom_fields = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($custom_fields as $custom_field) {
          $checkbox = "<input type=\"checkbox\" value=\"{$custom_field->id}\" class=\"form-check-input\">";

          $actions = '<div class="btn-custom_field">';
          $actions .= '<button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').'</button>';
          $actions .= '<ul class="dropdown-menu" style="">';
          $actions.= '<li><a class="dropdown-item" href="javascript:;" onclick="viewModal(\'modal\', \''.route('custom_fields.edit', $custom_field->id).'\')"><i class="bi bi-pencil"></i> '.__('app.edit').'</a></li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$custom_field->id.'\', \''.route('custom_fields.destroy', $custom_field->id).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul>';

          $required = $custom_field->is_required
            ? "<span class='badge bg-success-gradient'>".__('app.yes')."</span>"
            : "<span class='badge bg-danger-gradient'>".__('app.no')."</span>";

          $data['data'][] = [
            "DT_RowId" => "row_{$custom_field->id}",
            $checkbox,
            $custom_field->id,
            $custom_field->name,
            '[#'.$custom_field->tag.'#]',
            ucfirst($custom_field->type),
            $required,
            Helper::datetimeDisplay($custom_field->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        Helper::checkPermissions('custom_fields');
        $custom_field_type = $this->customFieldTypes();
        return view('custom_fields.create')->with(compact('custom_field_type'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => ['required', 'string', new UniqueCustomField],
        ]);
        $data = $this->_customFiedlData($request);
        $custom_field = CustomField::create($data);
        if(!empty($data['list_ids'])) {
          $list_ids = array_values($data['list_ids']);
          $custom_field->lists()->attach($list_ids);
        }
        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.custom_field') . " ({$request->name}) ". __('app.log_create'));

        return redirect()->back()
            ->with('success', __('app.custom_field') . ' ' . __('app.log_create'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(CustomField $custom_field): View
    {
        Helper::checkPermissions('custom_fields');
        $custom_field_lists = $custom_field->getListId()->toArray();
        $custom_field_type = $this->customFieldTypes();
        return view('custom_fields.edit')->with(compact('custom_field', 'custom_field_type', 'custom_field_lists'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, CustomField $custom_field): RedirectResponse
    {
        $request->validate([
          'name' => 'required|string'
        ]);
        $data = $this->_customFiedlData($request);
        $custom_field->fill($data)->save();
        if(!empty($data['list_ids'])) {
          $list_ids = array_values($data['list_ids']);
          $custom_field->lists()->sync($list_ids);
        }

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.custom_field') . " ({$request->name}) ". __('app.log_update'));

        return redirect()->back()
            ->with('success', __('app.custom_field') . ' ' . __('app.log_update'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $names = json_encode(array_values(CustomField::whereIn('id', $ids)->pluck('name')->toArray()));
          $destroy = CustomField::whereIn('id', $ids)->delete();
        } else {
          $names = CustomField::whereId($id)->value('name');
          $destroy = CustomField::destroy($id);
        }
        
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.custom_field') . " ({$names}) ". __('app.log_delete'));

        $data = [
            'success' => true,
            'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200); 
    }

    /**
    * Return custom fields types
    */
    public function customFieldTypes(): Array
    {
        return [
          'text' => 'Text Field',
          'number' => 'Number Field',
          'textarea' => 'Textarea',
          'date'  => 'Date Field',
          'radio' => 'Radio Buttons',
          'checkbox' => 'Checkboxes',
          'dropdown' => 'Dropdown'
        ];
    }

    /**
    * Retrun data for store or update
    */
    private function _customFiedlData($request): Array
    {
        $input = $request->except('_token');
        $input['tag'] = \Illuminate\Support\Str::slug($input['name'], '-');
        $list_ids = !empty($input['list_ids']) ? array_values($input['list_ids']) : null;
        $input['user_id'] = Auth::user()->id;
        $input['app_id'] = Auth::user()->app_id;
        return $input;
    }
}
