<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\DripRequest;
use App\Models\Drip;
use App\Http\Helper\Helper;
use Auth;

class DripController extends Controller
{
    public function index(): View
    {
        Helper::checkPermissions('drips');
        $page = 'campaigns_drips';
        $page_title = __('app.drips');
        $breadcrumbs = [
            __('app.campaigns') => '#',
            __('app.drips') => route('drips.index'),
            __('app.manage') => '#'
        ];
        $groups = \App\Models\Group::groups(config('custom.group_drips'));
        return view('drips.index',compact('page', 'page_title', 'breadcrumbs', 'groups'));
    }

     /**
    * Retrun JSON datatable data
    */
    public function getDrips(Request $request)
    {
        $result = Drip::join('groups', 'drips.group_id', '=', 'groups.id')
          ->select('drips.id', 'drips.name as name', 'drips.email_subject', 'drips.send', 'drips.after_minutes', 'drips.created_at', 'groups.id as group_id', 'groups.name as group_name')
          ->where('drips.app_id', Auth::user()->app_id);

        $columns = ['drips.id', 'groups.name', 'drips.name', 'drips.email_subject', 'drips.after_minutes', 'drips.created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $drips = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($drips as $drip) {
          $checkbox = "<input class=\"form-check-input\" type=\"checkbox\" value=\"{$drip->id}\">";
          $actions = '<div class="btn-group">
            <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').' <span class="caret"></span></button><ul class="dropdown-menu" role="menu">';
          $actions .= '<li><a class="dropdown-item" href="'.route('drips.edit', [$drip->id]).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="move(\''.$drip->id.'\', \''.htmlentities($drip->name).'\')"><i class="bi bi-arrows"></i> '.__('app.move').'</a></li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$drip->id.'\', \''.route('drips.destroy', [$drip->id]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';

          $actions .= '</ul></div>';

          $group_name = "<span id='{$drip->group_id}'>$drip->group_name<span>";
          $send = $drip->send;
          if($send == 'After') {
            if($drip->after_minutes >= 1440) {
              // 1440 = 1 Day;
              $time = round($drip->after_minutes / 1440, 2);
              $duration = __('app.days');
            } elseif($drip->after_minutes >= 60) {
              // 60 = 1 Hour;
              $time = round($drip->after_minutes / 60, 2);
              $duration = __('app.hours');
            } else {
              $time = $drip->after_minutes;
              $duration = __('app.minutes');
            }
            $send = "{$drip->send}  $time $duration";
          }

          $email_subject = '<a class="link-info" target="_blank" href="'.route('drip.campaign.show', [$drip->id]).'">'.$drip->email_subject.'</a>';
          
          $data['data'][] = [
            "DT_RowId" => "row_{$drip->id}",
            $checkbox,
            $drip->id,
            $group_name,
            $drip->name,                        
            $email_subject,
            $send,
            Helper::datetimeDisplay($drip->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    public function create(): View
    {
        Helper::checkPermissions('drips');
        $page = 'campaigns_drips';
        $page_title = __('app.add_new_drip');
        $breadcrumbs = [
          __('app.campaigns') => '#',
          __('app.drips') => route('drips.index'),
          __('app.add_new_drip') => '#'
        ];
        
        $groups = \App\Models\Group::groups(config('custom.group_drips'));
        return view('drips.create',compact('page', 'page_title', 'breadcrumbs', 'groups'));
    }

    public function store(DripRequest $request): RedirectResponse
    {
        $data = $this->_dripData($request);
        $drip = Drip::create($data);

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.drip') . " ({$request->name}) ". __('app.log_create'));

        if($request->submit == 'save_continue')
          return redirect()->route('drips.create')
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
        else
          return redirect()->route('drips.edit', $drip->id)
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
    }


    public function edit(Drip $drip): View
    {
        Helper::checkPermissions('drips');
        $page = 'campaigns_drips';
        $page_title = __('app.edit_drip');
        $breadcrumbs = [
          __('app.campaigns') => '#',
          __('app.drips') => route('drips.index'),
          __('app.edit_drip') => '#'
        ];
        
        $groups = \App\Models\Group::groups(config('custom.group_drips'));
        return view('drips.edit',compact('page', 'page_title', 'breadcrumbs', 'groups', 'drip'));
    }


    public function update(DripRequest $request, $id): RedirectResponse
    {
      $drip = Drip::findOrFail($id);
      $data = $this->_dripData($request);
      $drip->fill($data)->save();

      activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.drip') . " ({$request->name}) ". __('app.log_update'));

      return redirect()->route('drips.edit', $drip->id)
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
    }

    /**
    * Retrun data for store or update
    */
    public function _dripData($request)
    {
        $input = $request->except('_token');
        $input['send_attributes'] = json_encode(['send'=>$input['send'], 'time'=>$input['time'], 'duration'=>$input['duration']]);
        if($input['send'] == 'After') {
          if($input['duration'] == 'hours') {
            // Convet hours into minutes
            $input['after_minutes'] = $input['time'] * 60;
          } elseif($input['duration'] == 'days') {
            // Convet days into minutes; 1 Day = 60*24 = 1440
            $input['after_minutes'] = $input['time'] * 1440; 
          } else {
            $input['after_minutes'] = $input['time'];
          }
        } else {
          // For Instant
          $input['after_minutes'] = 0;
        }
        $input['user_id'] = Auth::user()->id;
        $input['app_id'] = Auth::user()->app_id;
        return $input;
    }

    public function show($id): View
    {   
        $content_html = Drip::whereId($id)->value('content');
        return view('broadcasts.view')->with('content_html', $content_html);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $names = json_encode(array_values(Drip::whereIn('id', $ids)->pluck('name')->toArray()));
          $destroy = Drip::whereIn('id', $ids)->delete();
        } else {
          $names = Drip::whereId($id)->value('name');
          $destroy = Drip::destroy($id);
        }
        
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.drip') . " ({$names}) ". __('app.log_delete'));

        $data = [
            'success' => true,
            'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200); 
    }
}
