<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\JsonResponse;
use Spatie\Activitylog\Models\Activity;
use App\Http\Helper\Helper;
use Auth;

class MiscellaneousController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function activityLogs(Request $request): View
    {
        Helper::checkPermissions('activity_logs');
        $page = 'miscellaneous_activity_logs';
        $page_title = __('app.activity_logs');
        $breadcrumbs = [
            __('app.miscellaneous') => '#',
            __('app.activity_logs') => route('activity.logs')
        ];  
        return view('miscellaneous.activity_logs',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Retrun JSON datatable data
    */
    public function getActivityLogs(Request $request): Void
    {
        $result = Activity::where('properties->app_id', Auth::user()->app_id)
          ->where('id', '<>', 1) // avoid to get superadmin as came with installation
          ->with('causer'); 

        $columns = ['id', 'id', 'log_name', 'description', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $activity_logs = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($activity_logs as $log) {
            $name = ucfirst($log->log_name);
            if($log->log_name == 'delete')
                $name = "<span class='badge bg-danger-gradient'>".$name."</span>";
            elseif($log->log_name == 'copy')
                $name = "<span class='badge bg-warning-gradient'>".$name."</span>";
            elseif($log->log_name == 'update')
                $name = "<span class='badge bg-info-gradient'>".$name."</span>";
            elseif($log->log_name == 'create')
                $name = "<span class='badge bg-success-gradient'>".$name."</span>";
            elseif($log->log_name == 'schedule')
                $name = "<span class='badge bg-warning-gradient'>".$name."</span>";
            else
                $name = "<span class='badge bg-purple-gradient'>".$name."</span>";

          $checkbox = "<input type=\"checkbox\" value=\"{$log->id}\" class=\"form-check-input\">";

          $description = !empty($log->description) ? nl2br(wordwrap($log->description, 150, "\n", true)) : '---';

          $data['data'][] = [
            "DT_RowId" => "row_{$log->id}",
            $log->id,
            $log->causer['name'] ?? '---',
            $name,
            $description,
            Helper::datetimeDisplay($log->created_at)
          ];
        }
        echo json_encode($data);
    }

    /**
    * Get Error Logs
    */
    public function errorLogs(): view
    { 
        Helper::checkPermissions('error_logs');
        $page = 'miscellaneous_error_logs';
        $page_title = __('app.error_logs');
        $breadcrumbs = [
            __('app.miscellaneous') => '#',
            __('app.error_logs') => route('error.logs')
        ]; 

        return view('miscellaneous.error_logs')->with(compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Get PHP Info
    */
    public function phpinfo(): View
    { 
        Helper::checkPermissions('php_info');
        $page = 'miscellaneous_php_info';
        $page_title = __('app.php_info');
        $breadcrumbs = [
            __('app.miscellaneous') => '#',
            __('app.php_info') => route('php.info')
        ]; 
        return view('miscellaneous.phpinfo')->with(compact('page', 'page_title', 'breadcrumbs'));
    }

    public function notifications(Request $request): View
    {
        Helper::checkPermissions('system_notifications');
        $page = 'miscellaneous_notifications';
        $page_title = __('app.notifications');
        $breadcrumbs = [
            __('app.miscellaneous') => '#',
            __('app.notifications') => route('notifications')
        ];  
        return view('miscellaneous.notifications',compact('page', 'page_title', 'breadcrumbs'));
    }

    public function getNotifications(Request $request): Void
    {
        $result = \App\Models\Notification::whereIsRead(false)
            ->app(\Auth::user()->app_id);

        $columns = ['name', 'type', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $notifications = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($notifications as $notification) {

          $notification->type = $notification->type == 'export' ? __('app.download') : __('app.read');

          $type = "<a class='link-info' href='".route('notification.read', [$notification->id])."'>{$notification->type}</a>";

          $delete = '<a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$notification->id.'\', \''.route('notifications.delete', $notification->id).'\')"><i class="bi bi-trash"></i></a>';

          $data['data'][] = [
            $notification->name,
            $type,
            Helper::datetimeDisplay($notification->created_at),
            $delete
          ];
        }
        echo json_encode($data);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroyNotificatoins(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $names = json_encode(array_values(\App\Models\Notification::whereIn('id', $ids)->pluck('name')->toArray()));
          $destroy = \App\Models\Notification::whereIn('id', $ids)->delete();
        } else {
          $names = \App\Models\Notification::whereId($id)->value('name');
          $destroy = \App\Models\Notification::destroy($id);
        }
        
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.notification') . " ({$names}) ". __('app.log_delete'));

        $data = [
            'success' => true,
            'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200); 
    }
}
