<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use App\Models\Note;
use App\Http\Helper\Helper;
use Auth;

class NoteController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        Helper::checkPermissions('notes');
        $page = "organizer_notes";
        $page_title = __('app.notes');
        $breadcrumbs = [
            __('app.organizer') => '#',
            __('app.notes') => route('notes.index')
        ];

        $notes = Note::select('id', 'title', 'content', 'is_pinned', 'created_at')
            ->app()->orderBy('id', 'DESC')->get();

        return view('notes.index',compact('page', 'page_title', 'breadcrumbs', 'notes'));
    }

    public function create(): View
    {
        Helper::checkPermissions('notes');
        return view('notes.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
          'title' => 'required'
        ]);

        $data = $this->_noteData($request);
        Note::create($data);

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.note') . " ({$request->name}) ". __('app.log_create'));

        return redirect()->back()
            ->with('success', __('app.note') . ' ' . __('app.log_create'));
    }

    public function edit(Note $note): View
    {
        Helper::checkPermissions('notes');
         return view('notes.edit')->with(compact('note'));
    }

    public function update(Request $request, Note $note): RedirectResponse
    {
        $request->validate([
          'title' => 'required'
        ]);

        $data = $this->_noteData($request);
        $note->fill($data)->save();

        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.note') . " ({$request->title}) ". __('app.log_update'));

        return redirect()->back()
            ->with('success', __('app.note') . ' ' . __('app.log_update'));
    }

    /**
    * Retrun data for store or update
    */
    private function _noteData($request) 
    {
        $input = $request->except('_token');
        $input['user_id'] = Auth::user()->id;
        $input['app_id'] = Auth::user()->app_id;
        return $input;
    }

    public function destroy(int $id, Request $request): JsonResponse
    {
        $title = Note::whereId($id)->value('title');
        $destroy = Note::destroy($id);
        
        
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.note') . " ({$title}) ". __('app.log_delete'));

        $data = [
            'success' => true,
            'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200); 
    }


    public function pin(int $id, Request $request): JsonResponse
    {
        Note::whereId($id)->update(['is_pinned' => $request->is_pinned]);
        
        $msg = $request->is_pinned ? __('app.pinned_successfully') : __('app.unpinned_successfully');
        
        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.note') . " ({$request->title}) ". $msg);

        
        $data = [
            'success' => true,
            'message' => $msg
        ];

        return response()->json($data, 200); 
    }
}
