<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use App\Models\Pmta;
use App\Http\Helper\Helper;
use Auth;

class PmtaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        Helper::checkPermissions('pmta');
        $page = "mta_pmta";
        $page_title = __('app.pmta');
        $breadcrumbs = [
            __('app.pmta') => route('pmta.index'),
            __('app.manage') => '#'
        ];
        return view('pmtas.index',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
       * Retrun JSON datatable data
    */
    public function getPmtas(Request $request): Void
    {
        $result = Pmta::select('id', 'name', 'type', 'created_at', 'attributes')
            ->app();

        $columns = ['id', 'name', 'type', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $pmtas = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($pmtas as $pmta) {

            $attributes = json_decode($pmta->attributes);
            $pmta_web_monitor_url = ($attributes->server_ip ?? '').':'.($attributes->management_port ?? '').'/ui' ;

            $actions = '<div class="btn-custom_field">
            <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').' <span class="caret"></span></button><ul class="dropdown-menu" role="menu">';
            if($pmta->type == 'PowerMTA')
                $actions.= '<li><a class="dropdown-item" href="'.route('pmta.create.edit', ['id' => $pmta->id]).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
            else
                $actions.= '<li><a class="dropdown-item" href="javascript:;" onclick="viewModal(\'modal\', \''.route('pmta.edit.bounce', ['id' => $pmta->id]).'\')"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';

            $actions.= '<li><a class="dropdown-item" href="//'.$pmta_web_monitor_url.'" target="_blank"><i class="bi bi-bar-chart-line"></i> '.__('app.pmta_server_web_monitor').'</a></li>';
            $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$pmta->id.'\', \''.route('pmta.destroy', $pmta->id).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
            $actions .= '</ul></div>';
            $data['data'][] = [
            "DT_RowId" => "row_{$pmta->id}",
            $pmta->name,
            $pmta->type,
            Helper::datetimeDisplay($pmta->created_at),
            $actions
            ];
        }
        echo json_encode($data);
    }

    public function createEdit($id=null)
    {
        Helper::checkPermissions('pmta');
        $page = "mta_pmta";
        $page_title = __('app.pmta');
        $breadcrumbs = [
            __('app.pmta') => route('pmta.index'),
            __('app.manage') => '#'
        ];

        if(!empty($id)) {
          $pmta = json_decode(Pmta::whereId($id)->value('attributes'));
        } else {
          $pmta = null;
        }
        return view('pmtas.create')->with(compact('page', 'page_title', 'breadcrumbs', 'id', 'pmta'));
    }

    public function pmtaSteps(Request $request, $step=null, $pmta_id=null)
    { 
        if($request->step == 0) {
          return $this->validateServer($request);
        } elseif($request->step == 3) {
          return $this->IpsDomainsMapping($request);
        } elseif($request->step == 4) {
          return $this->bounces($request);
        } elseif($request->step == 5) {
          return $this->authentication($request);
        } elseif($request->step == 6) {
          return $this->config($request);
        } elseif($request->step == 7) {
          return $this->finish($request);
        } elseif($request->step == 10) {
          return $this->validateBounce($request);
        } elseif($request->step == 15) {
          // Was calling from pmta action while edit
          return $this->delete(1);
        } elseif($request->step == 16) {
          return $this->downloadPmtaSettings($pmta_id);
        }
    }

    /**
    * Create .zip according to PMTA settings
    */
    private function downloadPmtaSettings($id)
    {
        $path_pmta = str_replace('[user-id]', Auth::user()->id, config('custom.path_pmta'));
        Helper::dirCreateOrExist($path_pmta); // create dir if not exist

        $pmta = json_decode(Pmta::whereId($id)->value('attributes'), true);

        $main_domain = Helper::getAppURL(true);
        $files = [];

        // Create config file
        $config = $this->config($pmta, true);
        $config_file = $path_pmta.'config';
        \File::put($config_file, $config);

        $zip_file = $path_pmta.$pmta['server_name'].'.zip';

        $zipper = new \ZipArchive;
        //array_push($files, $config_file);
        Helper::createZipFile($zip_file, $config_file);

        // Main Domain DKIM 
        $private_key_file = $path_pmta."fallback.{$main_domain}.pem";
        \File::put($private_key_file, $pmta['domain_keys']["main_domain_private_key"]);
        //array_push($files, $private_key_file);
        Helper::createZipFile($zip_file, $private_key_file);

        $public_key_file = $path_pmta."{$main_domain}_public_key.txt";
        \File::put($public_key_file, $pmta['domain_keys']["main_domain_public_key"]);
        //array_push($files, $public_key_file);
        Helper::createZipFile($zip_file, $public_key_file);

        // Create DKIMs
        $domains = Helper::splitLineBreakWithComma($pmta['domains']);
        foreach ($domains as $domain) {
          // Main Domain DKIM 
          $sending_domain_part = explode('.', $domain);
          $private_key = $pmta['domain_keys']["{$sending_domain_part[0]}_private_key"];
          $pmta['dkim_selector'] = $pmta['dkim_selector'] ?? config('custom.dkim_selector');
          $private_key_file = $path_pmta.$pmta['dkim_selector'].".{$domain}.pem";
          \File::put($private_key_file, $private_key);
          //array_push($files, $private_key_file);
          Helper::createZipFile($zip_file, $private_key_file);

          $public_key_file = $path_pmta."{$domain}_public_key.txt";
          \File::put($public_key_file, $pmta['domain_keys']["{$sending_domain_part[0]}_public_key"]);

          //array_push($files, $public_key_file);
          Helper::createZipFile($zip_file, $public_key_file);
        }

        // Summary PDF File
        $dns_file = $path_pmta.'DNS.pdf';

        // Get Summary
        $html = $this->authentication($pmta, true);
        \PDF::loadHTML($html)->setPaper('a4')->setWarnings(false)->save($dns_file);

        //array_push($files, $dns_file);
        Helper::createZipFile($zip_file, $dns_file);

        foreach($files as $file) {
          try {
            unlink($file);
          } catch(\Exception $e) {
            //$msg = $e->getMessage();
          }
        }
        return [
          'msg' =>'success',
          'file' => route('download.pmta', "{$pmta['server_name']}.zip")
        ];
    }

    public function download($file)
    {
    $path_pmta = str_replace('[user-id]', Auth::user()->id, config('custom.path_pmta'));
    $zip_file = $path_pmta.$file;
    return response()->download($zip_file)->deleteFileAfterSend(true);
    }

    /**
    * Validate bounces
    */
    private function validateBounce($request)
    {

        $validate_cert = $request->selects[2] == 'Yes' ? true : false;
        // Create a new ClientManager instance
        $clientManager = new \Webklex\PHPIMAP\ClientManager();

        $data = [
          'host'          => $request->inputs[0],
          'port'          => $request->inputs[1],
          'encryption'    => $request->selects[1],
          'validate_cert' => $validate_cert,
          'username'      => $request->inputs[2],
          'password'      => $request->inputs[3],
          'protocol'      => $request->method     // imap or pop3
        ];

        // Set up the client configuration
        $client = $clientManager->make($data);

        try {
            // Connect to the IMAP server
            $client->connect();

            // update is_validate value
            $bounce = Bounce::findOrFail($request->id);
            $bounce->update([
                'is_validate' => true,
            ]);

            return '<span class="text-success">'.__('app.successfully_connected').'</span>';
        } catch (\Exception $e) {
            // Handle connection errors
            $msg = $e->getMessage();
            return "<span class='text-danger'>{$msg}</span>";
        }
    }

    /**
    * Domain & IPs mapping
    */
    private function IpsDomainsMapping($request)
    {
    $ips     = Helper::splitLineBreakWithComma($request->ips);
    $domains = Helper::splitLineBreakWithComma($request->domains);
    $ip_per_domain = $this->ipPerDomain($ips, $domains);

    $str = "<script>$(function () {
        $('[data-bs-toggle=\"popover\"]').popover({
          container: 'body'
        });
      });

      function displayBounce(chk_id, bounce_id){
        var checkBox = document.getElementById(chk_id);
        var bounce = document.getElementById(bounce_id);
        if(checkBox.checked == true ){
          bounce.style.display = 'none';
        } else {
          bounce.style.display = 'block';
        }
      }
    </script>";

    $str .= '<div class="row mb-4">
              <div class="col-12">
                <div class="d-flex align-items-center justify-content-between">
                  <div class="d-flex align-items-center">
                    <div class="p-2 border border-warning border-opacity-10 bg-warning-transparent rounded-circle me-3">
                      <span class="avatar avatar-xs avatar-rounded bg-warning svg-white">
                        <i class="bi bi-diagram-3"></i>
                      </span>
                    </div>
                    <h6 class="mb-0 text-warning fw-semibold fs-14">'.__('app.pmta_ips_domains_mapping').'</h6>
                  </div>
                  <a tabindex="-1" role="button"
                    data-bs-toggle="popover" data-bs-trigger="focus" data-bs-placement="left" title="'.__('app.pmta_ips_domains_mapping').'"
                    data-bs-content="'.__('app.pmta_ips_domains_mapping_help').'" data-bs-custom-class="header-primary">
                    <i class="bi bi-question-circle text-primary fs-5"></i>
                  </a>
                </div>
              </div>
            </div>
            
            <div class="alert alert-info mb-4">
              <i class="bi bi-info-circle me-2"></i>'.__('app.pmta_ips_domains_mapping_text').'
            </div>
            
            <div class="row mb-3">
              <div class="col-md-4">
                <div class="form-group">
                  <label class="form-label fw-semibold">'. __('app.vmta_shared_pool') .' 
                    <a tabindex="-1" role="button"
            data-bs-toggle="popover" data-bs-trigger="focus" data-bs-placement="right" title="'.__('app.vmta_shared_pool').'"
                      data-bs-content="'.__('app.vmta_shared_pool_help').'" data-bs-custom-class="header-primary">
                      <i class="bi bi-question-circle text-primary"></i>
                    </a>
                  </label>
              <select name="vmta_shared_pool" id="vmta_shared_pool" class="form-select">
                <option value="1">'.__('app.yes').'</option>
                <option value="0">'.__('app.no').'</option>                
                  </select>
                </div>
              </div>
            </div>';

    $str .="<div class='row '>";
    foreach($domains as $domain) {
      $domain = trim($domain);
      if(empty($domain)) continue;

      $str .= "<div class='col-md-4 mt-3'>";
      $str .= "<div class='card border border-primary border-opacity-10'>";
      $str .= "<div class='card-header bg-primary-transparent'>";
      $str .= "<div class='d-flex align-items-center'>";
      $str .= "<select name=\"protocol[{$domain}]\" class=\"form-select form-select-sm me-3\" style='width: auto;'>
              <option value=\"https://\">https://</option>
              <option value=\"http://\">http://</option>
            </select>";
      $str .= "<div class='p-2 border border-primary border-opacity-10 bg-primary-transparent rounded-circle me-3'>";
      $str .= "<span class='avatar avatar-xs avatar-rounded bg-primary svg-white'>";
      $str .= "<i class='bi bi-globe'></i>";
      $str .= "</span></div>";
      $str .= "<h6 class='mb-0 text-primary fw-semibold fs-14'>{$domain}</h6>";
      $str .= "</div></div>";
      $str .= "<div class='card-body'>";
      $str .= "<div class='mb-3'>";
      $str .= "<label class='form-label fw-semibold'>".__('app.assigned_ips')."</label>";
      $str .= "<div class='d-flex flex-wrap gap-2'>";
      for ($i=0; $i<$ip_per_domain; $i++) {
        if(!empty($ips)) {
          $str .= "<span class='badge bg-primary'>".array_shift($ips)."</span>";
        }
      }
      $str .= "</div></div>";

      $str .= "<div class='form-group mb-3'>";
      $str .= "<label class='form-label fw-semibold'>".__('app.pmta_from_name')." 
        <a tabindex='-1' role='button'
          data-bs-toggle='popover' data-bs-trigger='focus' data-bs-placement='right' title='".__('app.pmta_from_name')."'
          data-bs-content='".__('app.pmta_from_name_help')."' data-bs-custom-class='header-primary'>
          <i class='bi bi-question-circle text-primary'></i>
        </a>
      </label>";
      $str .= "<input type='text' name='from_name[{$domain}]' placeholder='".__('app.pmta_from_name')."' class='form-control' value='' required>";
      $str .= "</div>";

      $str .= "<div class='form-group mb-3'>";
      $str .= "<label class='form-label fw-semibold'>".__('app.pmta_from_email')." 
        <a tabindex='-1' role='button'
          data-bs-toggle='popover' data-bs-trigger='focus' data-bs-placement='right' title='".__('app.pmta_from_email')."'
          data-bs-content='".__('app.pmta_from_email_help')."' data-bs-custom-class='header-primary'>
          <i class='bi bi-question-circle text-primary'></i>
        </a>
      </label>";
      $str .= "<div class='input-group'>";
      $str .= "<input type='text' name='from_email[{$domain}]' placeholder='".__('app.pmta_from_email')."' class='form-control' value='' required>";
      $str .= "<div class='input-group-text'>@{$domain}</div>";
      $str .= "</div></div>";

      $str .= "<div class='form-group mb-3'>";
      $str .= "<label class='form-label fw-semibold'>".__('app.pmta_reply_email')." 
        <a tabindex='-1' role='button'
          data-bs-toggle='popover' data-bs-trigger='focus' data-bs-placement='right' title='".__('app.pmta_reply_email')."'
          data-bs-content='".__('app.pmta_reply_email_help')."' data-bs-custom-class='header-primary'>
          <i class='bi bi-question-circle text-primary'></i>
        </a>
      </label>";
      $str .= "<input type='email' name='reply_email[{$domain}]' placeholder='".__('app.pmta_reply_email')."' class='form-control' value='' required>";
      $str .= "</div>";

      $str .= "<div class='form-group mb-3'>";
      $str .= "<div class='form-check'>";
      $str .= "<input class='form-check-input' type='checkbox' id='chk_bounce_pmta_{$domain}' name='bounce_pmta[{$domain}]' value='Y' checked='checked' onclick=\"displayBounce('chk_bounce_pmta_{$domain}', 'bounce_pmta_{$domain}')\">";
      $str .= "<label class='form-check-label fw-semibold' for='chk_bounce_pmta_{$domain}'>".__('app.bounce_pmta_file')." 
        <a tabindex='-1' role='button'
          data-bs-toggle='popover' data-bs-trigger='focus' data-bs-placement='right' title='".__('app.bounce_pmta_file')."'
          data-bs-content='".__('app.bounce_pmta_file_help')."' data-bs-custom-class='header-primary'>
          <i class='bi bi-question-circle text-primary'></i>
        </a>
      </label>";
      $str .= "</div></div>";

      $str .= "<div class='form-group mb-3' id='bounce_pmta_{$domain}' style='display:none;'>";
      $str .= "<label class='form-label fw-semibold'>".__('app.pmta_bounce_email')." 
        <a tabindex='-1' role='button'
          data-bs-toggle='popover' data-bs-trigger='focus' data-bs-placement='right' title='".__('app.pmta_bounce_email')."'
          data-bs-content='".__('app.pmta_bounce_email_help')."' data-bs-custom-class='header-primary'>
          <i class='bi bi-question-circle text-primary'></i>
        </a>
      </label>";
      $str .= "<div class='input-group'>";
      $str .= "<input type='text' name='bounce_email[{$domain}]' placeholder='".__('app.pmta_bounce_email')."' class='form-control' value=''>";
      $str .= "<div class='input-group-text'>@{$domain}</div>";
      $str .= "</div></div>";

      $str .= "</div></div></div>";

      // No need to display empty domain with no ip assigned
      if(empty($ips)) break;
    }
    $str .= '</div>';
    return $str;
    }

    /**
    * Create bounces dynamically according to the sending domains
    */
    private function bounces($request)
    {
    $str = "<script>$(function () {
        $('[data-toggle=\"tooltip\"]').tooltip();
      });
    </script>";
    foreach($request->bounce_email as $key => $bounce) {
      $bounce_email = $bounce.'@'.$key;
      $div_id = 'bounce-'.rand();
      $str .= '<div class="accordion accordion-primary" id="accordionBounces">';
      $str .= "<div class='accordion-item'><h4 class='accordion-header' id='headingOne'><button class='accordion-button' type='button' data-bs-toggle='collapse' data-bs-target=\"#{$div_id}\" aria-expanded='true'                           aria-controls='collapseOne'>$bounce_email</button></div>";

      $str .= "<div id=\"{$div_id}\" class='accordion-collapse collapse show' aria-labelledby='headingOne' data-bs-parent='#accordionBounces'>";
      $str .= '<div class="accordion-body">';
      if(!empty($request->bounce_pmta[$key]) && $request->bounce_pmta[$key] == 'Y') {
        $str .= '<label class="col-md-12"><h6>'.__('app.bounce_pmta_file').'</h6></label>';
      } else {
        $str .= '<div class="row">
                 <div class="col-md-6 mt-2">
                  <label class="form-label">'.__('app.pmta_bounce_method').'</label>
                  <div class="col-md-12">
                      <select name="'.$div_id.'[]" class="form-select form-select-lg">
                        <option value="imap">IMAP</option>
                        <option value="pop3">POP3</option>
                      </select>
                  </div>
          </div>
          <div class="col-md-6 mt-2">
            <label class="form-label">'.__('app.pmta_bounce_host').'</label>
            <div class="col-md-12">
                <input type="text" class="form-control" name="'.$div_id.'[]" value="" placeholder="'.__('app.pmta_bounce_host').'" required>
            </div>
          </div>
          <div class="col-md-6 mt-2">
            <label class="form-label">'.__('app.pmta_bounce_port').'</label>
            <div class="col-md-12">
              <div class="input-group from-group">
                <input type="text" class="form-control" name="'.$div_id.'[]" value="143" placeholder="'.__('app.pmta_bounce_port').'" required>
              </div>
            </div>
          </div>
          <div class="col-md-6 mt-2">
            <label class="form-label">'.__('app.pmta_bounce_username').'</label>
            <div class="col-md-12">
              <div class="input-group from-group">
                <input type="text" class="form-control" name="'.$div_id.'[]" value="'.$bounce_email.'" placeholder="'.__('app.pmta_bounce_username').'" required>
              </div>
            </div>
          </div>
          <div class="col-md-6 mt-2">
            <label class="form-label">'.__('app.pmta_bounce_password').'</label>
            <div class="col-md-12">
              <div class="input-group from-group">
                <input type="text" class="form-control" name="'.$div_id.'[]" value="" placeholder="'.__('app.pmta_bounce_password').'" required>
              </div>
            </div>
          </div>
          <div class="col-md-6 mt-2">
            <label class="form-label">'.__('app.pmta_bounce_encryption').'</label>
            <div class="col-md-12">
              <div class="input-group from-group">
                <select name="'.$div_id.'[]" class="form-select form-select-lg" required>
                  <option value="none" selected="selected">None</option>
                  <option value="ssl">SSL</option>
                  <option value="tls">TLS</option>
                </select>
              </div>
            </div>
          </div>
          <div class="col-md-6 mt-2">
            <label class="form-label">'.__('app.pmta_bounce_validate_cert').'</label>
            <div class="col-md-12">
              <div class="input-group from-group">
                <select name="'.$div_id.'[]" class="form-select form-select-lg" required>
                  <option value="No">'.__('app.no').'</option>
                  <option value="Yes">'.__('app.yes').'</option>
                </select>
              </div>
            </div>
          </div>
          <div class="col-md-6 mt-2">
            <label class="form-label">'.__('app.pmta_bounce_delete_after_processing').'</label>
            <div class="col-md-12">
              <div class="input-group from-group">
                <select name="'.$div_id.'[]" class="form-select form-select-lg">
                  <option value="No">'.__('app.no').'</option>
                  <option value="Yes">'.__('app.yes').'</option>
                </select>
              </div>
            </div>
          </div>
          <div class="form-group">
            <div class="mt-2">
              <button type="button" class="btn btn-success btn-loading" onclick="validateBounce(\''.$div_id.'\')">'.__('app.validate').'</button><span id="msg-'.$div_id.'" style="padding-left:20px;"></span>
            </div>
          </div>';
      }
      
      $str .= '</div></div>';
      $str .= '</div></div>';
    }
    return $str;
    }

    /**
    * DNS entries for PMTA
    */
    private function authentication($request, $pmta_data=false)
    {
    $pmta = $pmta_data ? $request : $request->all();

    $main_domain = Helper::getAppURL(true);
    $smtp_a_record = $pmta['smtp_host'];
    $keys = Helper::generateKeys();
    $domain_keys['data']["main_domain_public_key"] = $keys['public_key'];
    $domain_keys['data']["main_domain_private_key"] = $keys['private_key'];
    $dkim_host = "fallback._domainkey";
    $dkim_value = 'v=DKIM1; k=rsa; p='.str_replace(['-----BEGINPUBLICKEY-----', '-----ENDPUBLICKEY-----'], ['', ''], $keys['public_key']);
    $str = "<div class='accordion accordion-primary' id='accordionAuthentication'>";
    $str .= "<div class='accordion-item'><h4 class='accordion-header' id='headingSMTP'><button class='accordion-button' type='button' data-bs-toggle='collapse' data-bs-target='#smtp-record' aria-expanded='true'                           aria-controls='collapseOne'>$main_domain</button></div>";
    $str .= "<div id='smtp-record' class='accordion-collapse collapse show' aria-labelledby='headingSMTP' data-bs-parent='#accordionBounces'>";
    $str .= "<div class='accordion-body'>
              <table class='table table-striped table-bordered table-hover'>
                <tr>
                <td style='padding:10px;' width='40%'><strong>".__('app.host')."</strong></td>
                <td style='padding:10px;' width='10%'><strong>".__('app.type')."</strong></td>
                <td style='padding:10px;' width='50%'><strong>".__('app.value')."</strong></td>
                </tr>
                <tr>
                <td style='padding:10px;'>{$smtp_a_record}</td>
                <td style='padding:10px;'>A</td>
                <td style='padding:10px;'>
                <div class='input-group'>
                <input type='text' class='form-control' value='{$pmta['server_ip']}' readonly='readonly' id='main-domain' onclick='copy(this.id)'>
                <div class='input-group-text' style='cursor: pointer;' onclick='copy(\"main-domain\")'><i class='bi bi-copy'></i></div></div>
                </td>
                </tr>
                <tr>
                <td style='padding:10px;'>{$dkim_host}</td>
                <td style='padding:10px;'>TXT</td>
                <td style='padding:10px;'>
                <div class='input-group'>
                <input type='text' class='form-control' value='{$dkim_value}' id='main-domain-dkim' readonly='readonly'>
                <div class='input-group-text' style='cursor: pointer;' onclick='copy(\"main-domain-dkim\")'><i class='bi bi-copy'></i></div></div>
                </td>
                </tr>
              </table>
             </div></div>";
    $str .= "</div></div></div>";

    $ips = Helper::splitLineBreakWithComma($pmta['ips']);
    $domains = Helper::splitLineBreakWithComma($pmta['domains']);
    $ip_per_domain = $this->ipPerDomain($ips, $domains);

    foreach($domains as $domain) {
      $keys = Helper::generateKeys();
      $domain_part = explode('.', $domain)[0];
      $domain_keys['data']["{$domain_part}_public_key"] = $keys['public_key'];
      $domain_keys['data']["{$domain_part}_private_key"] = $keys['private_key'];
      $pmta['dkim_selector'] = $pmta['dkim_selector'] ?? config('custom.dkim_selector');
      $pmta['dmarc_selector'] = $pmta['dmarc_selector'] ?? config('custom.dmarc_selector');
      $pmta['tracking_selector'] = $pmta['tracking_selector'] ?? config('custom.tracking_selector');
      $dkim_host = $pmta['dkim_selector']."._domainkey";

      $dkim_value = 'v=DKIM1; k=rsa; p='.str_replace(['-----BEGINPUBLICKEY-----', '-----ENDPUBLICKEY-----'], ['', ''], $keys['public_key']);
      $dmarc_attributes = "";
      if($pmta['dmarc_policy'] != 'none') {
          $dmarc_attributes = "pct=100;";
      }
      $dmarc_value = "v=DMARC1;p={$pmta['dmarc_policy']};";
      $cname_host = "{$pmta['tracking_selector']}";
      $dmarc_host = "{$pmta['dmarc_selector']}";

      if(Helper::isSubdomain($domain)) {
        $dkim_host .= ".{$domain_part}";
        $dmarc_host .= ".{$domain_part}";
      }
      
      $div_id = 'domain-'.rand();
      $str .= "<div class='accordion accordion-primary' id='accordionDNS'>";
      $str .= "<div class='accordion-item'><h4 class='accordion-header' id='headingDNS'><button class='accordion-button' type='button' data-bs-toggle='collapse' data-bs-target=\"#{$div_id}\" aria-expanded='true'                           aria-controls='collapseOne'>$domain</button></div>";
      $str .= "<div id=\"{$div_id}\" class='accordion-collapse collapse show' aria-labelledby='headingDNS' data-bs-parent='#accordionDNS'>";
      $str .= "<div class='accordion-body'>
                <table class='table table-striped table-bordered table-hover'>
                  <tr>
                  <td style='padding:10px;' width='40%'><strong>".__('app.host')."</strong></td>
                  <td style='padding:10px;' width='10%'><strong>".__('app.type')."</strong></td>
                  <td style='padding:10px;' width='50%'><strong>".__('app.value')."</strong></td>
                  </tr>

                  <tr>
                  <td style='padding:10px;'>{$cname_host}</td>
                  <td style='padding:10px;'>CNAME</td>
                  <td style='padding:10px;'>
                  <div class='input-group'>
                  <input type='text' class='form-control' value='{$main_domain}' id='cname-{$div_id}' readonly='readonly'>
                  <div class='input-group-text' style='cursor: pointer;' onclick='copy(\"cname-{$div_id}\")'><i class='bi bi-copy'></i></div></div>
                  </td>
                  </tr>

                  <tr>
                  <td style='padding:10px;'>{$dkim_host}</td>
                  <td style='padding:10px;'>TXT</td>
                  <td style='padding:10px;'>
                  <div class='input-group'>
                  <input type='text' class='form-control' value='{$dkim_value}' id='key-{$div_id}' readonly='readonly'>
                  <div class='input-group-text' style='cursor: pointer;' onclick='copy(\"key-{$div_id}\")'><i class='bi bi-copy'></i></div></div>
                  </td>
                  </tr>

                  <tr>
                  <td style='padding:10px;'>{$dmarc_host}</td>
                  <td style='padding:10px;'>TXT</td>
                  <td style='padding:10px;'>
                  <div class='input-group'>
                  <input type='text' class='form-control' value='{$dmarc_value}' id='dmarc-{$div_id}' readonly='readonly'>
                  <div class='input-group-text' style='cursor: pointer;' onclick='copy(\"dmarc-{$div_id}\")'><i class='bi bi-copy'></i></div></div>
                  </td>
                  </tr>
                  ";

      // otherwise array shift will empty the ips array
      $ips1 = $ips;
      $spf_value = 'v=spf1 mx ';
      for ($i=0; $i<$ip_per_domain; $i++) {
        if(!empty($ips1)) {
          $ip = array_shift($ips1);
          $spf_value .= "ip4:{$ip} ";
        }
      }
      $spf_value .= '~all';

      $str .= "<tr>
      <td style='padding:10px;'>{$domain}</td>
      <td style='padding:10px;'>TXT</td>
      <td style='padding:10px;'>
      <div class='input-group'>
      <input type='text' class='form-control' value='{$spf_value}' id='spf-{$div_id}' readonly='readonly'>
      <div class='input-group-text' style='cursor: pointer;' onclick='copy(\"spf-{$div_id}\")'><i class='bi bi-copy'></i></div></div>
      </td>
      </tr>";

      $str .= "<tr>
      <td style='padding:10px;'>{$domain}</td>
      <td style='padding:10px;'>MX (Priority:10)</td>
      <td style='padding:10px;'>
      <div class='input-group'>
      <input type='text' class='form-control' value='{$domain}' id='mx-{$div_id}' readonly='readonly'>
      <div class='input-group-text' style='cursor: pointer;' onclick='copy(\"mx-{$div_id}\")'><i class='bi bi-copy'></i></div></div>
      </td>
      </tr>";
      
      $vmta_selector_number = 0;
      $ptr_data = [];
      for ($i=0; $i<$ip_per_domain; $i++) {
        if(!empty($ips)) {
          $ip = array_shift($ips);
          $selector_no = sprintf("%02d", $vmta_selector_number);
          $vmta_selector = $pmta['vmta_prefix'].$selector_no;

          $str .= "<tr>
          <td style='padding:10px;'>{$vmta_selector}.{$domain}</td>
          <td  style='padding:10px;'>A</td>
          <td style='padding:10px;'>
          <div class='input-group'>
          <input type='text' class='form-control' value='{$ip}' id='ip-{$div_id}' readonly='readonly'>
          <div class='input-group-text' style='cursor: pointer;' onclick='copy(\"ip-{$div_id}\")'><i class='bi bi-copy'></i></div></div>
          </td>
          </tr>";
          $ptr_data[$vmta_selector_number]['ip'] = $ip;
          $ptr_data[$vmta_selector_number]['domain'] = $vmta_selector.".{$domain}";
        }
        $vmta_selector_number += 1;
      }

      if(count($ptr_data)) {
        $str .= "<tr>
                  <td style='padding:10px;' colspan=3><strong>Reverse DNS (PTR)</strong></td>
                </tr>";
      }

      foreach($ptr_data as $key_prt => $value_ptr) {
          $str .= "<tr>
                    <td style='padding:10px;'>{$value_ptr['ip']}</td>
                    <td  style='padding:10px;'>PTR</td>
                    <td style='padding:10px;'>
              <div class='input-group'>
              <input type='text' class='form-control' value='{$value_ptr['domain']}' id='ptr-{$div_id}' readonly='readonly'>
              <div class='input-group-text' style='cursor: pointer;' onclick='copy(\"ptr-{$div_id}\")'><i class='bi bi-copy'></i></div></div>
              </td>
              </tr>";
      }

      $str .= "</table>
               </div></div>";
      $str .= "</div></div></div>";
    }

    foreach($domain_keys['data'] as $key => $value) {
      $str .= "<input type='hidden' name=domain_keys[$key] value='{$value}'> ";
    }

    return $str;
    }

    /**
    * Return config for PMTA
    */
    private function config($request, $pmta_data=false)
    {
    $pmta = $pmta_data ? $request : $request->all();
    $main_domain = Helper::getAppURL(true);

    $str = "";
    $admin_ips = explode(',', $pmta['admin_ips']);
    foreach($admin_ips as $ip) {
      $ip = trim($ip);
      if($ip) {
        $str .= "http-access ".$ip." admin\n";
      }
    }

    $str .= "http-access 0/0 monitor\n";

    $str .= "postmaster admin@".$main_domain."\n";
    $str .= "http-mgmt-port {$pmta['management_port']}\nrun-as-root yes\nsync-msg-update false\n\n";

    $str .= "
# BEGIN: BACKOFF RULES

<smtp-pattern-list common-errors> 
reply /generating high volumes of.* complaints from AOL/    mode=backoff 
reply /Excessive unknown recipients - possible Open Relay/  mode=backoff 
reply /^421 .* too many errors/                             mode=backoff 
reply /blocked.*spamhaus/                                   mode=backoff 
reply /451 Rejected/                                        mode=backoff 
</smtp-pattern-list>

<smtp-pattern-list blocking-errors>

# AOL Errors
reply /421 .* SERVICE NOT AVAILABLE/ mode=backoff
reply /554 .*aol\.com/ mode=backoff
reply /421dynt1/ mode=backoff
reply /HVU:B1/ mode=backoff
reply /DNS:NR/ mode=backoff
reply /RLY:NW/ mode=backoff
reply /DYN:T1/ mode=backoff
reply /RLY:BD/ mode=backoff
reply /RLY:CH2/ mode=backoff

# Yahoo Errors
reply /421 .* Please try again later/ mode=backoff
reply /421 Message temporarily deferred/ mode=backoff
reply /VS3-IP5 Excessive unknown recipients/ mode=backoff
reply /VSS-IP Excessive unknown recipients/ mode=backoff
reply /\[TS0[1-3]\] Messages? from/ mode=backoff
reply /\[GL01\] Message from/ mode=backoff

# Hotmail/Outlook Errors
reply /exceeded the (rate|connection) limit/ mode=backoff
reply /Mail rejected by Windows Live Hotmail for policy reasons/ mode=backoff
reply /mail\.live\.com\/mail\/troubleshooting\.aspx/ mode=backoff

# Road Runner / Adelphia Errors
reply /421 Message Rejected/ mode=backoff
reply /Client host rejected/ mode=backoff
reply /blocked using UCEProtect/ mode=backoff
reply /Mail Refused/ mode=backoff
reply /421 Exceeded allowable connection time/ mode=backoff
reply /amIBlockedByRR/ mode=backoff
reply /block-lookup/ mode=backoff
reply /Too many concurrent connections from source IP/ mode=backoff

# General / ISP Neutral
reply /too many/ mode=backoff
reply /Exceeded allowable connection time/ mode=backoff
reply /Connection rate limit exceeded/ mode=backoff
reply /refused your connection/ mode=backoff
reply /try (again )?later/ mode=backoff
reply /550 RBL/ mode=backoff
reply /TDC internal RBL/ mode=backoff
reply /connection refused/ mode=backoff
reply /www\.spamhaus\.org/ mode=backoff
reply /Message Rejected/ mode=backoff
reply /refused by antispam/ mode=backoff
reply /Service not available/ mode=backoff
reply /currently blocked/ mode=backoff
reply /locally blacklisted/ mode=backoff
reply /not currently accepting mail from your ip/ mode=backoff
reply /421.*closing connection/ mode=backoff
reply /421.*Lost connection/ mode=backoff
reply /476 connections from your host are denied/ mode=backoff
reply /421 Connection cannot be established/ mode=backoff
reply /421 temporary envelope failure/ mode=backoff
reply /421 4\.4\.2 Timeout while waiting for command/ mode=backoff
reply /450 Requested action aborted/ mode=backoff
reply /550 Access denied/ mode=backoff
reply /permanently deferred/ mode=backoff
reply /\d+\.\d+\.\d+\.\d+ blocked/ mode=backoff
reply /www\.spamcop\.net\/bl\.shtml/ mode=backoff

</smtp-pattern-list>

# END: BACKOFF RULES


# BEGIN: BOUNCE RULES

<bounce-category-patterns>

# Spam-related
/spam|junk mail|blacklist|blocked|U\.?C\.?E\.?|Adv(ertisements?)?|unsolicited|openRBL|realtime blackhole|basic\.wirehub\.nl\/blackholes\.html/ spam-related

# Virus
/\bvirus\b/ virus-related

# Content-based
/message +content|content +rejected/ content-related

# Quota / Mailbox Full
/quota|limit exceeded|mailbox +(is +)?full|\bstorage\b/ quota-issues

# Inactive
/(user|mailbox|recipient|rcpt|local part|address|account|mail drop|ad(d?)ressee) (has|has been|is)? *(currently|temporarily +)?(disabled|expired|inactive|not activated)/ inactive-mailbox
/(conta|usu.rio) inativ(a|o)/ inactive-mailbox

# Bad Mailbox
/Too many (bad|invalid|unknown|illegal|unavailable) (user|mailbox|recipient|rcpt|local part|address|account|mail drop|ad(d?)ressee)/ other
/(No such|bad|invalid|unknown|illegal|unavailable) (local +)?(user|mailbox|recipient|rcpt|local part|address|account|mail drop|ad(d?)ressee)/ bad-mailbox
/(user|mailbox|recipient|rcpt|local part|address|account|mail drop|ad(d?)ressee) +(\S+@\S+ +)?(not (a +)?valid|not known|not here|not found|does not exist|bad|invalid|unknown|illegal|unavailable)/ bad-mailbox
/\S+@\S+ +(is +)?(not (a +)?valid|not known|not here|not found|does not exist|bad|invalid|unknown|illegal|unavailable)/ bad-mailbox
/no mailbox here by that name|my badrcptto list|not our customer|no longer (valid|available)|have a \S+ account/ bad-mailbox

# Relaying and Domain
/\brelay(ing)?/ relaying-issues
/domain (retired|bad|invalid|unknown|illegal|unavailable)/ bad-domain
/domain no longer in use|domain (\S+ +)?(is +)?obsolete/ bad-domain

# Policy
/denied|prohibit|rejected|refused|allowed|banned|policy|suspicious activity/ policy-related

# Protocol
/bad sequence|syntax error/ protocol-errors

# Routing
/\broute\b|unroutable|unrouteable/ routing-errors

# DSN codes
/^2\.\d+\.\d+/ success
/^[45]\.1\.1/ bad-mailbox
/^[45]\.1\.2/ bad-domain
/^[45]\.3\.5/ bad-configuration
/^[45]\.4\.1/ no-answer-from-host
/^[45]\.4\.2/ bad-connection
/^[45]\.4\.4|^[45]\.4\.6/ routing-errors
/^[45]\.4\.7/ message-expired
/^[45]\.7\.1/ policy-related

# Fallback
// other

</bounce-category-patterns>

# END: BOUNCE RULES


# BEGIN: FBL

<feedback-loop-processor>
deliver-unmatched-email no
deliver-matched-email yes
forward-unmatched-to auto-feedback@port25.com
forward-errors-to auto-feedback@port25.com

<address-list>
   # You can enable this for live FBL emails:
   # address /contact@yourdomain.com/
</address-list>
</feedback-loop-processor>

<acct-file /var/log/pmta/fbl.csv>
records feedback-loop
record-fields f *,header_subject
</acct-file>

# END: FBL\n\n";

    $virtual_mta = $virtual_mta_pool = $smtp_user = $smtp_listener = $dkim_selector = '';

    $smtp_ports = explode(',', $pmta['smtp_port']);
    foreach($smtp_ports as $smtp_port) {
      $smtp_port = trim($smtp_port);
      if($smtp_port) {
        $smtp_listener .= "smtp-listener {$pmta['server_ip']}:{$smtp_port}\n";
      }
    }

    $ips = Helper::splitLineBreakWithComma($pmta['ips']);
    $domains = Helper::splitLineBreakWithComma($pmta['domains']);
    $ip_per_domain = $this->ipPerDomain($ips, $domains);

    $loop = 0;
    foreach ($domains as $domain) {
            $domain = trim($domain);
            $vmta_selector_number = 0;
            $shared_vmta_pool = '';
            for ($i=0; $i<$ip_per_domain; $i++) {
                if (!empty($ips)){
                    $ip = array_shift($ips);
                    $selector_no = sprintf("%02d", $vmta_selector_number);
                    $vmta_selector_number = sprintf("%02d", $vmta_selector_number);

    $virtual_mta .= "
    <virtual-mta {$pmta['vmta_prefix']}".$vmta_selector_number.".{$domain}>
      smtp-source-host {$ip} {$pmta['vmta_prefix']}{$vmta_selector_number}.{$domain}
    </virtual-mta>\n";


    if(!$pmta['vmta_shared_pool']) {
    $virtual_mta_pool .= "
    <virtual-mta-pool vmtap".$vmta_selector_number.".{$domain}>
      virtual-mta {$pmta['vmta_prefix']}{$vmta_selector_number}.{$domain}
    </virtual-mta-pool>\n";

    $smtp_username = str_replace('.', '', $domain).$i;
    $smtp_password = substr(hash('ripemd160',$domain), 20);
    $smtp_user .= "
    <smtp-user {$smtp_username}>
      password {$smtp_password}
      source {server{$loop}}
    </smtp-user>
    <source {server{$loop}}>
      default-virtual-mta vmtap{$vmta_selector_number}.{$domain}
      always-allow-relaying yes
      smtp-service yes
    </source>\n";
    } else {
    $shared_vmta_pool .= "    virtual-mta {$pmta['vmta_prefix']}{$vmta_selector_number}.{$domain} \n";
    }
                    if($ip != $pmta['server_ip']) {
                      foreach($smtp_ports as $smtp_port) {
                        $smtp_port = trim($smtp_port);
                        if($smtp_port) {
                          $smtp_listener .= "smtp-listener {$ip}:{$smtp_port}\n";
                        }
                      }
                      
                    }
                }
                $vmta_selector_number += 1;
                $loop++;
            }

    if($pmta['vmta_shared_pool']) {
    $virtual_mta_pool .= "
    <virtual-mta-pool vmtap01".".{$domain}>\n";
    $virtual_mta_pool .= $shared_vmta_pool;
    $virtual_mta_pool .= "
    </virtual-mta-pool>\n";

    $smtp_username = str_replace('.', '', $domain).'01';
    $smtp_password = substr(hash('ripemd160',$domain), 20);
    $smtp_user .= "
    <smtp-user {$smtp_username}>
      password {$smtp_password}
      source {server{$loop}}
    </smtp-user>
    <source {server{$loop}}>
      default-virtual-mta vmtap01.{$domain}
      always-allow-relaying yes
      smtp-service yes
    </source>\n";
    }
            $pmta['dkim_selector'] = $pmta['dkim_selector'] ?? config('custom.dkim_selector');
            $dkim_selector .= "domain-key ".$pmta['dkim_selector'].",{$domain}, /etc/pmta/dkim/".$pmta['dkim_selector'].".{$domain}.pem\n";
        }
    $str .= "# BEGIN: VMTAs";
        $str .= $virtual_mta;
    $str .= "# END: VMTAs\n\n";

    $str .= "# BEGIN: VMTA POOLs";
        $str .= $virtual_mta_pool;
    $str .= "# END: VMTA POOLs\n\n";

    $str .= "# BEGIN: SMTPs";
        $str .= $smtp_user;
    $str .= "# END: SMTPs\n\n";
        $str .= $smtp_listener;

        $tls_value = (!empty($pmta['smtp_encryption']) && $pmta['smtp_encryption'] != 'none') ? 'yes' : 'no';

        $str  .="
    <source 0/0>
    jobid-header Message-ID 
    process-x-job yes
    hide-message-source yes
    allow-unencrypted-plain-auth yes
    hide-message-source yes
    always-allow-relaying yes
    add-received-header no
    process-x-virtual-mta yes
    max-message-size unlimited 
    smtp-service yes
    require-auth true
    add-message-id-header yes
    </source>\n\n";

    $str .= "# DKIM SELECTORS START \n";
    $str .= $dkim_selector;
    $str .= "# DKIM SELECTORS END \n";

        $str.= "
    <domain aol.com>
    max-smtp-out 4
    max-msg-per-connection 20
    connection-idle-timeout 1m
    use-starttls  {$tls_value}
    require-starttls no
    </domain>

    <domain yahoo.com>
    max-smtp-out 3
    max-msg-per-connection 20
    smtp-greeting-timeout 5m
    mx-connection-attempts 5
    smtp-pattern-list blocking-errors
    backoff-to-normal-after 2h 
    backoff-max-msg-rate 50/h 
    backoff-retry-after 90m
    dkim-sign yes
    use-starttls  {$tls_value}
    require-starttls no
    </domain>


    <domain gmail.com>
    max-smtp-out 3
    max-msg-per-connection 50
    smtp-greeting-timeout 5m
    mx-connection-attempts 5
    smtp-pattern-list blocking-errors
    backoff-to-normal-after 2h 
    backoff-max-msg-rate 50/h
    backoff-retry-after 90m
    dkim-sign yes
    use-starttls  {$tls_value}
    require-starttls no
    </domain>

    <domain hotmail.com>
    max-smtp-out 3
    max-msg-per-connection 50
    smtp-pattern-list blocking-errors
    backoff-to-normal-after 2h 
    backoff-max-msg-rate 50/h
    backoff-retry-after 90m
    dkim-sign yes
    use-starttls  {$tls_value}
    require-starttls no
    </domain>

    <domain msn.com>
    max-smtp-out 3
    max-msg-per-connection 50
    smtp-pattern-list blocking-errors
    backoff-to-normal-after 2h
    backoff-max-msg-rate 50/h
    backoff-retry-after 90m
    dkim-sign yes
    use-starttls  {$tls_value}
    require-starttls no
    </domain>

    <domain att.net>
    max-smtp-out 2
    dkim-sign yes
    use-starttls  {$tls_value}
    require-starttls no
    </domain>

    <domain comcast.net>
    dkim-sign yes
    max-smtp-out 2
    max-msg-per-connection 20
    use-starttls  {$tls_value}
    require-starttls no
    </domain>


    <domain *>
    max-smtp-out 300
    bounce-after 5d
    retry-after  10m
    max-msg-per-connection 100
    dkim-sign yes
    dkim-identity sender-or-from
    dkim-identity-fallback @{$main_domain}
    #    log-commands    yes
    backoff-to-normal-after 2h
    backoff-to-normal-after-delivery true
    backoff-retry-after 30m
    backoff-max-msg-rate  10/m
    bounce-upon-no-mx yes
    smtp-pattern-list blocking-errors
    use-starttls  {$tls_value}
    require-starttls no
    </domain>\n\n";

    $str .= "log-file {$pmta['log_file_path']}\n";
    $str .= "log-rotate 10\n";

    $str .= "
    <acct-file {$pmta['acct_file_path']}>
    record-fields delivery *,envId,jobId,bounceCat
    move-interval 5m
    delete-after 7d
    max-size 50M
    user-string from
    </acct-file>\n";

    $str .= "
    <acct-file {$pmta['diag_file_path']}>
    move-interval 1d
    delete-after 7d
    </acct-file>\n\n";

    $str .= "
    <spool {$pmta['spool_path']}>
    priority 50
    delete-file-holders false
    </spool>\n\n";

    //    $str .= "spool {$pmta['spool_path']}\n\n";

    $str .= "host-name {$pmta['smtp_host']}".'.'.$main_domain;

    return $str;
    }

    private function finish($request)
    {
    try {
      $ssh = new \phpseclib3\Net\SSH2($request->server_ip, $request->server_port);
      if (!$ssh->login($request->server_username, $request->server_password)) {
        return '<span class="alert text-danger">'.__('app.login_failed').'</span>';
      } else {
        $step = $this->pmtaFiles($request);
        if($step === true) {
          $step = $this->createConfigFile($request);
          if($step === true) {
            $step = $this->createAppEntries($request);
            if($step === true) {
              $this->restartServices($request);
              activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.setup_pmta') . " ({$request->server_name}) ". __('app.log_create')); // log
              session()->flash('msg', trans('app.pmta_msg_success'));
              return 'success';
            }
          }
        }
        return $step;
      }
    } catch (\Exception $e) {
      return '<span class="text-danger">'.$e->getMessage().'</span>';
    }
    }

    private function pmtaFiles($request)
    {
    $sftp = new \phpseclib3\Net\SFTP($request->server_ip, $request->server_port);
    if (!$sftp->login($request->server_username, $request->server_password)) {
       return '<span class="alert text-danger">'.__('app.login_failed').'</span>';
    } else {
      // Create log files directory if not exist
      $log_file_path = substr($request->log_file_path, 0, strrpos($request->log_file_path, '/'));
      if(!$sftp->file_exists($log_file_path)) {
        try {
          $sftp->mkdir($log_file_path, -1, true);
        } catch(\Exception $e) {
          return '<span class="text-danger">'.$e->getMessage().'</span>';
        }
      }

      // Create accountings files directory if not exist
      $acct_file_path = substr($request->acct_file_path, 0, strrpos($request->acct_file_path, '/'));
      if(!$sftp->file_exists($acct_file_path)) {
        try {
          $sftp->mkdir($acct_file_path, -1, true);
        } catch(\Exception $e) {
          return '<span class="text-danger">'.$e->getMessage().'</span>';
        }
      }

      // Create diag files directory if not exist
      $diag_file_path = substr($request->diag_file_path, 0, strrpos($request->diag_file_path, '/'));
      if(!$sftp->file_exists($diag_file_path, -1, true)) {
        try {
          $sftp->mkdir($diag_file_path);
        } catch(\Exception $e) {
          return '<span class="text-danger">'.$e->getMessage().'</span>';
        }
      }

      // Create dkim files directory if not exist
      if(!$sftp->file_exists($request->dkim_path)) {
        try {
          $sftp->mkdir($request->dkim_path, -1, true);
        } catch(\Exception $e) {
          return '<span class="text-danger">'.$e->getMessage().'</span>';
        }
      }
      return true;
    }
    }

    /**
    * Create PMTA config file
    */
    private function createConfigFile($request)
    {
    $ssh = new \phpseclib3\Net\SSH2($request->server_ip, $request->server_port);
    try {
      if (!$ssh->login($request->server_username, $request->server_password)) {
        return '<span class="alert alert-danger">'.__('app.login_failed').'</span>';
      } else {
        // Backup old config pmta file
        $ssh->exec("cp {$request->path}/config {$request->path}/config-backup-".date('Y-m-d'));

        // Create new config file for pmta
        $sftp = new \phpseclib3\Net\SFTP($request->server_ip, $request->server_port);
        if (!$sftp->login($request->server_username, $request->server_password)) {
          return '<span class="alert text-danger">'.__('app.login_failed').'</span>';
        } else {
          $pmta_data =  $this->config($request);
          $sftp->put("{$request->path}/config", $pmta_data);
        }
        return true;
      }
    } catch (\Exception $e) {
      return '<span class="text-danger">'.$e->getMessage().'</span>';
    }
    }

    /**
    * Sending Servers, Sending Domains & Bounces entries for MailCarry
    */
    private function createAppEntries($request)
    {
    $input = $request->except('_token');
    // Save/Edit pmta data
    if(!empty($input['pmta_id'])) {
      Pmta::whereId($input['pmta_id'])->update([
        'name' => $input['server_name'],
        'attributes' => json_encode($input)
      ]);
      $pmta_id = $input['pmta_id'];
    } else {
      $pmta = Pmta::create([
        'name' => $input['server_name'],
        'attributes' => json_encode($input),
        'app_id' => Auth::user()->app_id,
        'user_id' => Auth::user()->id
      ]);
      $pmta_id = $pmta->id;
    }

    $ips     = Helper::splitLineBreakWithComma($input['ips']);
    $domains = Helper::splitLineBreakWithComma($input['domains']);
    $ip_per_domain = $this->ipPerDomain($ips, $domains);

    // otherwise array shift will empty the ips array
    $ips1 = $ips;

    // Create group for sending server if not exist
    $group_sending_server = app('App\Http\Controllers\GroupController')->createGroup(config('custom.group_sending_servers'), 'PowerMTA');
    $group_sending_domain = app('App\Http\Controllers\GroupController')->createGroup(config('custom.group_sending_domains'), 'PowerMTA');

    $sftp = new \phpseclib3\Net\SFTP($request->server_ip, $request->server_port);
    $sftp->login($request->server_username, $request->server_password);

    // Upload Main Domain Private Key
    $main_domain = Helper::getAppURL(true);
    $private_key = $input['domain_keys']["main_domain_private_key"];
    $private_key_file = "fallback.{$main_domain}.pem";
    $sftp->put($request->dkim_path.'/'.$private_key_file, $private_key);


    // Delete all data before new update otherwise duplicate the data
    if(!empty($input['pmta_id'])) {
      // Delete Bounces
      \App\Models\Bounce::wherePmta($input['pmta_id'])->delete();
      // Delete SendingDomain
      \App\Models\SendingDomain::wherePmta($input['pmta_id'])->delete();
      // Delete TrackingDomain
      \App\Models\TrackingDomain::wherePmta($input['pmta_id'])->delete();
      // Delete SendingServer
      \App\Models\SendingServer::wherePmta($input['pmta_id'])->delete();      
    }

    $shared_pool_ip_no = 0;
    foreach($domains as $domain) {
      $domain = trim($domain);

      $sending_domain_part = explode('.', $domain);

      // Bounces data
      // If bounces processed by pmta files
      $bounce_id = null;
      if(!empty($input['bounce_pmta'][$domain]) && $input['bounce_pmta'][$domain] == 'Y') {
        $bounce_id = null;
      } else {
        $password = $input["bounce-{$sending_domain_part[0]}"][4];
        $bounce_data = [
          'email'   => $input['bounce_email'][$domain]."@{$domain}",
          'method'  => $input["bounce-{$sending_domain_part[0]}"][0],
          'host'  => $input["bounce-{$sending_domain_part[0]}"][1] ?? null,
          'username'   => $input["bounce-{$sending_domain_part[0]}"][3],
          'password'  => !empty($password) ? \Crypt::encrypt($password) : null,
          'port'      => $input["bounce-{$sending_domain_part[0]}"][2] ?? 143,
          'encryption'      => $input["bounce-{$sending_domain_part[0]}"][5],
          'user_id'  => Auth::user()->id,
          'app_id'  => Auth::user()->app_id,
          'validate_cert' => $input["bounce-{$sending_domain_part[0]}"][6],
          'pmta'    => $pmta_id
        ];
        $bounce = \App\Models\Bounce::create($bounce_data);
        $bounce_id = $bounce->id;
      }


      $spf_value = 'v=spf1 mx ';
      for ($i=0; $i<$ip_per_domain; $i++) {
        if(!empty($ips1)) {
          $ip = array_shift($ips1);
          $spf_value .= "ip4:{$ip} ";
        }
      }
      $spf_value .= '~all';


      // Upload Domain Private Keys
      $private_key = $input['domain_keys']["{$sending_domain_part[0]}_private_key"];
      $private_key_file = $request->dkim_selector.".{$domain}.pem";
      $sftp->put($request->dkim_path.'/'.$private_key_file, $private_key);

      $public_key = $input['domain_keys']["{$sending_domain_part[0]}_public_key"];

      $dkim_host = $input['dkim_selector']."._domainkey";

      if(Helper::isSubdomain($domain)) {
        $dkim_host .= ".{$sending_domain_part[0]}";
      }

      // Sending Domain data
      $sending_domain_data = [
        'group_id'  => $group_sending_domain->id,
        'domain'   => $domain,
        'protocol'     => $input['protocol'][$domain],
        'public_key' => $public_key,
        'private_key' => $private_key,
        
        'host_dkim' => $dkim_host,
        'value_dkim' => "v=DKIM1; k=rsa; p=".str_replace(['-----BEGINPUBLICKEY-----', '-----ENDPUBLICKEY-----'], ['', ''], $public_key),

        'host_spf' => $domain,
        'value_spf' => $spf_value,
        
        'host_dmarc' => "{$input['dmarc_selector']}.".$domain,
        'value_dmarc' =>$dmarc_value = "v=DMARC1;p={$input['dmarc_policy']};",

        'user_id' => Auth::user()->id,
        'app_id'  => Auth::user()->app_id,
        'pmta'    => $pmta_id
      ];

      
      \App\Models\SendingDomain::create($sending_domain_data);


      // Tracking Domain Data
      $tracking_domain_data = [
        'domain'   => $domain,
        'protocol'     => $input['protocol'][$domain],
        'user_id' => Auth::user()->id,
        'app_id'  => Auth::user()->app_id,
        'pmta'    => $pmta_id
      ];
      \App\Models\TrackingDomain::create($tracking_domain_data);


      // Sending Server data
      if(!$input['vmta_shared_pool']) {
        for ($i=0; $i<$ip_per_domain; $i++) {
          if(!empty($ips)) {
            $ip = array_shift($ips);
            $smtp_username = str_replace('.', '', $domain).$i;
            $smtp_password = substr(hash('ripemd160',$domain), 20);
            $data = [
              'type' => 'smtp',
              'host' => $ip,
              'username' => $smtp_username,
              'password' => $smtp_password,
              'port' => $input['smtp_port'],
              'encryption' => $input['smtp_encryption'],
              'body_encoding' => $input['body_encoding']
            ];
            $smtp_data = [
              'name'      => $ip,
              'type' => 'smtp',
              'group_id'  => $group_sending_server->id,
              'from_name' => $input['from_name'][$domain],
              'from_email'  => $input['from_email'][$domain].'@'.$domain,
              'reply_email' => $input['reply_email'][$domain],
              'tracking_domain' => "{$input['protocol'][$domain]}{$input['tracking_selector']}.{$domain}",
              'sending_attributes' => app('App\Http\Controllers\SendingServerController')->sendingServerAttributes($data),
              'speed_attributes' => json_encode(['speed'=>'Unlimited', 'limit'=>null, 'duration'=>null]),
              'bounce_id' => $bounce_id,
              'user_id' => Auth::user()->id,
              'app_id'  => Auth::user()->app_id,
              'pmta'    => $pmta_id
            ];
            \App\Models\SendingServer::create($smtp_data);
          }
        }
      } else {
          $ip = $ips[$shared_pool_ip_no];;
          $smtp_username = str_replace('.', '', $domain).'01';
          $smtp_password = substr(hash('ripemd160',$domain), 20);
          $data = [
              'type' => 'smtp',
              'host' => $ip,
              'username' => $smtp_username,
              'password' => $smtp_password,
              'port' => $input['smtp_port'],
              'encryption' => $input['smtp_encryption'],
              'body_encoding' => $input['body_encoding']
            ];
            $smtp_data = [
              'name'      => $ip,
              'type' => 'smtp',
              'group_id'  => $group_sending_server->id,
              'from_name' => $input['from_name'][$domain],
              'from_email'  => $input['from_email'][$domain].'@'.$domain,
              'reply_email' => $input['reply_email'][$domain],
              'tracking_domain' => "{$input['protocol'][$domain]}{$input['tracking_selector']}.{$domain}",
              'sending_attributes' => app('App\Http\Controllers\SendingServerController')->sendingServerAttributes($data),
              'speed_attributes' => json_encode(['speed'=>'Unlimited', 'limit'=>null, 'duration'=>null]),
              'bounce_id' => $bounce_id,
              'user_id' => Auth::user()->id,
              'app_id'  => Auth::user()->app_id,
              'pmta'    => $pmta_id
            ];
            \App\Models\SendingServer::create($smtp_data);
            $shared_pool_ip_no+=$ip_per_domain;

      }

      
    }
    return true;
    }

    /**
    * Check server connection
    */
    public function validateServer($request)
    {
    try {
      $ssh = new \phpseclib3\Net\SSH2($request->ip, $request->port);
      if (!$ssh->login($request->username, $request->password)) {
        return '<span class="alert text-danger">'.__('app.login_failed').'</span>';
      } else {
        return '<span class="alert text-success">'.__('app.msg_successfully_connected').'</span>';
      }
    } catch (\Exception $e) {
      return '<span class="text-danger">'.$e->getMessage().'</span>';
    }
    }

    /**
    * Restard server services
    */
    public function restartServices($request)
    {
    try {
      $ssh = new \phpseclib3\Net\SSH2($request->server_ip, $request->server_port);
      if (!$ssh->login($request->server_username, $request->server_password)) {
        return '<span class="alert text-danger">'.__('app.login_failed').'</span>';
      } else {
        if($request->server_os == 'ubuntu24.04' || $request->server_os == 'ubuntu22.04' || $request->server_os == 'ubuntu20.04' || $request->server_os == 'other') {
          $ssh->exec('/etc/init.d/pmta restart');
          sleep(2);
          $ssh->exec('/etc/init.d/pmtahttp restart');
        } elseif($request->server_os == 'centos8.x' || $request->server_os == 'centos9.x' || $request->server_os == 'centos10.x' || $request->server_os == 'almalinux9.x') {
          $ssh->exec('systemctl restart pmta.service');
          sleep(2);
          $ssh->exec('systemctl restart pmtahttp.service');
        }
      }
    } catch (\Exception $e) {
      return '<span class="text-danger">'.$e->getMessage().'</span>';
    }
    }

    /**
    * Delete PMTA
    */
    public function destroy($id)
    {
      // Delete Sending Domains
      \App\Models\SendingDomain::wherePmta($id)->delete();

      // Delete Bounces
      \App\Models\Bounce::wherePmta($id)->delete();

      // Delete Sending Servers
      \App\Models\SendingServer::wherePmta($id)->delete();

      $name = Pmta::whereId($id)->value('name');
      $destroy = Pmta::destroy($id);

      activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.setup_pmta') . " ({$name}) ". __('app.log_delete'));

      $data = [
              'success' => true,
              'message' => __('app.deleted_successfully')
          ];

      return response()->json($data, 200); 

    }

    /**
    * Create Bounce server for PMTA
    */
    public function createBouncePMTA()
    {
    Helper::checkPermissions('pmta'); // check user permission
    return view('pmta.create_bounce');
    }

    /**
    * Save Bounce server for PMTA
    */
    public function saveBouncePMTA(Request $request)
    {
    Helper::checkPermissions('pmta'); // check user permission
    $request->validate([
      'server_name' => 'required|string|max:255',
      'server_ip' => 'required|ip',
      'server_username' => 'required|string|max:255',
    ]);
    $input = $request->except('_token');
    $pmta = Pmta::create([
      'name' => $input['server_name'],
      'attributes' => json_encode($input),
      'type' => 'Bounce',
      'app_id' => Auth::user()->app_id,
      'user_id' => Auth::user()->id
    ]);
    activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.setup_pmta_bounce') . " ({$request->server_name}) ". __('app.log_create')); // log
    }

    /**
    * Retrun edit view for PMTA Bounce
    */
    public function editBouncePMTA($id)
    {
    Helper::checkPermissions('pmta'); // check user permission
    $pmta = Pmta::whereId($id)->where('app_id', \Auth::user()->app_id)->first();
    $pmta = json_decode($pmta->attributes);
    return view('pmta.edit_bounce')->with(compact('id','pmta'));
    }

    /**
    * Update Bounce server for PMTA
    */
    public function updateBouncePMTA(Request $request)
    {
    Helper::checkPermissions('pmta'); // check user permission
    $request->validate([
      'server_name' => 'required|string|max:255',
      'server_ip' => 'required|ip',
      'server_username' => 'required|string|max:255',
    ]);

    $input = $request->except('_token');
    $data = [
      'name' => $input['server_name'],
      'attributes' => json_encode($input),
      'type' => 'Bounce',
      'app_id' => Auth::user()->app_id,
      'user_id' => Auth::user()->id
    ];

    $pmta = Pmta::findOrFail($request->id);
    $pmta->fill($data)->save();

    activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.setup_pmta_bounce') . " ({$request->server_name}) ". __('app.log_create')); // log
    }

    public function restart($operation, $id) {
      $pmta = Pmta::whereId($id)->where('app_id', \Auth::user()->app_id)->first();
      $pmta = json_decode($pmta->attributes);
      $ssh = new \phpseclib3\Net\SSH2($pmta->server_ip, $pmta->server_port);
      try {
        if (!$ssh->login($pmta->server_username, $pmta->server_password)) {
          return '<span class="alert text-danger">'.__('app.login_failed').'</span>';
        }
      } catch (\Exception $e) {
        return '<span class="text-danger">'.$e->getMessage().'</span>';
      }
      switch ($operation) {
        case "restart":
          if($pmta->server_os == 'ubuntu24.04' || $pmta->server_os == 'ubuntu22.04' || $pmta->server_os == 'ubuntu20.04' || $pmta->server_os == 'other') {
              $ssh->exec('/etc/init.d/pmta restart');
              sleep(2);
              $ssh->exec('/etc/init.d/pmtahttp restart');
            } elseif($pmta->server_os == 'centos8.x' || $pmta->server_os == 'centos9.x' || $pmta->server_os == 'centos10.x' || $pmta->server_os == 'almalinux9.x') {
              $ssh->exec('systemctl restart pmta.service');
              sleep(2);
              $ssh->exec('systemctl restart pmtahttp.service');
            }
          return '<span class="text-success">'.__('app.pmta_server_restarted').'</span>';
          break;
        case "stop":
          if($pmta->server_os == 'ubuntu24.04' || $pmta->server_os == 'ubuntu22.04' || $pmta->server_os == 'ubuntu20.04' || $pmta->server_os == 'other') {
              $ssh->exec('/etc/init.d/pmta stop');
              sleep(2);
              $ssh->exec('/etc/init.d/pmtahttp stop');
            } elseif($pmta->server_os == 'centos8.x' || $pmta->server_os == 'centos9.x' || $pmta->server_os == 'centos10.x' || $pmta->server_os == 'almalinux9.x') {
              $ssh->exec('systemctl stop pmta.service');
              sleep(2);
              $ssh->exec('systemctl stop pmtahttp.service');
            }
          return '<span class="text-success">'.__('app.pmta_server_stoped').'</span>';
          break;
        case "start":
          if($pmta->server_os == 'ubuntu24.04' || $pmta->server_os == 'ubuntu22.04' || $pmta->server_os == 'ubuntu20.04' || $pmta->server_os == 'other') {
              $ssh->exec('/etc/init.d/pmta start');
              sleep(2);
              $ssh->exec('/etc/init.d/pmtahttp start');
            } elseif($pmta->server_os == 'centos8.x' || $pmta->server_os == 'centos9.x' || $pmta->server_os == 'centos10.x' || $pmta->server_os == 'almalinux9.x') {
              $ssh->exec('systemctl start pmta.service');
              sleep(2);
              $ssh->exec('systemctl start pmtahttp.service');
            }
          return '<span class="text-success">'.__('app.pmta_server_started').'</span>';
          break;
          case "pmta-reset-counters":
            $ssh->exec('pmta reset counters');
          return '<span class="text-success">'.__('app.pmta_reset_counters').'</span>';
          break;
          case "pmta-flush-queue":
            $ssh->exec('pmta delete --queue=*/*');
          return '<span class="text-success">'.__('app.pmta_reset_counters').'</span>';
          break;
          case "pmta-reset":
            $ssh->exec('pmta reset counters');
            $ssh->exec('pmta delete --queue=*/*');
            if($pmta->server_os == 'ubuntu24.04' || $pmta->server_os == 'ubuntu22.04' || $pmta->server_os == 'ubuntu20.04' || $pmta->server_os == 'other') {
              $ssh->exec('/etc/init.d/pmta restart');
              sleep(2);
              $ssh->exec('/etc/init.d/pmtahttp restart');
            } elseif($pmta->server_os == 'centos8.x' || $pmta->server_os == 'centos9.x' || $pmta->server_os == 'centos10.x' || $pmta->server_os == 'almalinux9.x') {
              $ssh->exec('systemctl restart pmta.service');
              sleep(2);
              $ssh->exec('systemctl restart pmtahttp.service');
            }
          return '<span class="text-success">'.__('app.pmta_reset_server').'</span>';
          break;
      }
    }

    // ip per domains
    public function ipPerDomain($ips, $domains) {
    return ceil(count($ips) / count($domains));
    }
}
