<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Requests\CreateReplyServerRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use App\Models\ReplyServer;
use App\Http\Helper\Helper;
use Illuminate\View\View;
use Webklex\PHPIMAP\ClientManager;
use Auth;
use Crypt;

class ReplyServerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        Helper::checkPermissions('reply_servers');
        $page = "reply_servers";
        $page_title = __('app.reply_servers');
        $breadcrumbs = [
            __('app.reply_servers') => route('reply_servers.index'),
            __('app.manage') => '#'
        ];
        $groups = \App\Models\Group::groups(config('custom.group_reply_server'));
        return view('reply_servers.index',compact('page', 'page_title', 'breadcrumbs', 'groups'));
    }

    /**
    * Retrun JSON datatable data
    */
    public function getReplyServers(Request $request)
    {
        $result = ReplyServer::leftJoin('groups', 'groups.id', '=', 'reply_servers.group_id')
          ->select('reply_servers.id', 'reply_servers.email', 'reply_servers.host', 'reply_servers.method', 'reply_servers.is_active', 'reply_servers.is_validate', 'reply_servers.created_at', 'groups.id as group_id', 'groups.name as group_name')
          ->app();

        $columns = ['reply_servers.id', 'reply_servers.id', 'reply_servers.email', 'groups.name', 'reply_servers.host', 'reply_servers.method', 'reply_servers.is_active', 'reply_servers.is_validate', 'reply_servers.created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $reply_servers = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($reply_servers as $reply_server) {
          $checkbox = "<input class=\"form-check-input\" type=\"checkbox\" value=\"{$reply_server->id}\">";
          $actions = '<div class="btn-group">
            <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').' <span class="caret"></span></button><ul class="dropdown-menu" role="menu">';
          $actions.= '<li><a class="dropdown-item" href="'.route('reply_servers.edit', [$reply_server->id]).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';          
          $actions .= '<li><a class="dropdown-item" href="'.route('reply_server.copy', [$reply_server->id]).'"><i class="bi bi-copy"></i> '.__('app.copy').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="move(\''.$reply_server->id.'\', \''.htmlentities($reply_server->email).'\')"><i class="bi bi-arrows"></i> '.__('app.move').'</a></li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$reply_server->id.'\', \''.route('reply_servers.destroy', [$reply_server->id]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul>
            </div>';

            $status = $reply_server->is_active 
            ? "<span class='badge bg-success-gradient'>".__('app.yes')."</small>"
            : "<span class='badge bg-danger-gradient'>".__('app.no')."</small>";

          $validate = $reply_server->is_validate 
            ? "<span class='badge bg-success-gradient'>".__('app.yes')."</small>"
            : "<span class='badge bg-danger-gradient'>".__('app.no')."</small>";

          $group_name = $reply_server->group_id && $reply_server->group_name ? "<span id='{$reply_server->group_id}'>$reply_server->group_name</span>" : "<span id='0'>-</span>";

          $data['data'][] = [
            "DT_RowId" => "row_{$reply_server->id}",
            $checkbox,
            $reply_server->id,
            $reply_server->email,
            $group_name,
            $reply_server->host,
            $reply_server->method,
            $status,
            $validate,
            Helper::datetimeDisplay($reply_server->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        Helper::checkPermissions('reply_servers');
        $page = "reply_servers";
        $page_title = __('app.add_new_reply_server');
        $breadcrumbs = [
            __('app.reply_servers') => route('reply_servers.index'),
            __('app.add_new_reply_server') => '#'
        ];
        $groups = \App\Models\Group::groups(config('custom.group_reply_server'));
        return view('reply_servers.create',compact('page', 'page_title', 'breadcrumbs', 'groups'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CreateReplyServerRequest $request): RedirectResponse
    {
        $data = $this->_replyServerData($request);
        $reply_server = ReplyServer::create($data);

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.reply_server') . " ({$request->email}) ". __('app.log_create'));

        if(isset($data['save_continue']))
            return redirect()->route('reply_servers.create')
            ->with('success', ($request->email . ' ' . __('app.created_successfully')));
        elseif(isset($data['save_validate']))
            return redirect()->route('reply_servers.edit', $reply_server->id)
            ->with('success', ($request->email . ' ' . __('app.created_successfully')));
        else
            return redirect()->route('reply_servers.index')
            ->with('success', ($request->email . ' ' . __('app.created_successfully')));
    }


    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ReplyServer $reply_server): View
    {
        Helper::checkPermissions('reply_servers');
        $page = "reply_servers";
        $page_title = __('app.edit_reply_server');
        $breadcrumbs = [
            __('app.reply_servers') => route('reply_servers.index'),
            __('app.edit_reply_server') => '#'
        ];
        $groups = \App\Models\Group::groups(config('custom.group_reply_server'));
        return view('reply_servers.edit',compact('page', 'page_title', 'breadcrumbs', 'reply_server', 'groups'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(CreateReplyServerRequest $request, ReplyServer $reply_server): RedirectResponse
    {
        $data = $this->_replyServerData($request);
        $reply_server->fill($data)->save();

        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.reply_server') . " ({$reply_server->email}) ". __('app.log_update'));

        return redirect()->back()
            ->with('success', ($request->email . ' ' . __('app.updated_successfully')));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $names = json_encode(array_values(ReplyServer::whereIn('id', $ids)->pluck('email')->toArray()));
          $destroy = ReplyServer::whereIn('id', $ids)->delete();
        } else {
          $names = ReplyServer::whereId($id)->value('email');
          $destroy = ReplyServer::destroy($id);
        }

        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.reply_server') . " ({$names}) ". __('app.log_delete')); 

        $data = [
                'success' => true,
                'message' => __('app.deleted_successfully')
            ];
        return response()->json($data, 200); 
    }

    /**
    * Retrun data for store or update
    */
    private function _replyServerData($request)
    {
        $input = $request->except('_token');
        $input['password'] = !empty($request->password) ? Crypt::encrypt($request->password) : null;
        $input['is_validate'] = false;
        $input['app_id'] = Auth::user()->app_id;
        $input['user_id'] = Auth::user()->id;
        return $input;
    }

    /**
    * Return after validate for IMAP or POP3
    */
    public function validateImap(Request $request)
    {
        $validate_cert = $request->validate_cert == 'Yes' ? true : false;

        // Create a new ClientManager instance
        $clientManager = new ClientManager();

        // Set up the client configuration
        $client = $clientManager->make([
            'host'          => $request->host,
            'port'          => $request->port,
            'encryption'    => $request->encryption,
            'validate_cert' => $validate_cert,
            'username'      => $request->username,
            'password'      => $request->password,
            'protocol'      => $request->method
        ]);

        try {
            // Connect to the IMAP server
            $client->connect();

            // update is_validate value
            $reply_server = ReplyServer::findOrFail($request->id);
            $reply_server->update([
                'is_validate' => true,
            ]);

            return '<span class="text-success">'.__('app.successfully_connected').'</span>';
        } catch (\Exception $e) {
            // Handle connection errors
            $msg = $e->getMessage();
            return "<span class='text-danger'>{$msg}</span>";
        }
    }

    /**
     * Retrun index after copy reply server
    */
    public function copy(int $id): RedirectResponse
    {
        Helper::checkPermissions('bounce_servers');
        $reply_server = ReplyServer::whereId($id)->app()->first();
        $reply_server->email = $reply_server->email .' - copy ';
        $reply_server->is_active = false;
        $copied_fbl = $reply_server->replicate();
        $copied_fbl->save();

        activity('copy')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.reply_server') . " ({$reply_server->email}) ". __('app.log_copy'));

        return redirect()->route('reply_servers.index')
                ->with('success', ($reply_server->email . ' ' . __('app.copied_successfully')));
    }

    /**
    * Retrun HTML data of a template
    */
    public function getReplyHTML(int $id)
    {
        $detail = \App\Models\Replies::where('schedule_campaign_stat_log_id', $id)->app()->value('detail');
        $content = json_decode($detail)->full_detail;
        return view('broadcasts.view')->with('content_html', $content);
    }
}
