<?php
    
namespace App\Http\Controllers;


use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\Rule;
use Illuminate\View\View;
use App\Models\User;
use App\Http\Helper\Helper;
use DB;
use Auth;
    
class RoleController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    function __construct()
    {
    }
    
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request): View
    {
        if($request->t == 2) {
            Helper::checkPermissions('clients_roles');
            $page = 'clients_roles';
            $breadcrumbs = [
                __('app.client_management') => '#',
                __('app.roles') => route('roles.index'),
            ];
        } else {
            Helper::checkPermissions('roles');
            $page = 'users_roles'; 
            $breadcrumbs = [
                __('app.user_management') => '#',
                __('app.roles') => route('roles.index'),
            ];
        }
        
        $page_title = __('app.roles');
        return view('roles.index',compact('page', 'page_title', 'breadcrumbs'));
    }

     /**
   * Retrun JSON datatable data
  */
  public function getRoles(Request $request): Void
  {
    $type = $request->t == 2 ? 'client' : 'user';
    $result = Role::select('id', 'display_name', 'created_at')
      ->where('app_id', Auth::user()->app_id)
      ->where('type', $type)
      ->where('id', '<>', 1); // avoid to get superadmin as came with installation
    
    $columns = ['id', 'id', 'display_name', 'id', 'created_at'];

    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $roles = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($roles as $role) {
      $checkbox = "<input type=\"checkbox\" value=\"{$role->id}\" class=\"form-check-input\">";


      $actions = '<div class="btn-group">';
      $actions .= '<button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').'</button>';
      $actions .= '<ul class="dropdown-menu" style="">';
      $actions .= '<li><a class="dropdown-item" href="'.route('roles.edit', $role->id).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
      $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$role->id.'\', \''.route('roles.destroy', $role->id).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
      $actions .= '</ul>';

      $permissions = "<a href='".route('roles.edit', $role->id)."' class='link-info link-offset-2 link-underline-opacity-25 link-underline-opacity-100-hover text-decoration-none'>".__('app.manage')."</a>";

      $data['data'][] = [
        "DT_RowId" => "row_{$role->id}",
        $checkbox,
        $role->id,
        $role->display_name,        
        $permissions,
        Helper::datetimeDisplay($role->created_at),
        $actions
      ];
    }
    echo json_encode($data);
  }
    
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request): View
    {
        if($request->t == 2) {
            Helper::checkPermissions('clients_roles');
            $page = 'clients_roles';
            $breadcrumbs = [
                __('app.client_management') => '#',
                __('app.roles') => route('roles.index', 't=2'),
                __('app.add_new_role') => '#'
            ];
        } else {
            Helper::checkPermissions('roles');
            $page = 'users_roles'; 
            $breadcrumbs = [
                __('app.user_management') => '#',
                __('app.roles') => route('roles.index', 't=1'),
                __('app.add_new_role') => '#'
            ];
        }
        $page_title = __('app.add_new_role');
        
        $permissions = Helper::systemPermissions();
        return view('roles.create',compact('permissions', 'page', 'page_title', 'breadcrumbs'));
    }
    
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request): RedirectResponse
    {
        $display_name = $request->input('name');
        $name = $display_name.'_'.uniqid();

        $type = $request->type == 2 ? 'client' : 'user';
        
        $role = Role::create([
            'name' => $name,
            'display_name' => $display_name,
            'type' => $type,
            'app_id' => Auth::user()->app_id
        ]);

        $role->syncPermissions(null); // first remove all permissions
        if(!empty($request->input('roles_permissions'))) {
          foreach($request->input('roles_permissions') as $roles_permission) {
            $permission = Permission::firstOrCreate(['name' => $roles_permission]);
            $permission->assignRole($role);
          }
        }

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.role') . " ({$request->name}) ". __('app.log_create')); // log
    
        return redirect()->route('roles.index', ['t' => $request->type])
            ->with('success',__('app.role') .' '. __('app.log_create'));
    }
    
    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Role $role): View
    {
        if($role->type == 'client') {
            Helper::checkPermissions('clients_roles');
            $page = 'clients_roles';
            $breadcrumbs = [
                __('app.client_management') => '#',
                __('app.roles') => route('roles.index', 't=2'),
                __('app.edit_role') => '#'
            ];
        } else {
            Helper::checkPermissions('roles');
            $page = 'users_roles'; 
            $breadcrumbs = [
                __('app.user_management') => '#',
                __('app.roles') => route('roles.index', 't=1'),
                __('app.edit_role') => '#'
            ];
        }

        $page_title = __('app.edit_role');
        $role = Role::whereId($role->id)->whereAppId(Auth::user()->app_id)->where('id', '<>', 1)->first();
        $role_permissions = $role->permissions->pluck('name', 'id'); // get all permissions of a role
        $permissions = Helper::systemPermissions();

        return view('roles.edit',compact('role','permissions', 'role_permissions', 'page', 'page_title', 'breadcrumbs'));
    }
    
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Role $role): RedirectResponse
    {
        // Ensure the role belongs to the current user's app_id
        $role = Role::whereId($role->id)->whereAppId(Auth::user()->app_id)->firstOrFail();
        
        $display_name = $request->input('name');
        $name = $display_name.'_'.uniqid();

        $role->name = $name;
        $role->display_name = $display_name;
        $role->save();

        $role->syncPermissions(null); // first remove all permissions
        if(!empty($request->input('roles_permissions'))) {
          foreach($request->input('roles_permissions') as $roles_permission) {
            $permission = Permission::firstOrCreate(['name' => $roles_permission]);
            $permission->assignRole($role);
          }
        }

        // save log
        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.role') . " ({$request->name}) ". __('app.log_update')); // log

        $t = $role->type == 'user' ? 1 : 2;
        return redirect()->route('roles.index', ['t' => $t])
            ->with('success',__('app.role') .' '. __('app.updated_successfully'));
    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $names = json_encode(array_values(Role::whereIn('id', $ids)
              ->where('app_id', Auth::user()->app_id)
              ->pluck('name')->toArray()));
          $destroy = Role::whereIn('id', $ids)
              ->where('app_id', Auth::user()->app_id)
              ->delete();
        } else {
          $names = Role::whereId($id)
              ->where('app_id', Auth::user()->app_id)
              ->value('name');
          $destroy = Role::whereId($id)
              ->where('app_id', Auth::user()->app_id)
              ->delete();
        }

        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.role') . " ({$names}) ". __('app.log_delete'));
        
        $data = [
                'success' => true,
                'message' => __('app.deleted_successfully')
            ];
        return response()->json($data, 200); 
    }
}
