<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use App\Http\Helper\Helper;
use App\Models\ScheduleCampaignStat;
use Auth;
use ZipArchive;

class ScheduleCampaignStatController extends Controller
{
    /**
    * Retrun index view
    */
    public function index()
    {
        Helper::checkPermissions('analatics_campaigns');
        $page = 'analatics_campaigns';
        $page_title = __('app.analatics_campaigns');
        $breadcrumbs = [
            __('app.analatics') => '#',
            __('app.campaigns') => route('stats.campaigns.index')
        ];
        return view('analatics.campaigns.index',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Retrun JSON datatable data
    */
    public function getStatsCampaigns(Request $request) {
        $result = ScheduleCampaignStat::whereAppId(Auth::user()->app_id);

        $columns = ['schedule_campaign_name', 'start_datetime', 'total', 'scheduled', 'sent', 'id', 'id', 'id', 'created_at', 'id', 'id'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $stats = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($stats as $stat) {
          $scheduled = '<a class="link-info" href="javascript:;" onclick="viewModal(\'modal\', \''.route('scheduled.detail.campaign', ['id' => $stat->schedule_campaign_id]).'\')">'.$stat->scheduled.'</a>';
          $progress = "( {$stat->sent} / {$stat->scheduled} ) " . Helper::getPercentage($stat->sent, $stat->scheduled);

          $opens_clicks = json_decode($stat->opens_clicks);

          $opens = $this->opensAllUniqueWithPercentage($stat->id, $stat->sent, $opens_clicks->total_opens, $opens_clicks->unique_opens);
          $clicks = $this->clicksAllUniqueWithPercentage($stat->id, $stat->sent, $opens_clicks->total_clicks, $opens_clicks->unique_clicks);

          $name = '<a class="link-info" href="javascript:;" onclick="viewModal(\'modal\', \''.route('detail.stat.campaign', [$stat->id, 'summary', 'analatics.campaigns.summary_popup']).'\')">'.$stat->schedule_campaign_name.'</a>';
          $detail = '<a class="link-info" href="'.route('detail.stat.campaign', ['id' => $stat->id]).'"><i class="bi bi-bar-chart"></i></a>';

          if(Helper::checkPermissions('analatics_campaigns_delete', false)) {
            $action = '<a class="text-danger" href="javascript:;" onclick="destroy(\''.$stat->id.'\', \''.route('stat.campaign.destroy', ['id' => $stat->id, 'name' => $stat->schedule_campaign_name]).'\')"><i class="bi bi-trash2"></i></a>';
          } else {
            $action = '---';
          }

          $replies = ScheduleCampaignStat::statLogReplies($stat->id)->get()->count();

          $data['data'][] = [
            "DT_RowId" => "row_{$stat->id}",
            $name,
            !empty($stat->start_datetime) ? Helper::datetimeDisplay($stat->start_datetime) : '---',
            $stat->total,
            $scheduled,
            $progress,
            $opens,
            $clicks,
            $replies,
            Helper::datetimeDisplay($stat->created_at),
            $detail,
            $action
          ];
        }
        echo json_encode($data);
    }

    /**
    * Retrurn all opens and unique opens with percentage
    */
    public function opensAllUniqueWithPercentage($stat_id, $stat_sent, $total_opens, $unique_opens)
    {
        return $opens = "( <a title='".__('app.unique')."'>{$unique_opens}</a> / <a title='".__('app.all')."'>{$total_opens}</a> ) " . Helper::getPercentage($unique_opens, $stat_sent);
    }

    /**
    * Retrurn all clicks and unique clicks with percentage
    */
    public function clicksAllUniqueWithPercentage($stat_id, $stat_sent, $total_clicks, $unique_clicks)
    {
        return $clicks = "( <a title='".__('app.unique')."'>{$unique_clicks}</a> / <a title='".__('app.all')."'>{$total_clicks}</a> ) " . Helper::getPercentage($unique_clicks, $stat_sent);
    }

    /**
    * Retrurn all opens and unique opens with percentage
    */
    public function opensAllUniqueWithPercentageLive($stat_id, $stat_sent)
    {
        $opens_all = ScheduleCampaignStat::statLogOpens($stat_id)->get()->count();
        $opens_unique = ScheduleCampaignStat::statLogOpens($stat_id, 'unique')->get()->count();
        return $opens = "( <a title='".__('app.unique')."'>{$opens_unique}</a> / <a title='".__('app.all')."'>{$opens_all}</a> ) " . Helper::getPercentage($opens_unique, $stat_sent);
    }

    /**
    * Retrurn all clicks and unique clicks with percentage
    */
    public function clicksAllUniqueWithPercentageLive($stat_id, $stat_sent)
    {
        $clicks_all = ScheduleCampaignStat::statLogClicks($stat_id)->get()->count();
        $clicks_unique = ScheduleCampaignStat::statLogClicks($stat_id, 'unique')->get()->count();
        return $clicks = "( <a title='".__('app.unique')."'>{$clicks_unique}</a> / <a title='".__('app.all')."'>{$clicks_all}</a> ) " . Helper::getPercentage($clicks_unique, $stat_sent);
    }

    /**
    * Retrurn detail view for a scheduled campaign
    */
    public function getDetailStat($id, $type=null, $view='analatics.campaigns.summary')
    {
        Helper::checkPermissions('analatics_campaigns');
        $schedule_stat = ScheduleCampaignStat::whereId($id)->app()->first();
        if(empty($type)) {
            $page = 'analatics_campaigns';
            $page_title = __('app.analatics').' - '.__('app.campaigns').' - '. __('app.detail').' - '.($schedule_stat->schedule_campaign_name ?? '---');
            $breadcrumbs = [
                __('app.analatics') => '#',
                __('app.campaigns') => route('stats.campaigns.index'),
                __('app.detail') => '#'
            ];
          return view('analatics.campaigns.detail')->with(compact('page', 'page_title', 'breadcrumbs', 'schedule_stat'));
        } else {
          switch($type) {
            case 'opens':
              return view('analatics.campaigns.opens')->with('stat_id', $id);
            break;
            case 'clicks':
              return view('analatics.campaigns.clicks')->with('stat_id', $id);
            break;
            case 'unsubscribed':
              return view('analatics.campaigns.unsubscribed')->with('stat_id', $id);
            break;
            case 'bounces':
              return view('analatics.campaigns.bounces')->with('stat_id', $id);
            break;
            case 'spam':
              return view('analatics.campaigns.spam')->with('stat_id', $id);
            break;
            case 'replies':
              return view('analatics.campaigns.replies')->with('stat_id', $id);
            break;
            case 'logs':
              return view('analatics.campaigns.logs')->with('stat_id', $id);
            break;
            default:
              $data = $this->getSummary($schedule_stat, $type, $view);
          }
          return $data;
        }
    }

    /**
    * Retrurn summary for a scheduled campaign
    */
    public function getSummary($schedule_stat, $type=null, $view='analatics.campaigns.summary')
    {
    if($type == 'summary_opens') {
      return $opens = $this->opensAllUniqueWithPercentageLive($schedule_stat->id, $schedule_stat->sent);
    } elseif($type == 'summary_clicks') {
      return $clicks = $this->clicksAllUniqueWithPercentageLive($schedule_stat->id, $schedule_stat->sent);
    } elseif($type == 'summary_unsubscribed') {
      return $unsubscribed = ScheduleCampaignStat::statLogClicks($schedule_stat->id)->where('link', 'like', '%/unsub/%')->whereUnsubConfirmed(true)->get()->count();
    } elseif($type == 'summary_bounces') {
      return $bounces = ScheduleCampaignStat::statLogBounces($schedule_stat->id)->get()->count();
    } elseif($type == 'summary_spam') {
      return $spam = ScheduleCampaignStat::statLogSpam($schedule_stat->id)->get()->count();
    } elseif($type == 'summary_replies') {
      return $replies = ScheduleCampaignStat::statLogReplies($schedule_stat->id)->get()->count();
    } elseif($type == 'summary_sent_lists') {
      $lists = \App\Models\ScheduleCampaignStat::statLogData($schedule_stat->id, 'list')->pluck('total', 'list')->toArray();
      $data = '';
      foreach($lists as $list => $total){
        $data .= "{$list} <strong>({$total})</strong>";
      }
      return $data;
    } elseif($type == 'summary_sent_sending_servers') {
      $sending_servers = \App\Models\ScheduleCampaignStat::statLogData($schedule_stat->id, 'sending_server')->pluck('total', 'sending_server')->toArray();
      $data = '';
      foreach($sending_servers as $sending_server => $total) {
        $data .= "{$sending_server} <strong>({$total})</strong>";
      }
      return $data;
    }

    return view($view)->with(compact('schedule_stat'));

    }

    /**
    * Retrurn opens for a scheduled campaign
    */
    public function getOpens(Request $request)
    {
    $result = ScheduleCampaignStat::statLogOpens($request->stat_id);

    $columns = ['email', 'list', 'sending_server', 'message_id', 'ip', 'country', 'zipcode', 'city', 'schedule_campaign_stat_log_opens.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $message_id = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->message_id.'\'}); } else { swal({title: \'\', text:\''. $stat->message_id.'\'}); }">'.substr($stat->message_id, 0, 20).'...</a>';
      $country = empty($stat->country) ? '---' : $stat->country;
      $city = empty($stat->city) ? '---' : $stat->city;
      $zipcode = empty($stat->zipcode) ? '---' : $stat->zipcode;
      $data['data'][] = [
        $stat->email,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $stat->ip,
        $country,
        $city,
        $zipcode,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
    }


    /**
    * Retrurn clicks for a scheduled campaign
    */
    public function getClicks(Request $request)
    {
    $result = ScheduleCampaignStat::statLogClicks($request->stat_id);

    $columns = ['email', 'link', 'list', 'sending_server', 'message_id', 'ip', 'country', 'zipcode', 'city', 'schedule_campaign_stat_log_clicks.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $country = empty($stat->country) ? '---' : $stat->country;
      $city = empty($stat->city) ? '---' : $stat->city;
      $zipcode = empty($stat->zipcode) ? '---' : $stat->zipcode;
      $message_id = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->message_id.'\'}); } else { swal({title: \'\', text:\''. $stat->message_id.'\'}); }">'.substr($stat->message_id, 0, 20).'...</a>';

      $link = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->link.'\'}); } else { swal({title: \'\', text:\''. $stat->link.'\'}); }">'.substr($stat->link, 0, 20).'...</a>';
      $data['data'][] = [
        $stat->email,
        $link,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $stat->ip,
        $country,
        $city,
        $zipcode,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
    }

    /**
    * Retrurn unsubscribed for a scheduled campaign
    */
    public function getUnsubscribed(Request $request)
    {
    $result = ScheduleCampaignStat::statLogClicks($request->stat_id)
      ->where('link', 'like', '%/unsub/%');

    $columns = ['email', 'link', 'list', 'sending_server', 'message_id', 'ip', 'country', 'zipcode', 'city', 'schedule_campaign_stat_log_clicks.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $email = $stat->email;

      if($stat->unsub_confirmed) {
        $email = "<span class='text-danger' data-bs-toggle='tooltip' data-bs-placement='top' title='confirmed' style='cursor: pointer;'>{$stat->email}</span>";
      }
      $country = empty($stat->country) ? '---' : $stat->country;
      $city = empty($stat->city) ? '---' : $stat->city;
      $zipcode = empty($stat->zipcode) ? '---' : $stat->zipcode;
      $link = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->link.'\'}); } else { swal({title: \'\', text:\''. $stat->link.'\'}); }">'.substr($stat->link, 0, 20).'...</a>';
      $message_id = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->message_id.'\'}); } else { swal({title: \'\', text:\''. $stat->message_id.'\'}); }">'.substr($stat->message_id, 0, 20).'...</a>';
      $data['data'][] = [
        $email,
        $link,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $stat->ip,
        $country,
        $city,
        $zipcode,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
    }

    /**
    * Retrurn bounces for a scheduled campaign
    */
    public function getBounces(Request $request)
    {
    $result = ScheduleCampaignStat::statLogBounces($request->stat_id);

    $columns = ['global_bounces.email', 'list', 'sending_server', 'message_id', 'type', 'code', 'detail', 'global_bounces.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $bounce_detail = json_decode($stat->detail);
      $detail = '<a href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire(\''.str_replace("'", "",$bounce_detail->full_detail).'\'); } else { swal(\''.str_replace("'", "",$bounce_detail->full_detail).'\'); }"><i class="fa fa-info-circle"></i></a> '. $bounce_detail->short_detail;
      $message_id = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->message_id.'\'}); } else { swal({title: \'\', text:\''. $stat->message_id.'\'}); }">'.substr($stat->message_id, 0, 20).'...</a>';

      $detail = "<pre>".wordwrap($detail, 50, "\n")."</pre>";
      $data['data'][] = [
        $stat->email,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $stat->type,
        $stat->code,
        $detail,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
    }

    /**
    * Retrurn spam for a scheduled campaign
    */
    public function getSpam(Request $request)
    {
    $result = ScheduleCampaignStat::statLogSpam($request->stat_id);

    $columns = ['global_spams.email', 'list', 'sending_server', 'message_id', 'detail', 'global_spams.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $message_id = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->message_id.'\'}); } else { swal({title: \'\', text:\''. $stat->message_id.'\'}); }">'.substr($stat->message_id, 0, 20).'...</a>';
      $data['data'][] = [
        $stat->email,
        $stat->list,
        $stat->sending_server,
        $message_id,
        json_decode($stat->detail)->full_detail,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
    }

    /**
    * Retrurn spam for a scheduled campaign
    */
    public function getReplies(Request $request)
    {
    $result = ScheduleCampaignStat::statLogReplies($request->stat_id);

    $columns = ['replies.email', 'list', 'sending_server', 'message_id', 'detail', 'replies.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $message_id = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->message_id.'\'}); } else { swal({title: \'\', text:\''. $stat->message_id.'\'}); }">'.substr($stat->message_id, 0, 20).'...</a>';
      $reply_html = '<a class="link-info" target="_blank" href="'.route('get.reply_html', ['id' => $stat->schedule_campaign_stat_log_id]).'">'.__('app.view').'</a>';

      $data['data'][] = [
        $stat->email,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $reply_html,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
    }

    /**
    * Retrurn all info for a scheduled campaign
    */
    public function getLogs(Request $request)
    {
    $result = \App\Models\ScheduleCampaignStatLog::whereScheduleCampaignStatId($request->stat_id);

    $columns = ['email', 'list', 'sending_server', 'message_id', 'status', 'created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      if($stat->status == 'Sent')
        $status = "<span class='badge bg-primary-gradient'>".$stat->status."</span>";
      elseif($stat->status == 'Opened')
        $status = "<span class='badge bg-info-gradient'>".$stat->status."</span>";
      elseif($stat->status == 'Clicked')
        $status = "<span class='badge bg-success-gradient'>".$stat->status."</span>";
      elseif($stat->status == 'Unsubscribed')
        $status = "<span class='badge bg-primary3-gradient'>".$stat->status."</span>";
      elseif($stat->status == 'Bounced')
        $status = "<span class='badge bg-danger-gradient'>".$stat->status."</span>";
      elseif($stat->status == 'Spammed')
        $status = "<span class='badge bg-primary2-gradient'>".$stat->status."</span>";
      else
        $status = "<span class='badge bg-danger-gradient'>".$stat->status."</span>";

      $message_id = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->message_id.'\'}); } else { swal({title: \'\', text:\''. $stat->message_id.'\'}); }">'.substr($stat->message_id, 0, 20).'...</a>';
    
      $data['data'][] = [
        $stat->email,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $status,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
    }

    /**
    * Retrurn view for scheduled detail
    */
    public function getScheduledDetail($id)
    {
    HHelper::checkPermissions('stats_campaigns'); // check user permission
    $schedule_stat = ScheduleCampaignStat::whereId($id)->app()->first();
    $name = $schedule_stat->name;
    $scheduled_detail = json_decode($schedule_stat->scheduled_detail);
    return view('schedule_campaigns.scheduled_detail')->with(compact('name', 'scheduled_detail'));
    }

    /**
    * Retrurn JSON for a sent campaign 
    */
    public function getSentData($id, $type = null)
    {
    if($type == 'country') {
      $data = ScheduleCampaignStat::getUniqueCountries($id)->get()->pluck('cnt', 'country_code')->toArray();
    } else {
      $opens_unique = ScheduleCampaignStat::statLogOpens($id, 'unique')->get()->count();
      $clicks_unique = ScheduleCampaignStat::statLogClicks($id, 'unique')->get()->count();
      $unsubscribed = ScheduleCampaignStat::statLogClicks($id)->where('link', 'like', '%/unsub/%')->whereUnsubConfirmed(true)->get()->count();
      $bounces = ScheduleCampaignStat::statLogBounces($id)->get()->count();
      $spam = ScheduleCampaignStat::statLogSpam($id)->get()->count();
      $replies = ScheduleCampaignStat::statLogReplies($id)->get()->count();
      $dataset = [
        'data' => [$opens_unique, $clicks_unique, $replies, $unsubscribed, $bounces, $spam],
        'backgroundColor' => ['#0EA5E8', '#21CE9E', '#202947', '#FF851B', '#DD4B39', '#FFC0CB'],
      ];
      $data = [
        'labels' => [
          "$opens_unique ".__('app.opens'),
          "$clicks_unique ".__('app.clicks'),
          "$replies ".__('app.replies'),
          "$unsubscribed ".__('app.unsubscribed'),
          "$bounces ".__('app.bounces'),
          "$spam ".__('app.spams')
        ],
        'datasets' => [$dataset],
      ];
    }
    return json_encode($data);
    }

    /**
    * Export campaign stat detail
    */
    public function export($id)
    {
    $schedule_stat = ScheduleCampaignStat::whereId($id)->app()->first();

    $path_export_stat = str_replace('[user-id]', Auth::user()->app_id, config('custom.path_export_stat_campaign'));
    Helper::dirCreateOrExist($path_export_stat); // create dir if not exist

    // Summary PDF File
    $file_summary = $path_export_stat.$id.'-summary.pdf';

    // Get Summary
    $html = $this->getSummary($schedule_stat);
    \PDF::loadHTML($html)->setPaper('a4')->setWarnings(false)->save($file_summary);

    // Opens CSV File
    $file_opens = $path_export_stat.$id.'-opens.csv';
    $this->exportOpens($id, $file_opens);

    // Clicks CSV File
    $file_clicks = $path_export_stat.$id.'-clicks.csv';
    $this->exportClicks($id, $file_clicks);

    // Unsbuscribed CSV File
    $file_unsubs = $path_export_stat.$id.'-unsubscribed.csv';
    $this->exportUnsubscribed($id, $file_unsubs);

    // Bounces CSV File
    $file_bounces = $path_export_stat.$id.'-bounces.csv';
    $this->exportBounces($id, $file_bounces);

    // Spam CSV File
    $file_spams = $path_export_stat.$id.'-spams.csv';
    $this->exportSpam($id, $file_spams);

    // Replies CSV File
    $file_replies = $path_export_stat.$id.'-replies.csv';
    $this->exportReplies($id, $file_replies);

    // Logs CSV File
    $file_logs = $path_export_stat.$id.'-logs.csv';
    $this->exportLogs($id, $file_logs);


    // Make Zip
    $zip_file = $path_export_stat.$id.'.zip';
    Helper::createZipFile($zip_file, $file_summary, 'summary.pdf');
    Helper::createZipFile($zip_file, $file_opens, 'opens.csv');
    Helper::createZipFile($zip_file, $file_clicks, 'clicks.csv');
    Helper::createZipFile($zip_file, $file_unsubs, 'unsubscribed.csv');
    Helper::createZipFile($zip_file, $file_bounces, 'bounces.csv');
    Helper::createZipFile($zip_file, $file_spams, 'spams.csv');
    Helper::createZipFile($zip_file, $file_logs, 'logs.csv');
    Helper::createZipFile($zip_file, $file_replies, 'replies.csv');

    // Delete other files after created zip
    unlink($file_summary);
    unlink($file_opens);
    unlink($file_clicks);
    unlink($file_unsubs);
    unlink($file_bounces);
    unlink($file_spams);
    unlink($file_logs);
    unlink($file_replies);

    }


    function aiAnalysis($id)
    {
      $schedule_stat = ScheduleCampaignStat::whereId($id)->app()->first();

      $subject = $schedule_stat->email_subject;

      $schedule = \App\Models\ScheduleCampaign::whereId($schedule_stat->schedule_campaign_id)->app()->first();
      $sending_content = $schedule->sending_content;

      if($sending_content == 'text')
          $content = $schedule->content_text;
      else
          $content = $schedule->content;

      $sent = $schedule_stat->scheduled;

      $opens_clicks = json_decode($schedule_stat->opens_clicks);
      $opens = $opens_clicks->unique_opens;
      $clicks = $opens_clicks->unique_clicks;

      $unsubscribed = ScheduleCampaignStat::statLogClicks($schedule_stat->id)->where('link', 'like', '%/unsub/%')->whereUnsubConfirmed(true)->get()->count();
      $bounces = ScheduleCampaignStat::statLogBounces($schedule_stat->id)->get()->count();
      $spam = ScheduleCampaignStat::statLogSpam($schedule_stat->id)->get()->count();

      $text = "Could you please provide your analysis and recommendations to help improve the results based on the following campaign statistics?:\n
      Email Subject: {$subject}\n
      Email Content: {$content}\n
      Total Sent: {$sent}\n
      Opens: {$opens}\n
      Clicks: {$clicks}\n
      Unsubscribed: {$unsubscribed}\n
      Bounced: {$bounces}\n
      Spams: {$spam}\n
      ";

      return $text;
    }

    /**
    * Download export campaign stat detail
    */
    public function exportDownload($id)
    {
    try{
      $path_export_stat = str_replace('[user-id]', Auth::user()->app_id, config('custom.path_export_stat_campaign'));
      $zip_file = $path_export_stat.$id.'.zip';
      return response()->download($zip_file)->deleteFileAfterSend(true);
    } catch(\Exception $e) {
      // if file not found
      return redirect(route('detail.stat.campaign', ['id' => $id]));
    }
    }

    // Export campaign opens
    public function exportOpens($id, $file)
    {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.ip'),
      __('app.country'),
      __('app.city'),
      __('app.zip_code'),
      __('app.datetime'),
    ];

    // Write file header

    $writer->insertOne($file_header);
    // Get Opens
    ScheduleCampaignStat::statLogOpens($id)
      ->chunk(1000, function ($opens)  use ($writer) {
        foreach($opens as $open) {
          $opens_data = [
            $open->email,
            $open->list,
            $open->sending_server,
            $open->message_id,
            $open->ip,
            $open->country,
            $open->city,
            $open->zipcode,
            Helper::datetimeDisplay($open->created_at),
          ];
          try {
            // Write open data
            $writer->insertOne($opens_data);
          } catch(\Exception $e) {}
        }
    });
    }

    // Export campaign clicks
    public function exportClicks($id, $file)
    {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.link'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.ip'),
      __('app.country'),
      __('app.city'),
      __('app.zip_code'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Clicks
    ScheduleCampaignStat::statLogClicks($id)
      ->chunk(1000, function ($clicks)  use ($writer) {
        foreach($clicks as $click) {
          $clicks_data = [
            $click->email,
            $click->link,
            $click->list,
            $click->sending_server,
            $click->message_id,
            $click->ip,
            $click->country,
            $click->city,
            $click->zipcode,
            Helper::datetimeDisplay($click->created_at),
          ];

          try {
            // Write click data
            $writer->insertOne($clicks_data);
          } catch(\Exception $e) {}
        }
    });
    }

    // Export campaign unsubscribed
    public function exportUnsubscribed($id, $file)
    {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.link'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.ip'),
      __('app.country'),
      __('app.city'),
      __('app.zip_code'),
      __('app.confirmed'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Clicks
    ScheduleCampaignStat::statLogClicks($id)
      ->where('link', 'like', '%/unsub/%')
      ->chunk(1000, function ($clicks)  use ($writer) {
        foreach($clicks as $click) {
          $clicks_data = [
            $click->email,
            $click->link,
            $click->list,
            $click->sending_server,
            $click->message_id,
            $click->ip,
            $click->country,
            $click->city,
            $click->zipcode,
            $click->unsub_confirmed,
            Helper::datetimeDisplay($click->created_at),
          ];

          try {
            // Write click data
            $writer->insertOne($clicks_data);
          } catch(\Exception $e) {}
        }
    });
    }

    // Export campaign bounces
    public function exportBounces($id, $file)
    {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.type'),
      __('app.code'),
      __('app.detail'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Bounces
    ScheduleCampaignStat::statLogBounces($id)
      ->chunk(1000, function ($bounces)  use ($writer) {
        foreach($bounces as $bounce) {
          $bounces_data = [
            $bounce->email,
            $bounce->list,
            $bounce->sending_server,
            $bounce->message_id,
            $bounce->type,
            $bounce->code,
            json_decode($bounce->detail)->short_detail,
            Helper::datetimeDisplay($bounce->created_at),
          ];

          try {
            // Write bounce data
            $writer->insertOne($bounces_data);
          } catch(\Exception $e) {}
        }
    });
    }

    // Export campaign spam
    public function exportSpam($id, $file)
    {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.detail'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Spam
    ScheduleCampaignStat::statLogSpam($id)
      ->chunk(1000, function ($spams)  use ($writer) {
        foreach($spams as $spam) {
          $spam_data = [
            $spam->email,
            $spam->list,
            $spam->sending_server,
            $spam->message_id,
            json_decode($spam->detail)->full_detail,
            Helper::datetimeDisplay($spam->created_at),
          ];

          try {
            // Write bounce data
            $writer->insertOne($spam_data);
          } catch(\Exception $e) {}
        }
    });
    }

    // Export campaign replies
    public function exportReplies($id, $file)
    {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.detail'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Replies
    ScheduleCampaignStat::statLogReplies($id)
      ->chunk(1000, function ($replies)  use ($writer) {
        foreach($replies as $reply) {
          $reply_data = [
            $reply->email,
            $reply->list,
            $reply->sending_server,
            $reply->message_id,
            json_decode($reply->detail)->full_detail,
            Helper::datetimeDisplay($reply->created_at),
          ];

          try {
            // Write bounce data
            $writer->insertOne($reply_data);
          } catch(\Exception $e) {}
        }
    });
    }

    // Export campaign logs
    public function exportLogs($id, $file)
    {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.latest_activity'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Logs
    \App\Models\ScheduleCampaignStatLog::whereScheduleCampaignStatId($id)
      ->chunk(1000, function ($logs)  use ($writer) {
        foreach($logs as $log) {
          $logs_data = [
            $log->email,
            $log->list,
            $log->sending_server,
            $log->message_id,
            $log->status,
            Helper::datetimeDisplay($log->created_at),
          ];

          try {
            // Write bounce data
            $writer->insertOne($logs_data);
          } catch(\Exception $e) {}
        }
    });
    }

    /**
    * Delete stat
    */
    public function destroy($id, $name)
    {
        $destroy = ScheduleCampaignStat::destroy($id);
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.stats') . " ({$name}) ". __('app.log_delete'));
        $data = [
          'success' => true,
          'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200);
    }
}
