<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Requests\CreateSendingServerRequest;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use App\Http\Helper\Helper;
use App\Models\SendingServer;
use Auth;

class SendingServerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        Helper::checkPermissions('delivery_servers'); // check user permission
        $page = 'delivery_servers';
        $page_title = __('app.delivery_servers');
        $breadcrumbs = [
            __('app.delivery_servers') => route('sending_servers.index'),
            __('app.manage') => '#'
        ];
        $groups = \App\Models\Group::groups(config('custom.group_sending_servers'));
        return view('sending_servers.index')->with(compact('page', 'page_title', 'breadcrumbs', 'groups'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function getSendingServers(Request $request)
    {
        $result = SendingServer::join('groups', 'groups.id', '=', 'sending_servers.group_id')
          ->select('sending_servers.id', 'sending_servers.name as name', 'sending_servers.type', 'sending_servers.total_sent', 'sending_servers.status', 'sending_servers.sending_attributes', 'sending_servers.speed_attributes', 'sending_servers.notification', 'sending_servers.created_at', 'groups.id as group_id', 'groups.name as group_name', 'sending_servers.from_email')
          ->app();
        
        $columns = ['sending_servers.id', 'sending_servers.id', 'sending_servers.name', 'groups.name', 'sending_servers.type', 'sending_servers.total_sent', 'sending_servers.speed_attributes', 'sending_servers.status', 'sending_servers.created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $sending_servers = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($sending_servers as $sending_server) {
          $checkbox = "<input class=\"form-check-input\" type=\"checkbox\" value=\"{$sending_server->id}\">";
          $actions = '<div class="btn-group">
            <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').' <span class="caret"></span></button><ul class="dropdown-menu" role="menu">';
          $actions .= '<li><a class="dropdown-item"  href="'.route('sending_servers.edit', [$sending_server->id]).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="'.route('sending_server.copy', [$sending_server->id]).'"><i class="bi bi-copy"></i> '.__('app.copy').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="move(\''.$sending_server->id.'\', \''.htmlentities($sending_server->name).'\')"><i class="bi bi-arrows"></i> '.__('app.move').'</a></li>';

          if($sending_server->status == 'Active')
            $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="setStatusMany(\''.route('sending_servers.set_status').'\', \'Inactive\' ,\''.$sending_server->id.'\')"><i class="bi bi-lock"></i> '.__('app.inactive').'</a></li>';
          else
            $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="setStatusMany(\''.route('sending_servers.set_status').'\', \'Active\' ,\''.$sending_server->id.'\')"><i class="bi bi-unlock"></i> '.__('app.active').'</a></li>';

          $actions .= '<li><a class="dropdown-item"  href="javascript:;" onclick="viewModal(\'modal\', \''.route('send.email', [0, $sending_server->id, 0, $sending_server->name]).'\')"><i class="bi bi-send"></i> '.__('app.send_test_email').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="resetSentCounter(\''.$sending_server->id.'\')"><i class="bi bi-bootstrap-reboot"></i> '.__('app.reset_counter').'</a></li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$sending_server->id.'\', \''.route('sending_servers.destroy', [$sending_server->id]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul></div>';

          $group_name = "<span id='{$sending_server->group_id}'>$sending_server->group_name<span>";

          $name = $sending_server->status == 'System Inactive' || $sending_server->status == 'System Paused' ?  '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire(\''.$sending_server->notification.'\'); } else { swal(\''.$sending_server->notification.'\'); }"><i class="bi bi-info-circle-fill text-danger"></i></a> '.$sending_server->name : $sending_server->name;

          $speed_attributes = json_decode($sending_server->speed_attributes);
          $speed = $speed_attributes->speed;

          if($speed == 'limited') {
            $limit = $speed_attributes->limit . ' '.ucfirst($speed_attributes->duration);
            $speed = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire(\''.$limit.'\'); } else { swal(\''.$limit.'\'); }"><i class="bi bi-info-circle-fill"></i></a> '. ucfirst($speed);
          } elseif($speed == 'warmup') {
            $speed = __('app.warmup_plan');
          } else {
            $speed = ucfirst(__('app.unlimited'));
          }

          if($sending_server->type == 'smtp' || $sending_server->type == 'gmail' || $sending_server->type == 'microsoft') {
            $sending_attributes = json_decode($sending_server->sending_attributes);
            $smtp_data = "$sending_attributes->host ; $sending_attributes->port ; $sending_attributes->encryption ; $sending_server->from_email" ;
            $info = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire(\''.$smtp_data.'\'); } else { swal(\''.$smtp_data.'\'); }"><i class="bi bi-info-circle-fill"></i></a> '. $sending_server->id;
          } else {
            $info = $sending_server->id;
          }

          $status = $sending_server->status == 'Active' 
            ? "<span class='badge bg-success-gradient'>{$sending_server->status}</span>"
            : "<span class='badge bg-danger-gradient'>{$sending_server->status}</span>";

          $badge_class = 'primary-gradient';
          if($sending_server->type == 'smtp') $badge_class = 'purple-gradient';
          elseif($sending_server->type == 'gmail') $badge_class = 'danger-gradient';
          elseif($sending_server->type == 'microsoft') $badge_class = 'info-gradient';
          elseif($sending_server->type == 'sendgrid_api') $badge_class = 'info-gradient';
          elseif($sending_server->type == 'mailgun_api') $badge_class = 'primary2-gradient';
          elseif($sending_server->type == 'sparkpost_api') $badge_class = 'primary3-gradient';
          elseif($sending_server->type == 'amazon_ses_api') $badge_class = 'warning-gradient';
          elseif($sending_server->type == 'postal_api') $badge_class = 'dark-gradient';
          $data['data'][] = [
            "DT_RowId" => "row_{$sending_server->id}",
            $checkbox,
            $info,
            $name,
            $group_name,
            "<span class='badge bg-{$badge_class}'>".Helper::sendingServers($sending_server->type)."</span>",
            $speed,
            $sending_server->total_sent,
            $status,
            Helper::datetimeDisplay($sending_server->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(): View
    {
        if(!Helper::allowedLimit(Auth::user()->app_id, 'no_of_sending_servers', 'sending_servers')) {
            return view('errors.not_allowed')->with('module', __('app.delivery_servers'));
        }
        Helper::checkPermissions('delivery_servers');
        $page = 'delivery_servers';
        $page_title = __('app.add_new_delivery_server');
        $breadcrumbs = [
            __('app.delivery_servers') => route('sending_servers.index'),
            __('app.add_new_delivery_server') => '#'
        ];
        
        $groups = \App\Models\Group::groups(config('custom.group_sending_servers'));
        $sending_domains = \App\Models\SendingDomain::getSendingDomains();
        $tracking_domains = \App\Models\TrackingDomain::getTrackingDomains();
        $bounces = \App\Models\Bounce::getBounces();
        $warmup_plans = \App\Models\WarmupPlan::getWarmupPlans();
        return view('sending_servers.create')->with(compact('page', 'page_title', 'breadcrumbs', 'groups', 'sending_domains', 'bounces', 'tracking_domains', 'warmup_plans'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        if(!Helper::allowedLimit(Auth::user()->app_id, 'no_of_sending_servers', 'sending_servers')) {
            return view('errors.not_allowed')->with('module', __('app.delivery_servers'));
        }
        $data = $this->_sendingServerData($request);
        $sending_server = SendingServer::create($data);
        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.delivery_server') . " ({$request->name}) ". __('app.log_create'));

        if(isset($data['save_continue']))
            return redirect()->route('sending_servers.create')
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
        elseif(isset($data['save_validate']))
            return redirect()->route('sending_servers.edit', $sending_server->id)
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
        else
            return redirect()->route('sending_servers.index')
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(SendingServer $sending_server)
    {
        Helper::checkPermissions('delivery_servers');
        $page = 'delivery_servers';
        $page_title = __('app.edit_delivery_server');
        $breadcrumbs = [
            __('app.delivery_servers') => route('sending_servers.index'),
            __('app.edit_delivery_server') => '#'
        ];
        $groups = \App\Models\Group::groups(config('custom.group_sending_servers'));
        $sending_domains = \App\Models\SendingDomain::getSendingDomains();
        $bounces = \App\Models\Bounce::getBounces();
        $tracking_domains = \App\Models\TrackingDomain::getTrackingDomains();
        $warmup_plans = \App\Models\WarmupPlan::getWarmupPlans();
        return view('sending_servers.edit')->with(compact('page', 'page_title', 'breadcrumbs', 'sending_server', 'groups', 'sending_domains', 'bounces', 'tracking_domains', 'warmup_plans'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, SendingServer $sending_server)
    {
        $data = $this->_sendingServerData($request);
        // There is no need to change "System Inactive" or "System Paused" to "Inactive," as doing so will prevent it from automatically reactivating.
        $data['status'] = ($sending_server->status == 'System Inactive' || $sending_server->status == 'System Paused') 
          && $data['status'] == 'Inactive' ? $sending_server->status : $data['status'];
        $sending_server->fill($data)->save();

        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.delivery_server') . " ({$request->name}) ". __('app.log_update'));

        return redirect()->route('sending_servers.edit', $sending_server->id)
            ->with('success', ($request->name . ' ' . __('app.updated_successfully')));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
            $ids = array_values($request->ids);
            $names = json_encode(array_values(SendingServer::whereIn('id', $ids)->pluck('name')->toArray()));
            $destroy = SendingServer::whereIn('id', $ids)->delete();
        } else {
            $names = SendingServer::whereId($id)->value('name');
            $destroy = SendingServer::destroy($id);
        }
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.delivery_server') . " ({$names}) ". __('app.log_delete'));

        $data = [
          'success' => true,
          'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200);
    }

    public function setStatus(Request $request): JsonResponse
    {
        if(!empty($request->action) && is_array($request->ids)) {
            $ids = array_values($request->ids);
            $names = json_encode(array_values(SendingServer::whereIn('id', $ids)->pluck('name')->toArray()));
            $update = SendingServer::whereIn('id', $ids)->update(['status' => $request->status]);
        } else {
            $names = SendingServer::whereId($request->ids)->value('name');
            $update = SendingServer::whereId($request->ids)->update(['status' => $request->status]);
        }
        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.delivery_server') . " ({$names}) ". __('app.log_update'));

        $data = [
          'success' => true,
          'message' => __('app.updated_successfully')
        ];

        return response()->json($data, 200);
    }

    /**
    * Retrun sending server fields respectively
    */
    public function getSendingServerFields($type, $action, $id=null)
    {
        if(empty($id)) {
          $settings = \DB::table('settings')->whereId(config('custom.app_id'))->select('sending_attributes')->first();
          $data = json_decode($settings->sending_attributes);
        } else {
          $data = json_decode(SendingServer::whereId($id)->value('sending_attributes'));
        }
        return view('sending_servers.attributes')->with(compact('type', 'action', 'data'));
    }

    /**
    * Retrun data for store or update
    */
    private function _sendingServerData($request)
    {
        $input = $request->except('_token');
        $input['bounce_id'] = !empty($input['bounce_id']) && $input['bounce_id'] != 0 ? $input['bounce_id'] : null;
        $input['tracking_domain'] = !empty($input['tracking_domain']) ? $input['tracking_domain'] : null;
        $input['sending_attributes'] = $this->sendingServerAttributes($input);
        $input['from_email'] = $input['from_email_part1'] . '@' . $input['from_email_part2'];
        if($input['speed'] == 'unlimited') {
          $input['limit'] = $input['duration'] = $input['warmup_plan'] = $input['warmup_day'] = null;
        } elseif($input['speed'] == 'warmup') {
          $schedule = \App\Models\WarmupPlan::whereId($input['warmup_plan'])->value('schedule');
          $input['warmup_day'] = 0; // 0 index = day 1
          $input['limit'] = json_decode($schedule)[$input['warmup_day']];
          $input['duration'] = 'daily';
          $input['daily_sent'] = 0;
        } else {
          $input['warmup_plan'] = $input['warmup_day'] = null;
        }

        $input['speed_attributes'] = json_encode([
          'speed' => $input['speed'], 
          'limit' => ($input['limit'] ?? 100),
          'duration' => $input['duration'],
          'warmup_plan' => $input['warmup_plan'],
          'warmup_day' => $input['warmup_day']
        ]);

        $input['app_id'] =  Auth::user()->app_id;
        $input['user_id'] = Auth::user()->id;

        return $input;
    }

    /**
     * Retrun JSON data for sending server
    */
    public function sendingServerAttributes($data)
    {
        switch($data['type']) {
              case 'smtp':
              case 'gmail':
              case 'microsoft':
                if(!empty($data['password'])) {
                  $data['password'] = $data['password'] == '#########' ? $data['smtpid'] : \Crypt::encrypt($data['password']);
                }

                $attributes = [
                  'host' => $data['host'],
                  'username' => $data['username'],
                  'password' => !empty($data['password']) ? $data['password'] : null,
                  'port' => $data['port'],
                  'encryption' => $data['encryption'],
                  //'body_encoding' => $data['body_encoding']
                ];
              break;
              case 'amazon_ses_api':
                $attributes = [
                  'access_key' => $data['access_key'],
                  'secret_key' => !empty($data['secret_key']) ? \Crypt::encrypt($data['secret_key']) : null,
                  'region' => $data['region'],
                  'process_reports' => $data['process_reports'] ?? null,
                  'amazon_configuration_set' => $data['amazon_configuration_set']
                ];
              break;
              case 'mailgun_api':
                $attributes = [
                  //'domain' => $data['domain'],
                  'domain' => $data['from_email_part2'],
                  'api_key' => !empty($data['api_key']) ? \Crypt::encrypt($data['api_key']) : null,
                  'process_reports' => $data['process_reports'] ?? null
                ];
              break;
              case 'sendgrid_api':
                $attributes = [
                  'api_key' => !empty($data['api_key']) ? \Crypt::encrypt($data['api_key']) : null,
                  'process_reports' => $data['process_reports']?? null
                ];
              break;
              case 'sparkpost_api':
                $attributes = [
                  'api_key' => !empty($data['api_key']) ? \Crypt::encrypt($data['api_key']) : null,
                  'process_reports' => $data['process_reports']?? null
                ];
              break;
              case 'elastic_email_api':
                $attributes = [
                  'api_key' => !empty($data['api_key']) ? \Crypt::encrypt($data['api_key']) : null,
                  'process_reports' => $data['process_reports']?? null
                ];
              break;
              case 'postal_api':
                $attributes = [
                  'api_key' => !empty($data['api_key']) ? \Crypt::encrypt($data['api_key']) : null,
                  'postal_server_url' => $data['postal_server_url'],
                  'process_reports' => $data['process_reports']?? null
                ];
              break;
              default: 
                $attributes = null;
              break;
        }
          return json_encode($attributes);
    }

    /**
     * Retrun success/fail after send a test email
    */
    public function SendEmailTest(Request $request)
    {
        // Debug: Log the request data
        \Log::info('SendEmailTest Request Data:', $request->all());
        
        // Custom validation for multiple emails
        $request->validate([
          'email' => 'required|string|max:2500',
          'sending_server_id' => 'sometimes|integer',
          'sending_server_ids' => 'sometimes|array|min:1'
        ],
        [
          'sending_server_id.required' => 'Delivery Server is required.',
          'sending_server_ids.required' => 'Delivery Server is required.',
          'email.max' => 'The email field must not be greater than 2500 characters.',
        ]);

        // Check if at least one sending server is provided
        // If sending_server_ids exists and has values, no need to check sending_server_id
        if (!empty($request->sending_server_ids) && is_array($request->sending_server_ids) && count($request->sending_server_ids) > 0) {
            // Use sending_server_ids, no need to check sending_server_id
        } elseif (empty($request->sending_server_id)) {
            return response()->json(['errors' => ['sending_server_id' => ['Delivery Server is required.']]], 422);
        }

        // Validate multiple emails
        $emails = array_map('trim', explode(',', $request->email));
        $emails = array_filter($emails, function($email) {
            return !empty($email);
        });

        if (empty($emails)) {
            return response()->json(['errors' => ['email' => [__('app.email_required')]]], 422);
        }

        // Validate each email individually
        $invalidEmails = [];
        foreach ($emails as $email) {
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $invalidEmails[] = $email;
            }
        }

        if (!empty($invalidEmails)) {
            return response()->json(['errors' => ['email' => [__('app.invalid_emails') . ': ' . implode(', ', $invalidEmails)]]], 422);
        }

        // Handle both single and multiple selection
        if(isset($request->sending_server_ids) && is_array($request->sending_server_ids)) {
          $request->sending_server_id = $request->sending_server_ids[0];
        }
        $sending_server = SendingServer::whereId($request->sending_server_id)
          ->with('bounce')
          ->first();
        if($sending_server->status == 'Active') {
          $connection = Helper::configureSendingNode($sending_server->type, $sending_server->sending_attributes, 'relayzo');

          $app_url = Helper::getAppURL();

          if($connection['success']) {
            $message = Helper::configureEmailBasicSettings($sending_server);

            if(in_array($sending_server->type, Helper::sendingServersFramworkSuported())) {
              // set to multiple emails - add each email individually
              foreach($emails as $email) {
                  $message->to($email);
              }
              
              // adding the envelope becuase bounce email having issue
              $envelopeAddresses = array_map(function($email) {
                  return new \Symfony\Component\Mime\Address($email);
              }, $emails);
              
              $envelope = new \Symfony\Component\Mailer\Envelope(
                  new \Symfony\Component\Mime\Address($sending_server->from_email, "$sending_server->from_name"), // From email
                  $envelopeAddresses // Envelope recipient(s)
              );

              // get sending domain from email
              $sending_domain = Helper::getSendingDomainFromEmail($sending_server->from_email);

              // set headers
              $headers= $message->getHeaders();
              
              if(!empty($sending_domain->protocol))
                $headers->addIdHeader('Message-ID', Helper::getCustomMessageID($sending_domain->protocol.$sending_domain->domain));

              $headers->addTextHeader('List-Unsubscribe', sprintf('<mailto:%s?subject=unsubscribe>', urlencode($sending_server['from_email'])));
              $headers->addTextHeader('List-Unsubscribe-Post', 'List-Unsubscribe=One-Click');

              if($request->broadcast_id) {
                $broadcast = \App\Models\Broadcast::whereId($request->broadcast_id)->first();

                // Get random contact to display proper data
                $contact = $this->_getRandomContact();

                // Process broadcast data with contact replacement
                $broadcast_data = $this->_getRandomContactRepalceData($broadcast, $contact, true);
                
                // Replace system variables
                $processed_data = $this->_processSystemVariables($contact, $broadcast_data['subject'], $broadcast_data['content'], $sending_server, $app_url);
                $subject = $processed_data['subject'];
                $content = $processed_data['content'];
                $tracking_domain = $processed_data['tracking_domain'];

                $message->from(new \Symfony\Component\Mime\Address($sending_server->from_email, $sending_server->from_name));

                $message->subject(Helper::decodeString($subject));
                //$content = str_replace('[$unsub-link$]',"",$broadcast->content_html);

                
               // $content = Helper::convertLinksForClickTracking(0, $tracking_domain, $content, $app_url, '/click/');

                $message->html($content);
                $message->text(htmlspecialchars_decode(strip_tags($content)));
              } elseif($request->template_id) {
                $template = \App\Models\Template::whereId($request->template_id)->first();
                
                // Get random contact to display proper data
                $contact = $this->_getRandomContact();

                // Process template data with contact replacement
                $template_data = $this->_getRandomContactRepalceData($template, $contact, false);
                
                // Replace system variables
                $processed_data = $this->_processSystemVariables($contact, $template_data['subject'], $template_data['content'], $sending_server, $app_url);
                $subject = $processed_data['subject'];
                $content = $processed_data['content'];
                $tracking_domain = $processed_data['tracking_domain'];
                
                $content = str_replace('[$unsub-link$]',"",$content);
                $content = Helper::convertLinksForClickTracking(0, $tracking_domain, $content, $app_url, '/click/');

                $message->from(new \Symfony\Component\Mime\Address($sending_server->from_email, $sending_server->from_name));
                $message->subject(Helper::decodeString($subject));
                $message->html($content);
                $message->text(htmlspecialchars_decode(strip_tags($content)));
              } else {
                $datetime = date('Y-m-d H:i:s');
                $message->subject("Test Email - {$sending_server->name} - $datetime");
                $message->html("<!DOCTYPE html><html><head><title>Test Email - {$sending_server->name}</title></head><body><p>This is a test email, sent using {$sending_server->name} at {$datetime}</p></body></html>");
                $message->text("This is a test email, sent using {$sending_server->name} at {$datetime}");

              }
             try {
                $result = $connection['transport']->send($message, $envelope);
                $status = 'Sent';
                $msg = __('app.msg_successfully_sent');
                $msg = "<span class='text-success'>".$msg."</span>";
              } catch (\Exception $e) {
                 $error = $e->getMessage();
                 $status = 'Failed';
                 $msg = "<span class='text-danger'>{$error}</span>";
              }
            } elseif($sending_server->type == 'sendgrid_api') {
              // Add multiple emails to SendGrid
              foreach($emails as $email) {
                  $message->addTo($email);
              }
              if($request->broadcast_id) {
                $broadcast = \App\Models\Broadcast::whereId($request->broadcast_id)->first();
                
                // Get random contact to display proper data
                $contact = $this->_getRandomContact();

                // Process broadcast data with contact replacement
                $broadcast_data = $this->_getRandomContactRepalceData($broadcast, $contact, true);
                
                // Replace system variables
                $processed_data = $this->_processSystemVariables($contact, $broadcast_data['subject'], $broadcast_data['content'], $sending_server, $app_url);
                $subject = $processed_data['subject'];
                $content = $processed_data['content'];
                
                $message->setSubject(Helper::decodeString($subject));
                $message->addContent("text/html", $content);
              } elseif($request->template_id) {
                $template = \App\Models\Template::whereId($request->template_id)->first();
                
                // Get random contact to display proper data
                $contact = $this->_getRandomContact();

                // Process template data with contact replacement
                $template_data = $this->_getRandomContactRepalceData($template, $contact, false);
                
                // Replace system variables
                $processed_data = $this->_processSystemVariables($contact, $template_data['subject'], $template_data['content'], $sending_server, $app_url);
                $subject = $processed_data['subject'];
                $content = $processed_data['content'];
                
                $message->setSubject(Helper::decodeString($subject));
                $message->addContent("text/html", $content);
              } else {
                $datetime = date('Y-m-d H:i:s');
                $message->setSubject("Test Email - {$sending_server->name} - $datetime");
                $message->addContent("text/html", "<!DOCTYPE html><html><head><title>Test Email - {$sending_server->name}</title></head><body><p>This is a test email, sent using {$sending_server->name} at {$datetime}</p></body></html>");
              }

              $sendgrid = new \SendGrid(\Crypt::decrypt(json_decode($sending_server->sending_attributes)->api_key));
              try {
                $response = $sendgrid->send($message);
                // status start with 2 consider as sent
                if(substr($response->statusCode(), 1) == 2) {
                  $status = 'Sent';
                  $msg = "<span class='text-success'>".__('app.msg_successfully_sent')."</span>";
                } else {
                  $status = 'Failed';
                  $msg = "<span class='text-danger'>".json_decode($response->body())->errors[0]->message."</span>";
                }
                
              } catch (\Exception $e) {
                $error = $e->getMessage();
                $status = 'Failed';
                $msg = "<span class='text-danger'>{$error}</span>";
              }
            } elseif($sending_server->type == 'elastic_email_api') {

                if($request->broadcast_id) {
                    $broadcast = \App\Models\Broadcast::whereId($request->broadcast_id)->first();
                    
                    // Get random contact to display proper data
                    $contact = $this->_getRandomContact();

                    // Process broadcast data with contact replacement
                    $broadcast_data = $this->_getRandomContactRepalceData($broadcast, $contact, true);
                    
                    // Replace system variables
                    $processed_data = $this->_processSystemVariables($contact, $broadcast_data['subject'], $broadcast_data['content'], $sending_server, $app_url);
                    $email_subject = $processed_data['subject'];
                    $email_body = $processed_data['content'];
                } elseif($request->template_id) {
                    $template = \App\Models\Template::whereId($request->template_id)->first();
                    
                    // Get random contact to display proper data
                    $contact = $this->_getRandomContact();

                    // Process template data with contact replacement
                    $template_data = $this->_getRandomContactRepalceData($template, $contact, false);
                    
                    // Replace system variables
                    $processed_data = $this->_processSystemVariables($contact, $template_data['subject'], $template_data['content'], $sending_server, $app_url);
                    $email_subject = $processed_data['subject'];
                    $email_body = $processed_data['content'];
                } else {
                    $datetime = date('Y-m-d H:i:s');
                    $email_subject = "Test Email - {$sending_server->name} - $datetime";
                    $email_body = "<!DOCTYPE html><html><head><title>Test Email - {$sending_server->name}</title></head><body><p>This is a test email, sent using {$sending_server->name} at {$datetime}</p></body></html>";
                }

                try {
                    // Send to multiple emails using ElasticEmail
                    foreach($emails as $email) {
                    \App\Services\ElasticEmailService::send(json_decode($sending_server->sending_attributes)->api_key,
                          $sending_server->from_email, $sending_server->from_name, $email, $sending_server->reply_email, $email_subject, $email_body
                    );
                    }
                    $status = 'Sent';
                    $msg = "<span class='text-success'>".__('app.msg_successfully_sent')."</span>";
                } catch (\GuzzleHttp\Exception\RequestException $e) {
                    $error = $e->getMessage();
                    $status = 'Failed';
                    $msg = "<span class='text-danger'>{$error}</span>";
                }
            } elseif($sending_server->type == 'postal_api') {

                if($request->broadcast_id) {
                    $broadcast = \App\Models\Broadcast::whereId($request->broadcast_id)->first();
                    
                    // Get random contact to display proper data
                    $contact = $this->_getRandomContact();

                    // Process broadcast data with contact replacement
                    $broadcast_data = $this->_getRandomContactRepalceData($broadcast, $contact, true);
                    
                    // Replace system variables
                    $processed_data = $this->_processSystemVariables($contact, $broadcast_data['subject'], $broadcast_data['content'], $sending_server, $app_url);
                    $email_subject = $processed_data['subject'];
                    $email_body = $processed_data['content'];
                } elseif($request->template_id) {
                    $template = \App\Models\Template::whereId($request->template_id)->first();
                    
                    // Get random contact to display proper data
                    $contact = $this->_getRandomContact();

                    // Process template data with contact replacement
                    $template_data = $this->_getRandomContactRepalceData($template, $contact, false);
                    
                    // Replace system variables
                    $processed_data = $this->_processSystemVariables($contact, $template_data['subject'], $template_data['content'], $sending_server, $app_url);
                    $email_subject = $processed_data['subject'];
                    $email_body = $processed_data['content'];
                } else {
                    $datetime = date('Y-m-d H:i:s');
                    $email_subject = "Test Email - {$sending_server->name} - $datetime";
                    $email_body = "<!DOCTYPE html><html><head><title>Test Email - {$sending_server->name}</title></head><body><p>This is a test email, sent using {$sending_server->name} at {$datetime}</p></body></html>";
                }

                try {
                    // Send to multiple emails using Postal API
                    $allSent = true;
                    $lastError = '';
                    
                    foreach($emails as $email) {
                    $result = \App\Services\PostalAPIService::send(
                        json_decode($sending_server->sending_attributes)->postal_server_url,
                        json_decode($sending_server->sending_attributes)->api_key,
                        $sending_server->from_email,
                        $sending_server->from_name,
                            $email,
                        $sending_server->reply_email,
                        $email_subject,
                        $email_body
                    );
                        
                        if($result !== 'send') {
                            $allSent = false;
                            $lastError = $result;
                        }
                    }
                    
                    if($allSent) {
                        $status = 'Sent';
                        $msg = "<span class='text-success'>".__('app.msg_successfully_sent')."</span>";
                    } else {
                        $status = 'Failed';
                        $msg = "<span class='text-danger'>{$lastError}</span>";
                    }
                } catch (\GuzzleHttp\Exception\RequestException $e) {
                    $error = $e->getMessage();
                    $status = 'Failed';
                    $msg = "<span class='text-danger'>{$error}</span>";
                }
            }

            if($status == 'Sent') {
              Helper::updateSendingServerCounters($sending_server->id);
            }
            return $msg;
          } else {
            return "<span class='text-danger'>{$connection['msg']}</span>";
          }
        } else {
          if($sending_server->status == 'Inactive')
            return "<span class='text-danger'>".__('app.delivery_server_inactive')."</span>";
          else
            return "<span class='text-danger'>{$sending_server->notification}</span>";
        }
    }

    /**
     * Retrun index after copy sending server
    */
    public function copy(int $id): RedirectResponse
    {
        Helper::checkPermissions('delivery_servers'); // check user permission
        $sending_server = SendingServer::whereId($id)->app()->first();
        $sending_server->name = $sending_server->name .' - copy ';
        $sending_server->total_sent = 0;
        $sending_server->hourly_sent = 0;
        $sending_server->daily_sent = 0;
        $copied_sending_server = $sending_server->replicate();
        $copied_sending_server->save();

        activity('copy')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.delivery_server') . " ({$sending_server->name}) ". __('app.log_copy'));

        return redirect()->route('sending_servers.index')
                ->with('success', ($sending_server->name . ' ' . __('app.copied_successfully')));
    }

    public function resetCounter($id)
    {
        return SendingServer::whereId($id)->update(['total_sent' => 0]);
    }

    /**
     * Get random contact for test emails
     */
    private function _getRandomContact()
    {
        // Get random contact to display proper data
        $contactIds = \App\Models\Contact::where('app_id', Auth::user()->app_id)->pluck('id');
        
        if ($contactIds->isEmpty()) {
            // No contacts available, create a dummy contact for testing
            \Log::warning('SendEmailTest: No contacts found in database, using dummy contact for testing');
            $contact = (object) [
                'id' => 0,
                'email' => 'test@example.com',
                'first_name' => 'Test',
                'last_name' => 'User',
                'list' => (object) ['name' => 'Test List'],
                'customFields' => collect()
            ];
        } else {
            $randomId = $contactIds->random();
            $contact = \App\Models\Contact::whereId($randomId)
              ->with('list')
              ->with('customFields')
              ->first();
              
            // Fallback if contact not found
            if (!$contact) {
                \Log::warning('SendEmailTest: Random contact not found, using dummy contact for testing');
                $contact = (object) [
                    'id' => 0,
                    'email' => 'test@example.com',
                    'first_name' => 'Test',
                    'last_name' => 'User',
                    'list' => (object) ['name' => 'Test List'],
                    'customFields' => collect()
                ];
            }
        }
        
        return $contact;
    }

    /**
     * Process broadcast or template content with contact replacement
     * Works for all sending server types
     */
    private function _getRandomContactRepalceData($broadcastOrTemplate, $contact, $isBroadcast = true)
    {
        if ($isBroadcast) {
            $content = $broadcastOrTemplate->content_html;
            $subject = $broadcastOrTemplate->email_subject;
        } else {
            $content = $broadcastOrTemplate->content;
            $subject = Helper::decodeString($broadcastOrTemplate->name);
        }

        // Replace spintags
        $content = Helper::replaceSpintags(Helper::decodeString($content));
        $subject = Helper::replaceSpintags(Helper::decodeString($subject));

        // Replace custom field
        $content = Helper::replaceCustomFields($content, $contact->customFields);
        $subject = Helper::replaceCustomFields($subject, $contact->customFields);

        // Replace Spintax [Random: hi|hello|hey]
        $content = Helper::spinTaxParser($content);
        $subject = Helper::spinTaxParser($subject);

        // Default Parser
        $content = Helper::defaultValueParser($content);
        $subject = Helper::defaultValueParser($subject);

        return [
          'subject' => $subject,
          'content' => $content
        ];
    }

    /**
     * Process and replace system variables in content and subject
     * Works for all sending server types
     */
    private function _processSystemVariables($contact, $subject, $content, $sending_server, $app_url)
    {
        $tracking_domain = !empty($sending_server->tracking_domain) ? $sending_server->tracking_domain : $app_url;
        
        $data_values = [
          'sender-name'    => $sending_server->from_name,
          'sender-email'   => $sending_server->from_email,
          'domain'         => $tracking_domain,
        ];

        // Replace system variables
        $subject = Helper::replaceSystemVariables($contact, $subject, $data_values);
        $content = Helper::replaceSystemVariables($contact, $content, $data_values);

        return [
          'subject' => $subject,
          'content' => $content,
          'tracking_domain' => $tracking_domain
        ];
    }
}
