<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use App\Models\Setting;
use App\Http\Helper\Helper;

class SettingController extends Controller
{

    /**
     * Show the form for a resource.
     */
    public function index(): View
    {
        Helper::checkPermissions('settings_applications');
        $page = 'settings_application';
        $page_title = __('app.application');
        $breadcrumbs = [
            __('app.settings') => route('settings')
        ];

        $settings = Setting::find(config('custom.app_id'));

        return view('settings.index',compact('settings', 'page', 'page_title', 'breadcrumbs'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update($id, Request $request): RedirectResponse
    {
        $this->validate($request, [
            'app_name' => 'required',
            'app_url' => 'required|url',
            'max_allowed_file_size' => 'required|int|max:'.Helper::getMaxFileSize($only_value=true)
        ]);
    
        $input = $request->except('_token');

        $input['sending_attributes'] = app('App\Http\Controllers\SendingServerController')->sendingServerAttributes($input);
        $settings = Setting::where('id', config('custom.app_id'))->update([
            'app_name' => $input['app_name'],
            'app_url' => str_replace('www.', '', $input['app_url']),
            'tracking' => $input['tracking'],
            'sending_type' => $input['type'],
            'from_name' => $input['from_name'],
            'from_email' => $input['from_email_part1'] . '@' . $input['from_email_part2'],
            'sending_type' => $input['type'],
            'sending_attributes' => $input['sending_attributes'],
            'attributes->max_file_size' => $input['max_allowed_file_size'],
            'attributes->activity_log_retention' => $input['activity_log_retention']
        ]);

        if($request->hasFile('file_login_page_logo')){ 
          $request->validate([
            'file_login_page_logo' => 'image|mimes:png|max:2048', // 2MB
          ],
          [
            'file_login_page_logo.image' => 'The Login Page Logo file must be an image.',
            'file_login_page_logo.mimes' => 'The Login Page Logo must be a PNG file.',
            'file_login_page_logo.max' => 'The Login Page Logo size must not exceed 2MB.',
          ]);

          $file = $request->file('file_login_page_logo');
          $file_ext = $file->getClientOriginalExtension();
          $file_name = 'login_logo.'.$file_ext;
          $file->move(config('custom.path_app_images'), $file_name);
        }

        if($request->hasFile('file_application_logo')){ 
          $request->validate([
            'file_application_logo' => 'image|mimes:png|max:2048', // 2MB
          ],
          [
            'file_application_logo.image' => 'The Application Logo file must be an image.',
            'file_application_logo.mimes' => 'The Application Logo must be a PNG file.',
            'file_application_logo.max' => 'The Application Logo size must not exceed 2MB.',
          ]);

          $file = $request->file('file_application_logo');
          $file_ext = $file->getClientOriginalExtension();
          $file_name = 'site_logo.'.$file_ext;
          $file->move(config('custom.path_app_images'), $file_name);
        }

        if($request->hasFile('file_favicon')){ 
          $request->validate([
            'file_favicon' => 'mimes:ico,png,svg||max:2048', // 2MB
          ],
          [
            'file_favicon.mimes' => 'The Favicon Icon must be a ICO file.',
            'file_favicon.max' => 'The Favicon Icon size must not exceed 2MB.',
          ]);

          $file = $request->file('file_favicon');
          $file_name = 'favicon.ico';
          $file->move(config('custom.path_app_images'), $file_name);
        }

        return redirect()->route('settings', compact('settings'))
            ->with('success',__('app.settings') .' '. __('app.updated_successfully'));;
        
    }

    // get API view and regenerate API Token
    public function api(Request $request): View
    {
        Helper::checkPermissions('settings_api');
        $page = 'settings_api';
        $page_title = __('app.api');
        $breadcrumbs = [
            __('app.settings') => route('api')
        ];

        $user = \App\Models\User::whereAppId(\Auth::user()->app_id)->first();
        if($request->method() == 'PUT') {
          $user->fill([
            'api_token' => \Illuminate\Support\Str::random(60)
          ])->save();
        }

        return view('settings.api',compact('user', 'page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Save API status
    */
    public function apiStatus(Request $request)
    {
        $user = \App\Models\User::whereAppId(\Auth::user()->app_id)->first();
        if($request->method() == 'PUT') {
          $user->fill([
            'api_enabled' => $request->value
          ])->save();
          return true;
        }
        return false;
    }

    // get API view and regenerate API Token
    public function chatGPTSettings(Request $request): View
    {
        Helper::checkPermissions('settings_chatGPT');
        $page = 'settings_chatGPT';
        $page_title = __('app.chatGPT');
        $breadcrumbs = [
            __('app.settings') => route('chatGPT')
        ];

        $user = \App\Models\User::whereAppId(\Auth::user()->app_id)->first();
        if($request->method() == 'PUT') {
          $user->fill([
            'chatgpt_key' => $request->chatgpt_key,
            'chatgpt_model' => $request->chatgpt_model,
            'chatgpt_enabled' => $request->chatgpt_enabled
          ])->save();

          session()->flash('success', 'Updated successfully.');
        }

        activity('update')->withProperties(['app_id' => \Auth::user()->app_id])->log(__('app.settings') . "  ". __('app.chatGPT'). ' ' . __('app.log_update'));

        return view('settings.chatGPT',compact('user', 'page', 'page_title', 'breadcrumbs'));
    }

    // get General settings view
    public function generalSettings(Request $request): View
    {
        Helper::checkPermissions('settings_api');
        $page = 'settings_general';
        $page_title = __('app.general');
        $breadcrumbs = [
            __('app.settings') => route('general.settings')
        ];

        $user = \App\Models\User::whereAppId(\Auth::user()->app_id)->first();
        if($request->method() == 'PUT') {
          $user->fill([
            'unsubscribed_mark_as_suppressed' => $request->unsubscribed_mark_as_suppressed
          ])->save();

          session()->flash('success', 'Updated successfully.');
        }

        activity('update')->withProperties(['app_id' => \Auth::user()->app_id])->log(__('app.settings') . "  ". __('app.general'). ' ' . __('app.log_update'));

        return view('settings.general',compact('user', 'page', 'page_title', 'breadcrumbs'));
    }

    public function chatGPT(Request $request)
    {
      $text = $request->content;
      $chatGPT_response = null;
      if(!empty($text) && $text != 'null') {
        $chatGPT_response = Helper::chatGPT($text);
      }
      return view('includes.chatGPT')->with(compact('chatGPT_response'));
    }
}
