<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use App\Models\Spintag;
use App\Http\Helper\Helper;
use Illuminate\View\View;
use Auth;

class SpintagController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        Helper::checkPermissions('spintags');
        $page = "campaigns_spintag";
        $page_title = __('app.spintags');
        $breadcrumbs = [
            __('app.spintags') => route('spintags.index'),
            __('app.manage') => '#'
        ];  
        return view('spintags.index',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
       * Retrun JSON datatable data
    */
    public function getSpintags(Request $request): Void
    {
        $result = Spintag::select('id', 'name', 'tag', 'values', 'created_at')
            ->app();
        
        $columns = ['id', 'name', 'tag', 'values', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $spintags = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($spintags as $spintag) {
          $checkbox = "<input class=\"form-check-input\" type=\"checkbox\" value=\"{$spintag->id}\">";
          $actions = '<div class="btn-custom_field">
            <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').' <span class="caret"></span></button><ul class="dropdown-menu" role="menu">';
          $actions.= '<li><a class="dropdown-item" href="javascript:;" onclick="viewModal(\'modal\', \''.route('spintags.edit', [$spintag->id]).'\')"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
          $actions .= '<li><a class="dropdown-item" class="text-red" href="javascript:;" onclick="destroy(\''.$spintag->id.'\', \''.route('spintags.destroy', [$spintag->id]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul></div>';
          $data['data'][] = [
            "DT_RowId" => "row_{$spintag->id}",
            $checkbox,
            $spintag->name,
            '[%'.$spintag->tag.'%]',
            Helper::datetimeDisplay($spintag->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        Helper::checkPermissions('spintags');
        return view('spintags.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
          'name' => 'required|unique:spintags'
        ]);
        $data = $this->_spintagData($request);
        Spintag::create($data);
        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.spintag') . " ({$request->name}) ". __('app.log_create'));

        return redirect()->back()
            ->with('success', __('app.spintag') . ' ' . __('app.log_create'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Spintag $spintag): View
    {
        Helper::checkPermissions('spintags');
        return view('spintags.edit')->with(compact('spintag'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Spintag $spintag): RedirectResponse
    {
        $request->validate([
          'name' => 'required'
        ]);

        $data = $this->_spintagData($request);
        $spintag->fill($data)->save();

        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.spintag') . " ({$request->name}) ". __('app.log_update'));

        return redirect()->back()
            ->with('success', __('app.spintag') . ' ' . __('app.log_update'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $names = json_encode(array_values(Spintag::whereIn('id', $ids)->pluck('name')->toArray()));
          $destroy = Spintag::whereIn('id', $ids)->delete();
        } else {
          $names = Spintag::whereId($id)->value('name');
          $destroy = Spintag::destroy($id);
        }
        
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.spintag') . " ({$names}) ". __('app.log_delete'));

        $data = [
            'success' => true,
            'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200); 
    }

    /**
    * Retrun data for store or update
    */
    private function _spintagData($request) 
    {
        $input = $request->except('_token');
        $input['values'] = !empty($input['values']) ? $input['values'] : null;
        $input['tag'] = \Illuminate\Support\Str::slug($input['name'], '-');
        $input['user_id'] = Auth::user()->id;
        $input['app_id'] = Auth::user()->app_id;
        return $input;
    }
}
