<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use App\Http\Requests\SplitTestRequest;
use App\Http\Helper\Helper;
use App\Models\SplitTest;
use Auth;

class SplitTestController extends Controller
{
    /**
    * Retrun index view
    */
    public function index(): view
    {
        Helper::checkPermissions('schedule_split_tests');
        $page = 'schedule_split_tests';
        $page_title = __('app.split_tests');
        $breadcrumbs = [
            __('app.schedule') => '#',
            __('app.split_tests') => route('split_tests.index')
        ];
        return view('split_tests.index',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Retrun JSON datatable data
    */
    public function getSplitTests(Request $request): void
    {
        $result = SplitTest::select('id', 'name', 'send_datetime', 'winning_criteria',
          'decision_percentage', 'action', 'created_at','status', 'scheduled','total', 'sent', 'action_data', 'sending_speed')
          ->whereAppId(Auth::user()->app_id)
          ->with('splitTestStat');

        $columns = ['id', 'name', 'send_datetime', 'winning_criteria', 'decision_percentage', 'action', 'id', 'status', 'scheduled', 'total', 'id', 'id', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $split_tests = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($split_tests as $split_test) {
            $sending_speed = json_decode($split_test->sending_speed);

            $checkbox = "<input class=\"form-check-input\" type=\"checkbox\" value=\"{$split_test->id}\">";
            $actions = '<div class="btn-group">
            <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').' <span class="caret"></span></button><ul class="dropdown-menu" role="menu">';

            $actions .= '<li><a class="dropdown-item" href="'.route('split_tests.edit', [$split_test->id]).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';

            if(!empty($split_test->splitTestStat->id)) {
                $name = '<a class="link-info" href="javascript:;" onclick="viewModal(\'modal\', \''.route('detail.stat.split_test', [$split_test->splitTestStat->id, 'summary', 'analatics.split_tests.summary_popup']).'\')"> '.$split_test->name.'</a>';

                $actions .= '<li><a class="dropdown-item" href="javascript:;" onclick="viewModal(\'modal\', \''.route('detail.stat.split_test', [$split_test->splitTestStat->id, 'summary', 'analatics.split_tests.summary_popup']).'\')"><i class="bi bi-pie-chart"></i> '.__('app.summary').'</a></li>';
                $actions .= '<li><a class="dropdown-item" href="'.route('detail.stat.split_test', ['id' => $split_test->splitTestStat->id]).'"><i class="bi bi-bar-chart"></i> '.__('app.stats').'</a></li>';
            } else {
                $name = $split_test->name;
            }

            if($split_test->status == 'RunningLimit' && !empty($sending_speed->limit)) {
                $actions .= '<li><a class="dropdown-item"  href="javascript:;" onclick="runLimitedToUnlimitedSplitTest(\''.$split_test->id.'\', \''.__('app.run_limited_to_unlmited_msg'). '\')" ><i class="bi bi-rocket-takeoff"></i> '.__('app.run_as_unlimited').'</a></li>';

                $initial_limit = $sending_speed->initial_limit??$sending_speed->limit;
                $actions .= '<li><a class="dropdown-item"  href="javascript:;" onclick="set2xSpeedSplitTest(\''.$split_test->id.'\',\''.$sending_speed->limit.'\',\''.$initial_limit.'\', \''.__('app.set_speed_2x_msg'). '\')" ><i class="bi bi-rocket"></i> '.__('app.set_speed_2x_title').'</a></li>';
            }

            $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$split_test->id.'\', \''.route('split_tests.destroy', [$split_test->id]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
                $actions .= '</ul></div>';

            $status = '';
            if($split_test->status == 'Running' || $split_test->status == 'RunningLimit' || $split_test->status == 'Resume') {
                $status = "<a href='javascript:;'>
                  <i class='bi bi-play-circle-fill text-success' id='play-{$split_test->id}' title='".__('app.play')."' onclick='pauseSplitTest(\"{$split_test->id}\");' style='display:none;'></i>
                  <i class='bi bi-pause-circle-fill text-warning' id='pause-{$split_test->id}' title='".__('app.pause')."' onclick='playSplitTest(\"{$split_test->id}\");'></i>
                  </a>";
                $split_test->status = 'Running'; // To overwrite RunningLimit to display
            } elseif($split_test->status == 'Paused') {
                $status = "<a href='javascript:;'>
                  <i class='bi bi-play-circle-fill text-success' id='play-{$split_test->id}' title='".__('app.play')."' onclick='pauseSplitTest(\"{$split_test->id}\");'></i>
                  <i class='bi bi-pause-circle-fill text-warning' id='pause-{$split_test->id}' title='".__('app.pause')."' onclick='playSplitTest(\"{$split_test->id}\");' style='display:none;'></i>
                  </a>";
            } elseif($split_test->status == 'System Paused') {
                $status = '<a href="javascript:;" onclick="viewModal(\'modal\', \''.route('sending_server.status', ['id' => $split_test->sending_server_ids]).'\')"><i class="bi bi-info-circle-fill text-danger"></i></a>';
            }
            $status .= " <span id='status-{$split_test->id}'>{$split_test->status}</span>";

            $spinner = '<div class="spinner-border spinner-border-sm text-dark" role="status"><spam class="visually-hidden"></span></div>';

            $progress = "( {$split_test->sent} / {$split_test->scheduled} ) " . Helper::getPercentage($split_test->sent, $split_test->scheduled);
            $scheduled = '<a class="link-info" href="javascript:;" onclick="viewModal(\'modal\', \''.route('scheduled.detail.campaign', ['id' => $split_test->id, 'type' => 'split_test']).'\')">'.$split_test->scheduled.'</a>';

            $limit = json_decode($split_test->sending_speed)->speed == 'unlimited' ? __('app.unlimited') : $sending_speed->limit;

            $action_data = json_decode($split_test->action_data);
            $data['data'][] = [
                "DT_RowId" => "row_{$split_test->id}",
                $checkbox,
                $name,
                Helper::datetimeDisplay($split_test->send_datetime),
                ucfirst($split_test->winning_criteria),
                $split_test->decision_percentage.' %',
                ucwords(str_replace('_', ' ', $split_test->action)),
                "$action_data->send_remaining_after $action_data->send_remaining_after_duration",
                $limit,
                $status,
                $split_test->total == null ? $spinner : $split_test->total,
                $split_test->scheduled == null ? $spinner : $scheduled,
                $split_test->total == null ? $spinner : $progress,
                Helper::datetimeDisplay($split_test->created_at),
                $actions
            ];
        }
        echo json_encode($data);
    }

    /**
    * Retrun create view
    */
    public function create(): view
    {
        Helper::checkPermissions('schedule_split_tests');
        $page = 'schedule_split_tests';
        $page_title = __('app.split_tests');
        $breadcrumbs = [
            __('app.schedule') => '#',
            __('app.split_tests') => route('split_tests.index')
        ];
        return view('split_tests.create',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Save data
    */
    public function store(SplitTestRequest $request): RedirectResponse
    {
        Helper::checkPermissions('schedule_split_tests');
        $data = $this->_splitTestData($request);
        SplitTest::create($data);

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.split_test') . " ({$request->name}) ". __('app.log_create'));
        return redirect()->route('split_tests.index')
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
    }

    /**
    * Retrun create view
    */
    public function edit($id): view
    {
        Helper::checkPermissions('schedule_split_tests');
        $page = 'schedule_split_tests';
        $page_title = __('app.split_tests');
        $breadcrumbs = [
            __('app.schedule') => '#',
            __('app.split_tests') => route('split_tests.index')
        ];
        $split_test = SplitTest::whereId($id)->app()->first();
        return view('split_tests.edit',compact('page', 'page_title', 'breadcrumbs', 'split_test'));
    }

    /**
    * Update data
    */
    public function update(Request $request, $id)
    {
        Helper::checkPermissions('schedule_split_tests');
        $split_test = SplitTest::findOrFail($id);
        // update only name and is_active
        $split_test->update($request->only(['name', 'is_active']));

        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.split_test') . " ({$request->name}) ". __('app.log_update'));

        return redirect()->route('split_tests.edit', $split_test->id)
            ->with('success', ($request->name . ' ' . __('app.updated_successfully')));
    }

    /**
    * Retrun data for store or update
    */
    public function _splitTestData($request)
    {
        $input = $request->except('_token');
        $carbon = new \Carbon\Carbon();
        if($input['send'] == 'now') {
          $input['send_datetime'] = $carbon->now();
        } else {
          // It has the correct value as per UTC for a calander
          $send_datetime = date('Y-m-d H:i:s', strtotime("{$input['send_datetime']}"));
          // Convert future datetime into UTC datetime
          $offsetSeconds =  $carbon->now(Auth::user()->timezone)->getOffset();
          $input['send_datetime'] = \Carbon\Carbon::parse($send_datetime, config('app.timezone'))->subSeconds($offsetSeconds);
        }

        if($input['from_detail'] == 'custom') {
            $input['from_detail_custom'] = json_encode([
                'from_name' => $input['from_name'],
                'from_email' => ($input['from_email_part1'] . '@' . $input['from_email_part2']),
                'reply_email' => $input['reply_email']
            ]);
        }

        $input['sending_datetime'] = $input['send_datetime'];
        $input['broadcast_ids'] = json_encode($request->broadcast_ids);
        $input['list_ids'] = json_encode($request->list_ids);
        $input['action_data'] = json_encode(['send_remaining_after'=>$input['send_remaining_after'], 'send_remaining_after_duration'=>$input['send_remaining_after_duration']]);
        $input['sending_server_ids'] = json_encode($request->sending_server_ids);
        $input['sending_speed'] = json_encode([
            'speed'=>$input['sending_speed'],
            'limit'=>$input['limit'],
            'duration'=>$input['duration'],
            'pause_limit' => $input['pause_limit'],
            'pause_duration' => $input['pause_duration']
        ]);
        $input['attributes'] = json_encode($input);
        $input['app_id'] = Auth::user()->app_id;
        $input['user_id'] = Auth::user()->id;
        return $input;
    }

    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
            $ids = array_values($request->ids);
            $names = json_encode(array_values(SplitTest::whereIn('id', $ids)->pluck('name')->toArray()));
            $destroy = SplitTest::whereIn('id', $ids)->delete();
        } else {
            $names = SplitTest::whereId($id)->value('name');
            $destroy = SplitTest::destroy($id);
        }
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.split_test') . " ({$names}) ". __('app.log_delete'));

        $data = [
          'success' => true,
          'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200);
    }

    /**
     * Update status
    */
    public function updateSplitTestStatus(Request $request, $id) {
      $status = $request->status;
      // For running status need to check either set to Running or RunningLimit(hourly)
      if($status == 'Resume') {
        $sending_speed = SplitTest::whereId($id)->value('sending_speed');
        if(json_decode($sending_speed)->limit) {
          $status = 'RunningLimit';
        }
      }
      SplitTest::whereId($id)->update(['status' => $status]);
    }

    /**
     * Initiate send split test process via scheduled command
    */
    public function processSplitTest($id = null, $thread_no = 1)
    {
      if (!empty($id)) {
        $send = \Artisan::call('run:split-test-campaigns', [
          'id'        => $id,
          'thread_no' => $thread_no
        ]);

        /*\App\Jobs\RunCampaign::dispatch($id, $thread_no, 'split_test')
            ->delay(now()->addSeconds(5))
            ->onQueue('high');*/
      } else {
        $split_tests = SplitTest::whereIn('status', ['Scheduled', 'RunningLimit', 'Resume'])
          ->where('sending_datetime', '<=', \Carbon\Carbon::now())
          ->orderBy('id', 'desc')
          ->get();
        if(count($split_tests) > 0) {
          foreach($split_tests as $split_test) {
            $this->_processSplitTestParallels($split_test);
          }
        }
      }
    }

    /**
     * Send parallel request to execute the campaign
    */
    private function _processSplitTestParallels($split_test)
    {
      $app_url = Helper::getAppURL();

      // Need to start thead_no with 1 instead 1 as default set 0
      $thread_no = $split_test->thread_no == 0 ? 1 : $split_test->thread_no;
      // If set to send hourly then need to send only one thread with thread no
      if(json_decode($split_test->sending_speed)->speed == 'limited') {
        $url = $app_url."/process_split_test/{$split_test->id}/{$thread_no}";
        Helper::getUrl($url);
      } else  {
        $threads = $thread_no + $split_test->threads;
        for($i=$thread_no; $i<$threads; $i++) {
          $url = $app_url."/process_split_test/{$split_test->id}/{$i}";
          Helper::getUrl($url);

          // no need to run extra otherwise it will throw job error.
          if($i >= $split_test->total_threads) break;
        }
      }
    }

    /**
     * Set split test to run limited to unlimited.
    */
    public function limitedToUnlimited(Request $request) {
      $sending_speed = json_encode([
        'speed' => 'unlimited',
        'limit' => null,
        'duration'=> 'hour'
      ]);
      SplitTest::whereId($request->id)->update([
        'status' => 'Resume',
        'send_datetime' => \Carbon\Carbon::now(),
        'thread_no' => 1,
        'sending_speed' => $sending_speed,
        ]
      );

      // Update stat table also
      \App\Models\SplitTestStat::whereSplitTestId($request->id)->update([
        'sending_speed' => $sending_speed,
        ]
      );
    }

    /**
     * Set split test to run limited to unlimited.
    */
    public function set2xSpeed(Request $request) {

      $new_limit = $request->limit*2;
      $sending_speed = json_encode([
        'speed' => 'limited',
        'limit' => $new_limit,
        'duration' => 'hour',
        'initial_limit' => $request->initial_limit,
        'running_speed' => '2X'
      ]);
      SplitTest::whereId($request->id)->update([
        'sending_speed' => $sending_speed,
        ]
      );

      // Update stat table also
      \App\Models\SplitTestStat::whereSplitTestId($request->id)->update([
        'sending_speed' => $sending_speed,
        ]
      );
    }
}
