<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\SplitTestStat;
use App\Http\Helper\Helper;
use Auth;

class SplitTestStatController extends Controller
{
  /**
   * Retrun index view
  */
  public function index()
  {
    Helper::checkPermissions('analatics_split_tests');
        $page = 'analatics_split_test';
        $page_title = __('app.analatics_split_tests');
        $breadcrumbs = [
            __('app.analatics') => '#',
            __('app.split_test') => route('stats.split_tests.index')
        ];
    return view('analatics.split_tests.index')->with(compact('page', 'page_title', 'breadcrumbs'));
  }

  /**
   * Retrun JSON datatable data
  */
  public function getSplitTests(Request $request) {
    $result = SplitTestStat::whereAppId(Auth::user()->app_id);

    $columns = ['id', 'start_datetime', 'id', 'id', 'id','id','id','id','id','id','id','created_at'];

    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $scheduled = '<a class="link-info" href="javascript:;" onclick="viewModal(\'modal\', \''.route('scheduled.detail.campaign', ['id' => $stat->split_test_id, 'type'=>'split_test']).'\')">'.$stat->scheduled.'</a>';
      $progress = "( {$stat->sent} / {$stat->scheduled} ) " . Helper::getPercentage($stat->sent, $stat->scheduled);

      $opens_clicks = json_decode($stat->opens_clicks);

      $opens = $this->opensAllUniqueWithPercentage($stat->id, $stat->sent, $opens_clicks->total_opens, $opens_clicks->unique_opens);
      $clicks = $this->clicksAllUniqueWithPercentage($stat->id, $stat->sent, $opens_clicks->total_clicks, $opens_clicks->unique_clicks);
      $replies = \App\Models\SplitTestStat::statLogReplies($stat->id)->get()->count();

      $name = '<a class="link-info" href="javascript:;" onclick="viewModal(\'modal\', \''.route('detail.stat.split_test', [$stat->id, 'summary', 'analatics.split_tests.summary_popup']).'\')">'.$stat->split_test_name.'</a>';
      $detail = '<a class="link-info" href="'.route('detail.stat.split_test', ['id' => $stat->id]).'"><i class="bi bi-bar-chart"></i></a>';

      if(Helper::checkPermissions('analatics_split_test_delete', false)) {
        $action = '<a class="text-danger" href="javascript:;" onclick="destroy(\''.$stat->id.'\', \''.route('stat.splittest.destroy', ['id' => $stat->id, 'name' => $stat->split_test_name]).'\')">'.__('app.delete').'</a>';
      } else {
        $action = '---';
      }

      $data['data'][] = [
        "DT_RowId" => "row_{$stat->id}",
        $name,
        !empty($stat->start_datetime) ? Helper::datetimeDisplay($stat->start_datetime) : '---',
        ucfirst($stat->winning_criteria),
        $stat->decision_percentage.' %',
        ucwords(str_replace('_', ' ', $stat->action)),
        $stat->total,
        $scheduled,
        $progress,
        $opens,
        $clicks,
        $replies,
        Helper::datetimeDisplay($stat->created_at),
        $detail,
        $action
      ];
    }
    echo json_encode($data);
  }

  /**
   * Retrurn detail view for an Split Test
  */
  public function getDetailStat($id, $type=null, $view='analatics.split_tests.summary')
  {
    Helper::checkPermissions('analatics_split_tests');
    $split_test_stat = SplitTestStat::whereId($id)->app()->first();
    if(empty($type)) {
      $page = 'analatics_split_test';
      $page_title = __('app.analatics').' - '.__('app.split_test').' - '. __('app.detail').' - '.($split_test_stat->split_test_name ?? '---');
      $breadcrumbs = [
            __('app.analatics') => '#',
            __('app.split_test') => route('stats.split_tests.index'),
            __('app.detail') => '#'
      ];
      return view('analatics.split_tests.detail')->with(compact('page', 'page_title', 'breadcrumbs', 'split_test_stat'));
    } else {
      switch($type) {
        case 'opens':
          return view('analatics.split_tests.opens')->with('stat_id', $id);
        break;
        case 'clicks':
          return view('analatics.split_tests.clicks')->with('stat_id', $id);
        break;
        case 'unsubscribed':
          return view('analatics.split_tests.unsubscribed')->with('stat_id', $id);
        break;
        case 'bounces':
          return view('analatics.split_tests.bounces')->with('stat_id', $id);
        break;
        case 'spam':
          return view('analatics.split_tests.spam')->with('stat_id', $id);
        break;
        case 'replies':
          return view('analatics.split_tests.replies')->with('stat_id', $id);
        break;
        case 'logs':
          return view('analatics.split_tests.logs')->with('stat_id', $id);
        break;
        default:
          $data = $this->getSummary($split_test_stat, $type, $view);
      }
      return $data;
    }
  }

  /**
   * Retrurn summary for an Split Test
  */
  public function getSummary($split_test_stat, $type=null, $view='analatics.split_tests.summary')
  {
    if($type == 'summary_sent') {
      return $sent = $split_test_stat->sent;
    } elseif($type == 'summary_opens') {
      return $opens = $this->opensAllUniqueWithPercentageLive($split_test_stat->id, $split_test_stat->sent);
    } elseif($type == 'summary_clicks') {
      return $clicks = $this->clicksAllUniqueWithPercentageLive($split_test_stat->id, $split_test_stat->sent);
    } elseif($type == 'summary_unsubscribed') {
      return $unsubscribed = SplitTestStat::statLogClicks($split_test_stat->id)->where('link', 'like', '%/unsub/%')->whereUnsubConfirmed(true)->get()->count();
    } elseif($type == 'summary_bounces') {
      return $bounces = \App\Models\SplitTestStat::statLogBounces($split_test_stat->id)->get()->count();
    } elseif($type == 'summary_spam') {
      return $spam = \App\Models\SplitTestStat::statLogSpam($split_test_stat->id)->get()->count();
    } elseif($type == 'summary_replies') {
      return $replies = \App\Models\SplitTestStat::statLogReplies($split_test_stat->id)->get()->count();
    } elseif($type == 'summary_sent_lists') {
      $lists = \App\Models\SplitTestStat::statLogData($split_test_stat->id, 'list')->pluck('total', 'list')->toArray();
      $data = '';
      foreach($lists as $list => $total){
        $data .= "{$list} <strong>({$total})</strong>";
      }
      return $data;
    } elseif($type == 'summary_sent_campaigns') {
      $broadcasts = \App\Models\SplitTestStat::statLogData($split_test_stat->id, 'broadcast')->pluck('total', 'broadcast')->toArray();
      $data = '';
      foreach($broadcasts as $broadcast => $total) {
        $data .= "{$broadcast} <strong>({$total})</strong>";
      }
      return $data;
    } elseif($type == 'summary_sent_sending_servers') {
      $sending_servers = \App\Models\SplitTestStat::statLogData($split_test_stat->id, 'sending_server')->pluck('total', 'sending_server')->toArray();
      $data = '';
      foreach($sending_servers as $sending_server => $total) {
        $data .= "{$sending_server} <strong>({$total})</strong>";
      }
      return $data;
    }

    return view($view)->with(compact('split_test_stat'));
  }

  /**
   * Retrurn all opens and unique opens with percentage
  */
  public function opensAllUniqueWithPercentage($stat_id, $stat_sent, $total_opens, $unique_opens)
  {
    return $opens = "( <a title='".__('app.unique')."'>{$unique_opens}</a> / <a title='".__('app.all')."'>{$total_opens}</a> ) " . Helper::getPercentage($unique_opens, $stat_sent);
  }

  /**
   * Retrurn all clicks and unique clicks with percentage
  */
  public function clicksAllUniqueWithPercentage($stat_id, $stat_sent, $total_clicks, $unique_clicks)
  {
    return $clicks = "( <a title='".__('app.unique')."'>{$unique_clicks}</a> / <a title='".__('app.all')."'>{$total_clicks}</a> ) " . Helper::getPercentage($unique_clicks, $stat_sent);
  }

  /**
   * Retrurn all opens and unique opens with percentage
  */
  public function opensAllUniqueWithPercentageLive($stat_id, $stat_sent)
  {
    $opens_all = SplitTestStat::statLogOpens($stat_id)->get()->count();
    $opens_unique = SplitTestStat::statLogOpens($stat_id, 'unique')->get()->count();
    return $opens = "( <a title='".__('app.unique')."'>{$opens_unique}</a> / <a title='".__('app.all')."'>{$opens_all}</a> ) " . Helper::getPercentage($opens_unique, $stat_sent);
  }

  /**
   * Retrurn all clicks and unique clicks with percentage
  */
  public function clicksAllUniqueWithPercentageLive($stat_id, $stat_sent)
  {
    $clicks_all = SplitTestStat::statLogClicks($stat_id)->get()->count();
    $clicks_unique = SplitTestStat::statLogClicks($stat_id, 'unique')->get()->count();
    return $clicks = "( <a title='".__('app.unique')."'>{$clicks_unique}</a> / <a title='".__('app.all')."'>{$clicks_all}</a> ) " . Helper::getPercentage($clicks_unique, $stat_sent);
  }

  /**
   * Retrurn JSON for an Split Test
  */
  public function getSentData($id, $type = null)
  {
    if($type == 'country') {
      $data = SplitTestStat::getUniqueCountries($id)->get()->pluck('cnt', 'country_code')->toArray();
    } else {
      $opens_unique = SplitTestStat::statLogOpens($id, 'unique')->get()->count();
      $clicks_unique = SplitTestStat::statLogClicks($id, 'unique')->get()->count();
      $unsubscribed = SplitTestStat::statLogClicks($id)->where('link', 'like', '%/unsub/%')->whereUnsubConfirmed(true)->get()->count();
      $bounces = SplitTestStat::statLogBounces($id)->get()->count();
      $spam = SplitTestStat::statLogSpam($id)->get()->count();
      $replies = SplitTestStat::statLogReplies($id)->get()->count();

      $dataset = [
          'data' => [$opens_unique, $clicks_unique, $replies, $unsubscribed, $bounces, $spam],
          'backgroundColor' => ['#0EA5E8', '#21CE9E', '#202947', '#FF851B', '#DD4B39', '#FFC0CB'],
        ];

      $data = [
        'labels' => [
          "$opens_unique ".__('app.opens'),
          "$clicks_unique ".__('app.clicks'),
          "$replies ".__('app.replies'),
          "$unsubscribed ".__('app.unsubscribed'),
          "$bounces ".__('app.bounces'),
          "$spam ".__('app.spam')
        ],
        'datasets' => [$dataset],
      ];
    }
    return json_encode($data);
  }

  /**
   * Retrurn opens for an Split Test
  */
  public function getOpens(Request $request)
  {
    $result = SplitTestStat::statLogOpens($request->stat_id);

    $columns = ['email', 'list', 'sending_server', 'message_id', 'ip', 'country', 'zipcode', 'city', 'split_test_stat_log_opens.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $message_id = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->message_id.'\'}); } else { swal({title: \'\', text:\''. $stat->message_id.'\'}); }">'.substr($stat->message_id, 0, 20).'...</a>';
      $country = empty($stat->country) ? '---' : $stat->country;
      $city = empty($stat->city) ? '---' : $stat->city;
      $zipcode = empty($stat->zipcode) ? '---' : $stat->zipcode;
      $data['data'][] = [
        $stat->email,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $stat->ip,
        $country,
        $city,
        $zipcode,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
  }

  /**
   * Retrurn clicks for an Split Test
  */
  public function getClicks(Request $request)
  {
    $result = SplitTestStat::statLogClicks($request->stat_id);

    $columns = ['email', 'link', 'list', 'sending_server', 'message_id', 'ip', 'country', 'zipcode', 'city', 'split_test_stat_log_clicks.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $country = empty($stat->country) ? '---' : $stat->country;
      $city = empty($stat->city) ? '---' : $stat->city;
      $zipcode = empty($stat->zipcode) ? '---' : $stat->zipcode;
      $link = '<a href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire(\''.$stat->link.'\'); } else { swal(\''.$stat->link.'\'); }">'.substr($stat->link, 0, 20).'...</a>';
      $message_id = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->message_id.'\'}); } else { swal({title: \'\', text:\''. $stat->message_id.'\'}); }">'.substr($stat->message_id, 0, 20).'...</a>';
      $data['data'][] = [
        $stat->email,
        $link,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $stat->ip,
        $country,
        $city,
        $zipcode,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
  }

  /**
    * Retrurn spam for a scheduled campaign
    */
    public function getReplies(Request $request)
    {
    $result = SplitTestStat::statLogReplies($request->stat_id);

    $columns = ['replies.email', 'list', 'sending_server', 'message_id', 'detail', 'replies.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $message_id = '<a class="link-info" href="javascript:;" onclick="if(typeof Swal !== \'undefined\') { Swal.fire({title: \'\', text:\''. $stat->message_id.'\'}); } else { swal({title: \'\', text:\''. $stat->message_id.'\'}); }">'.substr($stat->message_id, 0, 20).'...</a>';
      $reply_html = '<a class="link-info" target="_blank" href="'.route('get.reply_html', ['id' => $stat->schedule_campaign_stat_log_id]).'">'.__('app.view').'</a>';

      $data['data'][] = [
        $stat->email,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $reply_html,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
    }

  /**
   * Retrurn unsubscribed for an Split Test
  */
  public function getUnsubscribed(Request $request)
  {
    $result = SplitTestStat::statLogClicks($request->stat_id)
      ->where('link', 'like', '%/unsub/%');

    $columns = ['email', 'link', 'list', 'sending_server', 'message_id', 'ip', 'country', 'zipcode', 'city', 'split_test_stat_log_clicks.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
      $country = empty($stat->country) ? '---' : $stat->country;
      $city = empty($stat->city) ? '---' : $stat->city;
      $zipcode = empty($stat->zipcode) ? '---' : $stat->zipcode;
      $link = '<a href="javascript:;" onclick="swal(\''.$stat->link.'\')">'.substr($stat->link, 0, 20).'...</a>';
      $message_id = '<a class="link-info" href="javascript:;" onclick="swal({title: \'\', text:\''. $stat->message_id.'\'})">'.substr($stat->message_id, 0, 20).'...</a>';
      $data['data'][] = [
        $stat->email,
        $link,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $stat->ip,
        $country,
        $city,
        $zipcode,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
  }

  /**
   * Retrurn bounces for an Split Test
  */
  public function getBounces(Request $request)
  {
    $result = SplitTestStat::statLogBounces($request->stat_id);

    $columns = ['global_bounces.email', 'list', 'sending_server', 'message_id', 'type', 'code', 'detail', 'global_bounces.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
        $message_id = '<a class="link-info" href="javascript:;" onclick="swal({title: \'\', text:\''. $stat->message_id.'\'})">'.substr($stat->message_id, 0, 20).'...</a>';
      $data['data'][] = [
        $stat->email,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $stat->type,
        $stat->code,
        json_decode($stat->detail)->short_detail,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
  }

  /**
   * Retrurn spam for an Split Test
  */
  public function getSpam(Request $request)
  {
    $result = SplitTestStat::statLogSpam($request->stat_id);

    $columns = ['global_spams.email', 'list', 'sending_server', 'message_id', 'detail', 'global_spams.created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
        $message_id = '<a class="link-info" href="javascript:;" onclick="swal({title: \'\', text:\''. $stat->message_id.'\'})">'.substr($stat->message_id, 0, 20).'...</a>';
      $data['data'][] = [
        $stat->email,
        $stat->list,
        $stat->sending_server,
        $message_id,
        json_decode($stat->detail)->full_detail,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
  }

  /**
   * Retrurn all info for an Split Test
  */
  public function getLogs(Request $request)
  {
    $result = \App\Models\SplitTestStatLog::whereSplitTestStatId($request->stat_id);

    $columns = ['email', 'list', 'sending_server', 'message_id', 'status', 'created_at'];
    $data = Helper::dataFilters($request, $result, $columns);

    $result = $data['result'];
    $stats = $result->get();

    $data =  Helper::datatableTotals($data['total']);

    foreach($stats as $stat) {
        $message_id = '<a class="link-info" href="javascript:;" onclick="swal({title: \'\', text:\''. $stat->message_id.'\'})">'.substr($stat->message_id, 0, 20).'...</a>';
      $data['data'][] = [
        $stat->email,
        $stat->list,
        $stat->sending_server,
        $message_id,
        $stat->status,
        Helper::datetimeDisplay($stat->created_at),
      ];
    }
    echo json_encode($data);
  }

  /**
   * Export Split Test stat detail
  */
  public function export($id)
  {
    Helper::checkPermissions('analatics_split_tests_export');
    $schedule_stat = SplitTestStat::whereId($id)->app()->first();

    $path_export_stat = str_replace('[user-id]', Auth::user()->app_id, config('mc.path_export_stat_split_test'));
    Helper::dirCreateOrExist($path_export_stat); // create dir if not exist

    // Summary PDF File
    $file_summary = $path_export_stat.$id.'-summary.pdf';

    // Get Summary
    $html = $this->getSummary($schedule_stat);
    \PDF::loadHTML($html)->setPaper('a4')->setWarnings(false)->save($file_summary);

    // Opens CSV File
    $file_opens = $path_export_stat.$id.'-opens.csv';
    $this->exportOpens($id, $file_opens);

    // Clicks CSV File
    $file_clicks = $path_export_stat.$id.'-clicks.csv';
    $this->exportClicks($id, $file_clicks);

    // Unsubscribed CSV File
    $file_clicks = $path_export_stat.$id.'-unsubscribed.csv';
    $this->exportUnsubscribed($id, $file_clicks);

    // Bounces CSV File
    $file_bounces = $path_export_stat.$id.'-bounces.csv';
    $this->exportBounces($id, $file_bounces);

    // Spam CSV File
    $file_spam = $path_export_stat.$id.'-spam.csv';
    $this->exportSpam($id, $file_spam);

    // Replies CSV File
    $file_replies = $path_export_stat.$id.'-replies.csv';
    $this->exportReplies($id, $file_replies);

    // Logs CSV File
    $file_logs = $path_export_stat.$id.'-logs.csv';
    $this->exportLogs($id, $file_logs);


    // Make Zip
    $zip_file = $path_export_stat.$id.'.zip';
    Helper::createZipFile($zip_file, $file_summary, 'summary.pdf');
    Helper::createZipFile($zip_file, $file_opens, 'opens.csv');
    Helper::createZipFile($zip_file, $file_clicks, 'clicks.csv');
    Helper::createZipFile($zip_file, $file_unsubs, 'unsubscribed.csv');
    Helper::createZipFile($zip_file, $file_bounces, 'bounces.csv');
    Helper::createZipFile($zip_file, $file_spams, 'spams.csv');
    Helper::createZipFile($zip_file, $file_logs, 'logs.csv');
    Helper::createZipFile($zip_file, $file_replies, 'replies.csv');

    // Delete other files after created zip
    unlink($file_summary);
    unlink($file_opens);
    unlink($file_clicks);
    unlink($file_unsubs);
    unlink($file_bounces);
    unlink($file_spams);
    unlink($file_logs);
    unlink($file_replies);

  }

  /**
   * Download export Split Test stat detail
  */
  public function exportDownload($id)
  {
    try{
      $path_export_stat = str_replace('[user-id]', Auth::user()->app_id, config('mc.path_export_stat_split_test'));
      $zip_file = $path_export_stat.$id.'.zip';
      return response()->download($zip_file)->deleteFileAfterSend(true);
    } catch(\Exception $e) {
      // if file not found
      return redirect(route('detail.stat.split_test', ['id' => $id]));
    }
  }

  // Export Split Test opens
  public function exportOpens($id, $file)
  {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.ip'),
      __('app.country'),
      __('app.city'),
      __('app.zip_code'),
      __('app.datetime'),
    ];

    // Write file header

    $writer->insertOne($file_header);
    // Get Opens
    SplitTestStat::statLogOpens($id)
      ->chunk(1000, function ($opens)  use ($writer) {
        foreach($opens as $open) {
          $opens_data = [
            $open->email,
            $open->list,
            $open->sending_server,
            $open->message_id,
            $open->ip,
            $open->country,
            $open->city,
            $open->zipcode,
            Helper::datetimeDisplay($open->created_at),
          ];

          // Write open data
          $writer->insertOne($opens_data);
        }
    });
  }

  // Export Split Test clicks
  public function exportClicks($id, $file)
  {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.link'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.ip'),
      __('app.country'),
      __('app.city'),
      __('app.zip_code'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Clicks
    SplitTestStat::statLogClicks($id)
      ->chunk(1000, function ($clicks)  use ($writer) {
        foreach($clicks as $click) {
          $clicks_data = [
            $click->email,
            $click->link,
            $click->list,
            $click->sending_server,
            $click->message_id,
            $click->ip,
            $click->country,
            $click->city,
            $click->zipcode,
            Helper::datetimeDisplay($click->created_at),
          ];

          // Write click data
          $writer->insertOne($clicks_data);
        }
    });
  }

  // Export campaign replies
    public function exportReplies($id, $file)
    {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.detail'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Replies
    SplitTestStat::statLogReplies($id)
      ->chunk(1000, function ($replies)  use ($writer) {
        foreach($replies as $reply) {
          $reply_data = [
            $reply->email,
            $reply->list,
            $reply->sending_server,
            $reply->message_id,
            json_decode($reply->detail)->full_detail,
            Helper::datetimeDisplay($reply->created_at),
          ];

          try {
            // Write bounce data
            $writer->insertOne($reply_data);
          } catch(\Exception $e) {}
        }
    });
    }

  // Export SplitTestStat unsubscribed
  public function exportUnsubscribed($id, $file)
  {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.link'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.ip'),
      __('app.country'),
      __('app.city'),
      __('app.zip_code'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Clicks
    SplitTestStat::statLogClicks($id)
      ->where('link', 'like', '%/unsub/%')
      ->chunk(1000, function ($clicks)  use ($writer) {
        foreach($clicks as $click) {
          $clicks_data = [
            $click->email,
            $click->link,
            $click->list,
            $click->sending_server,
            $click->message_id,
            $click->ip,
            $click->country,
            $click->city,
            $click->zipcode,
            Helper::datetimeDisplay($click->created_at),
          ];

          // Write click data
          $writer->insertOne($clicks_data);
        }
    });
  }

  // Export SplitTestStat bounces
  public function exportBounces($id, $file)
  {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.type'),
      __('app.code'),
      __('app.detail'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Bounces
    SplitTestStat::statLogBounces($id)
      ->chunk(1000, function ($bounces)  use ($writer) {
        foreach($bounces as $bounce) {
          $bounces_data = [
            $bounce->email,
            $bounce->list,
            $bounce->sending_server,
            $bounce->message_id,
            $bounce->type,
            $bounce->code,
            json_decode($bounce->detail)->short_detail,
            Helper::datetimeDisplay($bounce->created_at),
          ];

          // Write bounce data
          $writer->insertOne($bounces_data);
        }
    });
  }

  // Export SplitTestStat spam
  public function exportSpam($id, $file)
  {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.detail'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Spam
    SplitTestStat::statLogSpam($id)
      ->chunk(1000, function ($spams)  use ($writer) {
        foreach($spams as $spam) {
          $spam_data = [
            $spam->email,
            $spam->list,
            $spam->sending_server,
            $spam->message_id,
            json_decode($spam->detail)->full_detail,
            Helper::datetimeDisplay($spam->created_at),
          ];

          // Write bounce data
          $writer->insertOne($spam_data);
        }
    });
  }

  // Export SplitTestStatLog logs
  public function exportLogs($id, $file)
  {
    // Create a .csv file to write data
    $writer = \League\Csv\Writer::createFromPath($file, 'w+');

    $file_header = [
      __('app.email'),
      __('app.list'),
      __('app.sending_server'),
      __('app.message_id'),
      __('app.latest_activity'),
      __('app.datetime'),
    ];

    // Write file header
    $writer->insertOne($file_header);

    // Get Logs
    \App\Models\SplitTestStatLog::whereSplitTestStatId($id)
      ->chunk(1000, function ($logs)  use ($writer) {
        foreach($logs as $log) {
          $logs_data = [
            $log->email,
            $log->list,
            $log->sending_server,
            $log->message_id,
            $log->status,
            Helper::datetimeDisplay($log->created_at),
          ];

          // Write bounce data
          $writer->insertOne($logs_data);
        }
    });
  }

  /**
   * Delete stat
  */
  public function destroy($id, $name)
  {
    $destroy = SplitTestStat::destroy($id);

    activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.stats') ." ". __('app.split_test') . " ({$name}) ". __('app.log_delete'));

    $data = [
          'success' => true,
          'message' => __('app.deleted_successfully')
        ];

    return response()->json($data, 200);
    
  }
}
