<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use App\Models\WarmupPlan;
use App\Http\Helper\Helper;
use Auth;

class WarmupPlanController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        Helper::checkPermissions('warmup_plans');
        $page = "warmup_servers";
        $page_title = __('app.warmup_plans');
        $breadcrumbs = [
            __('app.warmup_plans') => route('warmup_plans.index'),
            __('app.manage') => '#'
        ];  
        return view('warmup_plans.index',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
       * Retrun JSON datatable data
    */
    public function getWarmupPlans(Request $request): Void
    {
        $result = WarmupPlan::select('id', 'name', 'is_active', 'growth_factor', 'total_emails', 'created_at')
            ->app();

        $columns = ['id', 'id', 'name', 'is_active', 'growth_factor', 'total_emails', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $warmup_plans = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($warmup_plans as $warmup_plan) {
          $checkbox = "<input class=\"form-check-input\" type=\"checkbox\" value=\"{$warmup_plan->id}\">";

          $actions = '<div class="btn-group">
            <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').' <span class="caret"></span></button><ul class="dropdown-menu" role="menu">';            
          $actions .= '<li><a class="dropdown-item" href="'.route('warmup_plans.show', $warmup_plan->id).'"><i class="bi bi-eye"></i> '.__('app.view').'</a></li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$warmup_plan->id.'\', \''.route('warmup_plans.destroy', $warmup_plan->id).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul>';

          $name = '<a class="link-info" href="'.route('warmup_plans.show', $warmup_plan->id).'">'.$warmup_plan->name.'</a>';
          $status = $warmup_plan->is_active 
                ? "<span class='badge bg-success-gradient'>".__('app.yes')."</small>"
                : "<span class='badge bg-danger-gradient'>".__('app.no')."</small>";

          $data['data'][] = [
            "DT_RowId" => "row_{$warmup_plan->id}",
            $checkbox,
            $warmup_plan->id,
            $name,
            $status,
            $warmup_plan->growth_factor . ' %',
            $warmup_plan->total_emails,
            Helper::datetimeDisplay($warmup_plan->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        Helper::checkPermissions('warmup_plans');
        $growthFactors = $this->growthFactors();
        return view('warmup_plans.create')->with(compact('growthFactors'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $data = $this->_warmupPlanlData($request);
        $warmup_plan = WarmupPlan::create($data);

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.warmup_plan') . " ({$request->name}) ". __('app.log_create'));

        return redirect()->route('warmup_plans.show', $warmup_plan->id)
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function update(Request $request, WarmupPlan $warmup_plan): RedirectResponse
    {
        $warmup_plan->update($request->except(['_token', '_method', 'total_emails', 'growth_factor']));

        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.warmup_plan') . " ({$request->name}) ". __('app.log_update'));

        return redirect()->route('warmup_plans.show', $warmup_plan->id)
            ->with('success', ($request->name . ' ' . __('app.updated_successfully')));
    }

    public function show(WarmupPlan $warmup_plan): View
    {   Helper::checkPermissions('warmup_plans');
        $page = "warmup_servers";
        $page_title = __('app.warmup_plan');
        $breadcrumbs = [
            __('app.warmup_plans') => route('warmup_plans.index'),
            __('app.manage') => '#'
        ]; 
        return view('warmup_plans.show',compact('page', 'page_title', 'breadcrumbs', 'warmup_plan'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
          $ids = array_values($request->ids);
          $names = json_encode(array_values(WarmupPlan::whereIn('id', $ids)->pluck('name')->toArray()));
          $destroy = WarmupPlan::whereIn('id', $ids)->delete();
        } else {
          $names = WarmupPlan::whereId($id)->value('name');
          $destroy = WarmupPlan::destroy($id);
        }
        
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.warmup_plan') . " ({$names}) ". __('app.log_delete'));

        $data = [
            'success' => true,
            'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200); 
    }

    /**
    * Retrun data for store or update
    */
    private function _warmupPlanlData($request): Array
    {
        $input = $request->except('_token');
        $input['schedule'] = $this->generateWarmupSchedule($input['total_emails'], $input['growth_factor'], $days=180);
        $input['days'] = $days;
        $input['user_id'] = Auth::user()->id;
        $input['app_id'] = Auth::user()->app_id;
        return $input;
    }

    public function generateWarmupSchedule($totalEmails, $growthFactor, $warmupDays=180) 
    {
        $warmupSchedule = [];
        $remainingEmails = $totalEmails;
        
        // Calculate initial email count (start small)
        $dailyLimit = max(5, round($totalEmails / ($warmupDays * 20))); 

        for ($i = 0; $i < $warmupDays; $i++) {
            if ($remainingEmails <= 0) break; // Stop if total is reached

            if ($i == $warmupDays - 1 || $dailyLimit >= $remainingEmails) {
                // Ensure the last day sends exactly the remaining emails
                $warmupSchedule[] = $remainingEmails;
                break;
            } else {
                $emailsToday = min(round($dailyLimit), $remainingEmails);
                $warmupSchedule[] = $emailsToday;
                $remainingEmails -= $emailsToday;
                $dailyLimit *= $growthFactor; // Increase daily volume
            }
        }

        return json_encode($warmupSchedule);
    }

    /**
    * Return custom fields types
    */
    public function growthFactors(): Array
    {
        return [
          '1.1' => '1.1X',
          '1.2' => '1.2X',
          '1.3' => '1.3X',
          '1.4' => '1.4X',
          '1.5' => '1.5X',
          '1.6' => '1.6X',
          '1.7' => '1.7X',
          '1.8' => '1.8X',
          '1.9' => '1.9X',
          '2.0' => '2.0X',
        ];
    }
}
