<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use App\Notifications\ContactAdd;
use App\Models\Webform;
use App\Http\Helper\Helper;
use Auth;

class WebformController extends Controller
{
    //

    /**
   * Retrun index view
  */
    public function index(): view
    {
        Helper::checkPermissions('webforms'); // check user permission
        $page = 'contents_webforms';
        $page_title = __('app.webforms');
        $breadcrumbs = [
            __('app.contents_webforms') => route('webforms.index'),
            __('app.manage') => '#'
        ];
        return view('webforms.index',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Retrun JSON datatable data
    */
    public function getWebforms(Request $request): void
    {
        $result = Webform::select('id', 'name', 'created_at')
          ->app();

        $columns = ['id', 'id', 'name', 'created_at', 'id'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $webforms = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($webforms as $webform) {
          $checkbox = "<input class=\"form-check-input\" type=\"checkbox\" value=\"{$webform->id}\">";
          $actions = '<div class="btn-group">';
          $actions .= '<button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').'</button>';
          $actions .= '<ul class="dropdown-menu" style="">';

          $actions.= '<li><a class="dropdown-item" href="javascript:;" onclick="viewModal(\'modal\', \''.route('webforms.show', [$webform->id]).'\')"><i class="bi bi-eye"></i> '.__('app.view').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="'.route('webforms.edit', [$webform->id]).'"><i class="bi bi-pencil-square"></i> '.__('app.edit').'</a></li>';
          $actions.= '<li><a class="dropdown-item" href="javascript:;" onclick="viewModal(\'modal\', \''.route('webforms.show', [$webform->id, 'get_html' => 1]).'\')"><i class="bi bi-code"></i> '.__('app.webform_get_html').'</a></li>';
          $actions .= '<li><a class="dropdown-item" href="'.route('webform.copy', [$webform->id]).'"><i class="bi bi-copy"></i> '.__('app.copy').'</a></li>';
          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$webform->id.'\', \''.route('webforms.destroy', [$webform->id]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul></div>';
          $data['data'][] = [
            "DT_RowId" => "row_{$webform->id}",
            $checkbox,
            $webform->id,
            $webform->name,
            Helper::datetimeDisplay($webform->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
    * Retrun create view
    */
    public function create(): View
    {
        Helper::checkPermissions('webforms'); // check user permission
        $page = 'contents_webforms';
        $page_title = __('app.contents_webforms');
        $breadcrumbs = [
            __('app.contents_webforms') => route('webforms.index'),
            __('app.add_new_webform') => '#'
        ];
        return view('webforms.create')->with(compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
    * Save data
    */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'list_ids' => 'required|array|min:1', // Ensure list_ids is not empty and contains at least 
            'thankyou_page_custom_url' => 'url'
        ]);
        $data = $this->_webformData($request);
        $webform = Webform::create($data);
        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.webform') . " ({$request->name}) ". __('app.log_create'));

        return redirect()->route('webforms.index', $webform->id)
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
    }


    /**
     * Retrun edit view
    */
    public function edit($id): view
    {
      Helper::checkPermissions('webforms'); // check user permission
      $page = 'contents_webforms';
      $page_title = __('app.contents_webforms');
      $breadcrumbs = [
            __('app.contents_webforms') => route('webforms.index'),
            __('app.edit_webform') => '#'
      ];
      $webform = Webform::whereId($id)->app()->first();
      return view('webforms.edit')->with(compact('page', 'page_title', 'breadcrumbs', 'webform'));
    }

    /**
     * Update data
    */
    public function update(Request $request, $id): RedirectResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'list_ids' => 'required|array|min:1', // Ensure list_ids is not empty and contains at least 
            'thankyou_page_custom_url' => 'url'
        ]);
        $webform = Webform::findOrFail($id);
        $data = $this->_webformData($request);
        $webform->fill($data)->save();
        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.webform') . " ({$request->name}) ". __('app.log_update'));
        return redirect()->route('webforms.edit', $webform->id)
            ->with('success', ($request->name . ' ' . __('app.created_successfully')));
    }


    /**
     * Retrun data for store or update
    */
    public function _webformData($request)
    {
        $input = $request->except('_token');
        if(!empty($input['custom_fields'])) {
          $input['custom_field_ids'] = implode(',', $input['custom_fields']);
        } else {
            $input['custom_field_ids'] = null;
        }

        if(isset($input['list_ids'][0]))
            $input['list_id'] = $input['list_ids'][0]; // for single select the list_id will be at 0 location

        $input['attributes'] = json_encode($input);
        $input['user_id'] = Auth::user()->id;
        $input['app_id'] = Auth::user()->app_id;
        return $input;
    }

    /**
     * Return webform view
    */
    public function show($id, $get_html=0)
    {
        Helper::checkPermissions('webforms'); // check user permission
        $webform = Webform::whereId($id)->app()->first();
        $custom_fields = explode(',', $webform->custom_field_ids);
        $list = \App\Models\Lists::findOrFail($webform->list_id);
        $list_custom_fields = $list->customFields()->whereIn('custom_fields.id', $custom_fields)->get();
        return view('webforms.show')->with(compact('webform', 'list_custom_fields', 'get_html'));
    }

    /**
    * Delete one or more
    */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if(!empty($request->action)) {
            $ids = array_values($request->ids);
            $names = json_encode(array_values(Webform::whereIn('id', $ids)->pluck('name')->toArray()));
            $destroy = Webform::whereIn('id', $ids)->delete();
        } else {
            $names = Webform::whereId($id)->value('name');
            $destroy = Webform::destroy($id);
        }
        activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.webform') . " ({$names}) ". __('app.log_delete'));

        $data = [
          'success' => true,
          'message' => __('app.deleted_successfully')
        ];

        return response()->json($data, 200);
    }

    /**
     * Retrun index after copy webform
    */
    public function copy($id)
    {
      Helper::checkPermissions('webforms'); // check user permission
      $webform = Webform::whereId($id)->app()->first();
      $webform->name = $webform->name .' - copy ';
      $webform = $webform->replicate();
      $webform->save();
      activity('copy')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.webform') . " ({$webform->name}) ". __('app.log_copy')); // log
      return redirect()->route('webforms.index', $webform->id)
            ->with('success', ($webform->name . ' ' . __('app.created_successfully')));
    }



    /**
   * Retrun respecitve page after store webform data
  */
  public function saveWebFormData(Request $request) {

    if ($request->isMethod('post')) {
      $webform = Webform::whereId($request->form_id)->with('list')->first();
      $attributes = json_decode($webform->attributes);
      $custom_url = null;
      if(0 && $webform->list->double_optin == 'Yes') {
        $input['confirmed'] = 'No';
        $subscriber_page = 'confirm';
        $subscriber_email = 'confirm-email-webform';

        if($attributes->confirmation_page_option == 'custom') {
          $custom_url =  $attributes->confirmation_page_custom_url;
        }
      } else {
        $input['confirmed'] = 'Yes';
        $subscriber_page = 'thankyou';
        $subscriber_email = 'welcome-email';
       // if($attributes->thankyou_page_option == 'custom') {
          $custom_url =  $attributes->thankyou_page_custom_url;
        //}
      }
      $input = $request->all();
      $input['user_id'] = $webform->user_id;
      $input['app_id'] = $webform->app_id;
      $input['list_id'] = $webform->list_id;
      $input['is_active'] = true;
      $input['is_unsubscribed'] = false;
      $input['source'] = 'form';

      $contact = \App\Models\Contact::whereEmail($input['email'])->whereListId($input['list_id'])->select('id')->with('customFields')->first();
      if(!empty($contact->id)) {
        if($webform->duplicates == 'Overwrite') {
          \DB::table('custom_field_contact')->whereContactId($contact->id)->delete();
          app('App\Http\Controllers\ContactController')->contactCustomData($contact->id, $input);
        }
      } else {
        $contact = \App\Models\Contact::create($input);
        app('App\Http\Controllers\ContactController')->contactCustomData($contact->id, $input);
      }

      // send notfication
      if($webform->list->welcome_email == 'Yes' || !empty($webform->list->notify_email))
            $contact->notify(new ContactAdd('welcome-email', 'notify-email-contact-added'));

      if(!empty($custom_url)) {
        return redirect($custom_url);
      } else {
        // display page as per list settings opt-in
        return redirect()->route('page.show', ['slug' => $subscriber_page, 'id' => base64_encode($contact->id)]);
      }
    } else {
      return response()->json(['success' => 'success'], 200);
    }
  }
}
