<?php

namespace App\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;
use App\Http\Helper\Helper;

class Backup implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;
    protected $process, $user_id, $app_id, $backup_db, $backup_files;
    public $tries = 3;

    /**
     * Create a new job instance.
     */
    public function __construct($user_id, $app_id, $backup_db, $backup_files)
    {
        $this->user_id = $user_id;
        $this->app_id = $app_id;
        $this->backup_db = $backup_db;
        $this->backup_files = $backup_files;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        if(!$this->backup_files && !$this->backup_db) {
            \Log::warning('Backup job: No backup options selected');
            return;
        }

        try {
            $path_backup = str_replace('[user-id]', $this->user_id, config('custom.path_backup'));
            Helper::dirCreateOrExist($path_backup); // create dir if not exist
            $backup_zip = $path_backup.'backup.zip';

            // Remove existing backup file if it exists
            if(is_file($backup_zip)) {
                unlink($backup_zip);
            }

            // if backup is set for files
            if($this->backup_files) {
                $excludes = ['.git', 'vendor', 'geo', 'node_modules', '.env', 'storage/logs']; // Add any folders you don't want to include
                Helper::createZipFolder($backup_zip, base_path(), $excludes);
            }

            // if backup is set for db
            if($this->backup_db) {
                $file_sql = 'SITE_DB.sql';

                // Optimize mysqldump command with better options
                $process = new Process([
                    'mysqldump',
                    '--single-transaction',
                    '--routines',
                    '--triggers',
                    '--lock-tables=false',
                    '-u' . config('database.connections.mysql.username'),
                    '-p' . config('database.connections.mysql.password'),
                    config('database.connections.mysql.database')
                ]);

                $process->setTimeout(300); // 5 minutes timeout
                $process->run();

                // Check for errors
                if (!$process->isSuccessful()) {
                    throw new ProcessFailedException($process);
                }

                // Save the output to the file
                file_put_contents($file_sql, $process->getOutput());

                Helper::createZipFile($backup_zip, $file_sql);
                unlink($file_sql);
            }

            // save notification for user to inform and download
            $notification_name = __('app.msg_backup_export_successfully');
            $attributes = [
                'file' => $backup_zip
            ];
            $notification_attributes = json_encode($attributes);
            $notification = [
                'name' => $notification_name,
                'type' => 'export',
                'attributes' => $notification_attributes,
                'app_id' => $this->app_id,
                'user_id' => $this->user_id
            ];
            \App\Models\Notification::create($notification);
            
        } catch(\Exception $e) {
            \Log::error('Backup job failed: ' . $e->getMessage());
            throw $e; // Re-throw to mark job as failed
        }
    }
}
