<?php

namespace App\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Models\GlobalBounce;
use League\Csv\Writer;
use App\Http\Helper\Helper;

class BouncedExport implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $app_id, $user_id;
    public $tries = 1;

    /**
     * Create a new job instance.
     */
    public function __construct($app_id, $user_id)
    {
        $this->app_id = $app_id;
        $this->user_id = $user_id;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            $path_export_blacklist = str_replace('[user-id]', $this->user_id, config('custom.path_export_blacklist'));
            Helper::dirCreateOrExist($path_export_blacklist); // create dir if not exist

            $file = $path_export_blacklist. 'bounces-'.Helper::uniqueID().'.csv';
            $writer = Writer::createFromPath($file, 'w+'); // create a .csv file to write data

            $file_header = [
                __('app.email'),
                __('app.type'),
                __('app.code'),
                __('app.detail'),
                __('app.datetime'),
            ];

            $writer->insertOne($file_header); // write file header

            // Cache timezone to avoid repeated DB queries
            $timezone = \App\Models\User::getUserValue($this->user_id, 'timezone');
            
            // Check if there are any bounces to export
            // Match the same query logic as getSystemBounced in BlacklistController
            $bounce_count = \App\Models\GlobalBounce::whereNotNull('email')
                ->where(function($query) {
                    $query->where('app_id', $this->app_id)
                          ->orWhereNull('app_id');
                })
                ->count();
            
            if($bounce_count === 0) {
                \Log::info('BouncedExport: No bounces to export');
                // Still create notification for empty export
                $this->createNotification($file);
                return;
            }

            // Export all bounces matching the same criteria as the controller
            // Use chunking to handle large datasets efficiently
            $total_exported = 0;
            $chunk_count = 0;
            
            try {
                \App\Models\GlobalBounce::whereNotNull('email')
                    ->where(function($query) {
                        $query->where('app_id', $this->app_id)
                              ->orWhereNull('app_id');
                    })
                    ->orderBy('id') // Ensure consistent ordering
                    ->chunk(1000, function ($bounces) use ($writer, $timezone, &$total_exported, &$chunk_count) {
                        $chunk_count++;
                        $chunk_size = $bounces->count();
                        \Log::info("BouncedExport: Processing chunk {$chunk_count} with {$chunk_size} bounces");
                        
                        foreach($bounces as $bounce) {
                            try {
                                $bounce_data = [
                                    $bounce->email ?? '',
                                    $bounce->type ?? '',
                                    $bounce->code ?? '',
                                    json_decode($bounce->detail)->short_detail ?? '',
                                    Helper::datetimeDisplay($bounce->created_at, $timezone),
                                ];

                                $writer->insertOne($bounce_data); // write contact info
                                $total_exported++;
                            } catch(\Exception $e) {
                                \Log::error('BouncedExport: Error processing bounce ID ' . ($bounce->id ?? 'unknown') . ': ' . $e->getMessage());
                                // Continue processing other bounces
                            }
                        }
                    });
            } catch(\Exception $e) {
                \Log::error('BouncedExport: Error during chunking process: ' . $e->getMessage());
                \Log::error('BouncedExport: Stack trace: ' . $e->getTraceAsString());
                throw $e; // Re-throw to ensure job is marked as failed
            }
            
            \Log::info("BouncedExport: Completed - Exported {$total_exported} bounces out of {$bounce_count} total for app_id {$this->app_id} (processed {$chunk_count} chunks)");
            
            // Verify export count matches expected count
            if($total_exported < $bounce_count) {
                \Log::warning("BouncedExport: Export count mismatch! Expected {$bounce_count} but exported {$total_exported}. This may indicate an issue with the export process.");
            }

            // save notification for user to inform and download
            $this->createNotification($file);
            
        } catch(\Exception $e) {
            \Log::error('BouncedExport job failed: ' . $e->getMessage());
            throw $e; // Re-throw to mark job as failed
        }
    }

    /**
     * Create notification for the export
     */
    private function createNotification($file)
    {
        $notification_name = __('app.system_bounced') .' ' . __('app.msg_export_successfully');
        $attributes = [
            'file' => $file
        ];
        $notification_attributes = json_encode($attributes);
        $notification = [
            'name' => $notification_name,
            'type' => 'export',
            'attributes' => $notification_attributes,
            'app_id' => $this->app_id,
            'user_id' => $this->user_id
        ];
        \App\Models\Notification::create($notification);
    }
}
