<?php

namespace App\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Http\Helper\Helper;
use League\Csv\Writer;

class ListExport implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;
    public $tries = 1;
    protected $list_id, $user_id, $fields;

    /**
     * Create a new job instance.
     */
    public function __construct($list_id, $user_id, $fields)
    {
        $this->list_id = $list_id;
        $this->user_id = $user_id;
        $this->fields = $fields;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            $list = \App\Models\Lists::findOrFail($this->list_id);
            $path_export_list = str_replace('[user-id]', $this->user_id, config('custom.path_export_list'));
            Helper::dirCreateOrExist($path_export_list); // create dir if not exist

            $file = $path_export_list.\Illuminate\Support\Str::slug($list->name).'-'.Helper::uniqueID().'.csv';
            $writer = Writer::createFromPath($file, 'w+'); // create a .csv file to write data

            // create file header - cache custom fields
            $list_custom_fields = $list->customFields()->orderBy('custom_fields.id')->pluck('name');

            $file_header = [
                __('app.contact_email'),
                __('app.active'),
                __('app.unsubscribed'),
                __('app.confirmed'),
                __('app.verified'),          
            ];
            foreach($list_custom_fields as $custom_field) {
                array_push($file_header, $custom_field);
            }

            $writer->insertOne($file_header); // write file header

            // Build query with filters
            $list_contacts = \App\Models\Contact::whereListId($this->list_id)
                ->with('customFields');

            // Apply filters more efficiently
            $this->applyFilters($list_contacts);

            // Check if there are any contacts to export
            $contact_count = $list_contacts->count();
            if($contact_count === 0) {
                \Log::info('ListExport: No contacts to export for list ID ' . $this->list_id);
                // Still create notification for empty export
                $this->createNotification($list, $file);
                return;
            }

            $list_contacts->chunk(1000, function ($contacts) use ($writer, $list_custom_fields) {
                foreach($contacts as $contact) {
                    try {
                        $contact_data = [
                            $contact->email,
                            $contact->is_active ? 1 : 0,
                            $contact->is_unsubscribed ? 0 : 1,
                            $contact->is_confirmed ? 1 : 0,
                            $contact->is_verified ? 1 : 0,                
                        ];

                        // Process custom fields more efficiently
                        $custom_fields = $contact->customFields()->orderBy('custom_fields.id')->get();
                        if(!empty($custom_fields)) {
                            foreach($custom_fields as $custom_field) {
                                $custom_field_data = str_replace('||', ', ', $custom_field->pivot->data);
                                array_push($contact_data, $custom_field_data);
                            }
                        }

                        $writer->insertOne($contact_data); // write contact info
                    } catch(\Exception $e) {
                        \Log::error('ListExport: Error processing contact ID ' . $contact->id . ': ' . $e->getMessage());
                        // Continue processing other contacts
                    }
                }
            });

            // save notification for user to inform and download
            $this->createNotification($list, $file);
            
        } catch(\Exception $e) {
            \Log::error('ListExport job failed: ' . $e->getMessage());
            throw $e; // Re-throw to mark job as failed
        }
    }

    /**
     * Apply filters to the query
     */
    private function applyFilters($query)
    {
        if($this->fields['is_active'] == 'yes') {
            $query->where('is_active', true);
        } elseif($this->fields['is_active'] == 'no') {
            $query->where('is_active', false);
        }

        if($this->fields['is_confirmed'] == 'yes') {
            $query->where('is_confirmed', true);
        } elseif($this->fields['is_confirmed'] == 'no') {
            $query->where('is_confirmed', false);
        }

        if($this->fields['is_verified'] == 'yes') {
            $query->where('is_verified', true);
        } elseif($this->fields['is_verified'] == 'no') {
            $query->where('is_verified', false);
        }

        if($this->fields['is_subscribed'] == 'yes') {
            $query->where('is_unsubscribed', false);
        } elseif($this->fields['is_subscribed'] == 'no') {
            $query->where('is_unsubscribed', true);
        }
    }

    /**
     * Create notification for the export
     */
    private function createNotification($list, $file)
    {
        $notification_name = __('app.list'). " ({$list->name}) "  . __('app.msg_export_successfully');
        $attributes = [
            'file' => $file
        ];
        $notification_attributes = json_encode($attributes);
        $notification = [
            'name' => $notification_name,
            'type' => 'export',
            'attributes' => $notification_attributes,
            'app_id' => $list->app_id,
            'user_id' => $this->user_id
        ];
        \App\Models\Notification::create($notification);
    }
}
