<?php

namespace App\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Arr;
use App\Http\Helper\Helper;
use League\Csv\Writer;
use App\Models\Contact;

class ListVerify implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;
    protected $list_ids, $verifier_id, $user_id;
    public $tries = 3;

    public function __construct($list_ids, $verifier_id, $user_id)
    {
        $this->list_ids = $list_ids;
        $this->verifier_id = $verifier_id;
        $this->user_id = $user_id;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            $verifier = \App\Models\EmailVerifier::whereId($this->verifier_id)->first();
            
            if(!$verifier) {
                \Log::warning('ListVerify: Verifier not found: ' . $this->verifier_id);
                return;
            }

            $data_verifier = json_decode($verifier->attributes, true);
            $data_verifier['type'] = $verifier->type;

            $path_export = str_replace('[user-id]', $this->user_id, config('custom.path_export_verify'));
            Helper::dirCreateOrExist($path_export); //Create dir if not exist

            $file = $path_export.Helper::uniqueID().'.csv';
            $writer = Writer::createFromPath($file, 'w+'); // create a .csv file to write data

            $file_header = [
                __('app.email'),
                __('app.verified'),
                __('app.detail'),
            ];

            $writer->insertOne($file_header); // write file header

            $total_contacts = 0;
            foreach($this->list_ids as $list_id) {
                $list_contact_count = Contact::whereListId($list_id)
                    ->where('is_verified', false)
                    ->count();
                $total_contacts += $list_contact_count;
                
                Contact::whereListId($list_id)
                    ->where('is_verified', false)
                    ->select('email')
                    ->chunk(1000, function ($contacts) use ($writer, $data_verifier, $verifier) {

                        foreach($contacts as $contact) {
                            try {
                                $data_verifier['email'] = $contact->email;
                                $response = Helper::verifyEmail($data_verifier, $encrypt=true);
                                if($response['increment']) {
                                    Contact::whereEmail($contact->email)->update(['is_verified' => $response['success']]);
                                    $verifier->increment('total_verified');
                                }

                                $verified = $response['success'] ? 'Yes' : 'No';
                                $export_data = [
                                    $contact->email,
                                    $verified,
                                    $response['message']
                                ];

                                // Write verified info
                                $writer->insertOne($export_data);
                            } catch(\Exception $e) {
                                \Log::error('ListVerify: Error verifying email ' . $contact->email . ': ' . $e->getMessage());
                                // Continue processing other emails
                            }
                        }
                    });
            }

            if($total_contacts === 0) {
                \Log::info('ListVerify: No unverified contacts to process');
            }

            // save notification for user to inform and download
            $this->createNotification($verifier, $file);
            
        } catch(\Exception $e) {
            \Log::error('ListVerify job failed: ' . $e->getMessage());
            throw $e; // Re-throw to mark job as failed
        }
    }

    /**
     * Create notification for the verification
     */
    private function createNotification($verifier, $file)
    {
        $notification_name = __('app.email_verifiers'). " ({$verifier->name}) "  . __('app.report_available');
        $attributes = [
            'file' => $file
        ];
        $notification_attributes = json_encode($attributes);
        $notification = [
            'name' => $notification_name,
            'type' => 'export',
            'attributes' => $notification_attributes,
            'app_id' => $verifier->app_id,
            'user_id' => $this->user_id
        ];
        \App\Models\Notification::create($notification);
    }
}
