<?php

namespace App\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Models\Suppression;
use League\Csv\Writer;
use App\Http\Helper\Helper;

class SuppressionsExport implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $app_id, $user_id;
    public $tries = 1;

    /**
     * Create a new job instance.
     */
    public function __construct($app_id, $user_id)
    {
        $this->app_id = $app_id;
        $this->user_id = $user_id;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            $path_export_blacklist = str_replace('[user-id]', $this->user_id, config('custom.path_export_blacklist'));
            Helper::dirCreateOrExist($path_export_blacklist); // create dir if not exist

            $file = $path_export_blacklist. 'suppressions-'.Helper::uniqueID().'.csv';
            $writer = Writer::createFromPath($file, 'w+'); // create a .csv file to write data

            $file_header = [
                __('app.email'),
                __('app.group'),
                __('app.datetime'),
            ];

            $writer->insertOne($file_header); // write file header

            // Cache timezone to avoid repeated DB queries
            $timezone = \App\Models\User::getUserValue($this->user_id, 'timezone');
            
            // Check if there are any suppressions to export
            $suppression_count = \App\Models\Suppression::where('app_id', $this->app_id)
                ->orWhereNull('app_id')
                ->count();
            if($suppression_count === 0) {
                \Log::info('SuppressionsExport: No suppressions to export');
                // Still create notification for empty export
                $this->createNotification($file);
                return;
            }

            \App\Models\Suppression::join('groups', 'groups.id', '=', 'suppressions.group_id')
                ->select('suppressions.email', 'groups.name as group_name', 'suppressions.created_at')
                ->where(function($query) {
                    $query->where('suppressions.app_id', $this->app_id)
                          ->orWhereNull('suppressions.app_id');
                })
                ->chunk(1000, function ($suppressions) use ($writer, $timezone) {
                    foreach($suppressions as $suppression) {
                        try {
                            $suppression_data = [
                                $suppression->email,
                                $suppression->group_name,
                                Helper::datetimeDisplay($suppression->created_at, $timezone),
                            ];

                            $writer->insertOne($suppression_data); // write suppression info
                        } catch(\Exception $e) {
                            \Log::error('SuppressionsExport: Error processing suppression email ' . ($suppression->email ?? 'unknown') . ': ' . $e->getMessage());
                            // Continue processing other suppressions
                        }
                    }
                });

            // save notification for user to inform and download
            $this->createNotification($file);
            
        } catch(\Exception $e) {
            \Log::error('SuppressionsExport job failed: ' . $e->getMessage());
            throw $e; // Re-throw to mark job as failed
        }
    }

    /**
     * Create notification for the export
     */
    private function createNotification($file)
    {
        $notification_name = __('app.suppressions') .' ' . __('app.msg_export_successfully');
        $attributes = [
            'file' => $file
        ];
        $notification_attributes = json_encode($attributes);
        $notification = [
            'name' => $notification_name,
            'type' => 'export',
            'attributes' => $notification_attributes,
            'app_id' => $this->app_id,
            'user_id' => $this->user_id
        ];
        \App\Models\Notification::create($notification);
    }
}

