<?php

namespace App\Providers;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Log;

class ScheduleServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap services.
     * 
     * @param Schedule $schedule
     * @return void
     */
    public function boot(Schedule $schedule): void
    {
        // Only register schedules in production or when explicitly enabled
        if (!$this->shouldRegisterSchedules()) {
            return;
        }

        try {
            $this->registerMinuteTasks($schedule);
            $this->registerFrequentTasks($schedule);
            $this->registerHourlyTasks($schedule);
            $this->registerDailyTasks($schedule);
        } catch (\Exception $e) {
            Log::error('ScheduleServiceProvider: Failed to register schedules', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Register tasks that run every minute
     * 
     * @param Schedule $schedule
     * @return void
     */
    private function registerMinuteTasks(Schedule $schedule): void
    {
        // Core campaign processing
        $schedule->command('run:campaigns')
            ->everyMinute()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/campaigns.log'));

        // Counter management
        $schedule->command('set:counter')
            ->everyMinute()
            ->runInBackground();

        // Job processing
        $schedule->command('execute:jobs')
            ->everyMinute()
            ->runInBackground();

        // Segmentation processing
        $schedule->command('process:segmentation')
            ->everyMinute()
            ->runInBackground();

        // Trigger system
        $schedule->command('relayzo:triggers')
            ->everyMinute()
            ->runInBackground();

        $schedule->command('run:triggers-campaigns')
            ->everyMinute()
            ->runInBackground();

        // Drip campaigns
        $schedule->command('run:drips-campaigns')
            ->everyMinute()
            ->runInBackground();

        // Split testing
        $schedule->command('prepare:split-test')
            ->everyMinute()
            ->runInBackground();

        $schedule->command('run:split-test-campaigns')
            ->everyMinute()
            ->runInBackground();

        $schedule->command('winner:split-test')
            ->everyMinute()
            ->runInBackground();
    }

    /**
     * Register tasks that run every 15-30 minutes
     * 
     * @param Schedule $schedule
     * @return void
     */
    private function registerFrequentTasks(Schedule $schedule): void
    {
        // Open/click tracking processing
        $schedule->command('set:open_clicks')
            ->everyFifteenMinutes()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/tracking.log'));

        // Reply processing
        $schedule->command('process:replies')
            ->everyThirtyMinutes()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/replies.log'));
    }

    /**
     * Register tasks that run every hour
     * 
     * @param Schedule $schedule
     * @return void
     */
    private function registerHourlyTasks(Schedule $schedule): void
    {
        // Bounce processing
        $schedule->command('process:bounces')
            ->hourly()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/bounces.log'));

        // Spam processing
        $schedule->command('process:spams')
            ->hourly()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/spams.log'));
    }

    /**
     * Register tasks that run daily
     * 
     * @param Schedule $schedule
     * @return void
     */
    private function registerDailyTasks(Schedule $schedule): void
    {
        // Maintenance tasks (9 AM and 9 PM)
        $schedule->command('relayzo:maintenance')
            ->twiceDaily(9, 21)
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/maintenance.log'));

        // Cleanup tasks (runs at 2 AM)
        $schedule->command('relayzo:cleaner')
            ->dailyAt('02:00')
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/cleaner.log'));

        // Auto schedule next campaigns (runs at 1 AM)
        $schedule->command('schedule:auto-next')
            ->dailyAt('01:00')
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/auto-next.log'));
    }

    /**
     * Determine if schedules should be registered
     * 
     * @return bool
     */
    private function shouldRegisterSchedules(): bool
    {
        // Only register in production or when explicitly enabled
        return app()->environment('production') || 
               config('app.enable_schedules', false) ||
               app()->runningInConsole();
    }
}
