/**
 * Global DataTable Styling JavaScript
 * Apply to all DataTable instances across the application
 */

$(document).ready(function() {
    // Apply compact styling to all DataTables after they're initialized
    $('table[id$="data"], table[id^="data"]').on('draw.dt', function() {
        applyDataTableStyling($(this));
    });
    
    // Apply styling to all existing DataTables on page load
    setTimeout(function() {
        $('table[id$="data"], table[id^="data"]').each(function() {
            applyDataTableStyling($(this));
        });
    }, 100);
    
    // Apply styling when new DataTables are created
    $(document).on('init.dt', 'table', function() {
        applyDataTableStyling($(this));
    });
    
    // Enhanced handling for dynamically loaded content
    $(document).on('DOMNodeInserted', function(e) {
        if ($(e.target).is('tr') && $(e.target).closest('table[id$="data"], table[id^="data"]').length) {
            setTimeout(function() {
                applyDataTableStyling($(e.target).closest('table'));
            }, 50);
        }
    });
    
    // Handle AJAX-loaded content
    $(document).ajaxComplete(function() {
        setTimeout(function() {
            $('table[id$="data"], table[id^="data"]').each(function() {
                applyDataTableStyling($(this));
            });
        }, 100);
    });
});

/**
 * Apply DataTable styling to a specific table
 * @param {jQuery} $table - The table element
 */
function applyDataTableStyling($table) {
    if (!$table.length) return;
    
    // Apply line-height to all dynamically created rows
    $table.find('tbody tr').each(function() {
        $(this).css({
            'line-height': '2.0',
            'padding': '0.25rem 0.5rem'
        });
        
        // Apply to all cells in this row
        $(this).find('td').css({
            'line-height': '2.0',
            'padding': '0.25rem 0.5rem'
        });
    });
    
    // Apply to header
    $table.find('thead tr th').css({
        'line-height': '1.2',
        'padding': '0.5rem 0.75rem'
    });
    
    // Apply more height to dt_group elements
    $table.find('.dt_group').css({
        'line-height': '1.4',
        'padding': '0.75rem 1rem'
    });
    
    // Set ID column width to prevent overlap (2nd column)
    $table.find('th:nth-child(2), td:nth-child(2)').css({
        'min-width': '80px',
        'width': '80px',
        'white-space': 'nowrap'
    });
    
    // Apply responsive styling for mobile
    if (window.innerWidth <= 768) {
        $table.find('tbody tr td').css({
            'line-height': '2.0',
            'padding': '0.2rem 0.4rem'
        });
        
        $table.find('thead tr th').css({
            'line-height': '1.1',
            'padding': '0.35rem 0.5rem'
        });
    }
    
    // Force reflow to ensure styling is applied
    $table[0].offsetHeight;
}

/**
 * Reapply styling when window is resized
 */
$(window).on('resize', function() {
    $('table[id$="data"], table[id^="data"]').each(function() {
        applyDataTableStyling($(this));
    });
});

/**
 * Auto-apply styling to any new DataTable
 */
$(document).on('DOMNodeInserted', function(e) {
    if ($(e.target).is('table[id$="data"], table[id^="data"]')) {
        setTimeout(function() {
            applyDataTableStyling($(e.target));
        }, 50);
    }
});
